/*!
 * © 2021, 2022 David Vines
 *
 * sumoOyakata is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * sumoOyakata is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with sumoOyakata. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */

/* globals Names, ROT */
/* exported Rikishi */
class Rikishi {
	constructor(r) {
		this._name = (r && r._name) || Names.getAName();
		this._rank = (r && r._rank) || "";
		if (r) {
			this._stats = Object.assign({},r._stats);
			this._stats._end = this._stats._end || ROT.RNG.getPercentage(); // Allow load of old games
			this._history = r._history || [];
			this._wins = r._wins;
			this._losses = r._losses;
			this._played = [];
		} else {
			this._stats = {
				_int: ROT.RNG.getPercentage(),
				_spd: ROT.RNG.getPercentage(),
				_agl: ROT.RNG.getPercentage(),
				_str: ROT.RNG.getPercentage(),
				_wgt: ROT.RNG.getPercentage(),
				_bal: ROT.RNG.getPercentage(),
				_end: ROT.RNG.getPercentage()
			};
			this._history = [];
			this._wins = 0;
			this._losses = 0;
			this._played = [];
		}
	}
	getRank() {
		return this._rank;
	}
	setRank(rank) {
		this._rank = rank;
		this._cachedSo = undefined;
		this._cachedRs = undefined;
	}
	getIntelligence() {
		return this._stats._int;
	}
	getSpeed() {
		return this._stats._spd;
	}
	getAgility() {
		return this._stats._agl;
	}
	getStrength() {
		return this._stats._str;
	}
	getWeight() {
		return this._stats._wgt;
	}
	getBalance() {
		return this._stats._bal;
	}
	getEndurance() {
		return this._stats._end;
	}
	getHistory() {
		return this._history;
	}
	hasPlayed(rikishi) {
		return this._played.includes(rikishi);
	}
	played(rikishi) {
		this._played.push(rikishi);
	}
	prepBasho(bashoName) {
		this._played = [];
		this._wins = 0;
		this._losses = 0;
		this._cachedSo = undefined;
		this._cachedRs = undefined;
		for(let key of Object.keys(this._stats)) {
			let newvalue = this._stats[key] + Math.round(ROT.RNG.getNormal(0,5));
			if (newvalue < 1) newvalue = 1;
			this._stats[key] = newvalue;
		}
		this._history.push({basho: bashoName, rank: this._rank});
	}
	recordBasho(champion) {
		this._history[this._history.length-1].wins = this._wins;
		if (champion) {
			this._history[this._history.length-1].champion = "true";
		}
	}
	getLastBashoWins() {
		const last = this._history.slice(-3);
		let total = 0;
		for(let record of last) {
			if (record.wins) total += record.wins;
		}
		return total;
	}
	flattened() {
		let data = {};
		data._name = this._name;
		data._rank = this._rank;
		data._stats = Object.assign({},this._stats);
		data._wins = this._wins;
		data._losses = this._losses;
		data._previousBashos = this._previousBashos;
		data._played = [];
		for(let opp of this._played) {
			data._played.push(opp.toString());
		}
		data._history = this._history;
		return data;
	}
	getMove(them,options,time,result) {
		// Build the specific score matrix for this matchup
		let matrix = [];
		for(let i=0; i<options.length; i++) {
			matrix[i] = [];
			for(let j=0; j<options.length; j++) {
				matrix[i][j]  = options[i].getOdds(options[j],this,them)*(100-them.getSaveChance(time));
				matrix[i][j] -= options[j].getOdds(options[i],them,this)*(100-this.getSaveChance(time));
			}
		}
		let oddments = new Array(options.length).fill(0);
		let rowtotal = new Array(options.length).fill(0);
		let coltotal = new Array(options.length).fill(0);
		let ourmove = Math.floor(ROT.RNG.getUniform()*options.length);
		for(let i=0; i<this.getIntelligence()-1; i++) {
			oddments[ourmove]++;
			for(let j=0; j<options.length; j++) {
				coltotal[j] += matrix[ourmove][j];
			}
			let lowest = coltotal[0], lowcol = [0];
			for(let j=1; j<coltotal.length; j++) {
				if (coltotal[j] < lowest) {
					lowest = coltotal[j]; lowcol = [j];
				} else if (coltotal[j] === lowest) {
					lowcol.push(j);
				}
			}
			const theirmove = ROT.RNG.getItem(lowcol);

			for(let j=0; j<options.length; j++) {
				rowtotal[j] += matrix[j][theirmove];
			}
			let highest = rowtotal[0], highrow = [0];
			for(let j=1; j<rowtotal.length; j++) {
				if (rowtotal[j] > highest) {
					highest = rowtotal[j]; highrow = [j];
				} else if (rowtotal[j] === highest) {
					highrow.push(j);
				}
			}
			ourmove = ROT.RNG.getItem(highrow);
		}
		oddments[ourmove]++;

		result.newline(true);
		let text = `${this} odds are`;
		for(let i=0; i<oddments.length; i++) {
			text += ` ${options[i]}: ${oddments[i]}`;
		}
		result.addText(text);
		result.newline(false);

		// And make the choice
		let roll = ROT.RNG.getUniform()*this.getIntelligence(), choice = 0;
		while(roll > oddments[choice]) {
			roll -= oddments[choice++];
		}
		return options[choice];
	}
	getSaveChance(time) {
		return (100 - Math.exp(4.5-0.013*this._stats._bal)) * Math.pow((1-Math.exp(-0.016*this._stats._end)),time);
	}
	saves(time) {
		return (ROT.RNG.getPercentage() <= this.getSaveChance(time));
	}
	toString() {
		return this._name;
	}
	getScore() {
		return this._wins+"-"+this._losses;
	}
	addWin() {
		this._wins++;
		this._cachedSo = undefined;
	}
	addLoss() {
		this._losses++;
		this._cachedSo = undefined;
	}
	getWins() {
		return this._wins;
	}
	getLosses() {
		return this._losses;
	}
	getSortOrder() {
		if (!this._cachedSo) {
			this._cachedSo = this._wins*1e7 - this._losses*1e5 + this.getRankScore();
		}
		return this._cachedSo;
	}
	getRankScore() {
		if (!this._cachedRs) {
			this._cachedRs = Rikishi.getRankOrder(this._rank);
		}
		return this._cachedRs;
	}
}

Rikishi.getRankOrder = function(rank) {
	if (rank === "Yokozuna")  { return 600; }
	if (rank === "Ozeki")     { return 500; }
	if (rank === "Ozeki (±)") { return 400; }
	if (rank === "Sekiwake")  { return 300; }
	if (rank === "Komusubi")  { return 200; }
	if (rank.startsWith("Maegashira ")) { return (100-Number(rank.slice(11))); }
	if (rank === "Juryo")     { return 100; }
	return 0;

};