/*!
 * © 2021 David Vines
 *
 * sumoOyakata is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * sumoOyakata is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with sumoOyakata. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */

/* globals ROT, Result, Move, State, Rikishi */
/* exported Opening, Contination, Bout */

const Opening = new State("Opening", self => {
    const cha = new Move("Charge",Rikishi.prototype.getSpeed,Rikishi.prototype.getWeight);
    cha.bothSameProhibited = function(result,us,them) {
        result.addText(`Both ${us} and ${them} charge.`);
        return false;
    };
    cha.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`They crash heavily together in the middle of the dohyo.`);
    };
    cha.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} charges ${them}.`);
    };
    cha.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} left the ring first after both rikishi tumbled out of the dohyo. ${us} wins by Oshi-dashi.`);
        } else {
            result.addText(`${us} charges ${them} pushing ${them} all the way to the edge of the dohyo and out. ${us} wins by Oshi-dashi.`);
        }
    };

    const att = new Move("Attack",Rikishi.prototype.getStrength,Rikishi.prototype.getStrength);
    att.bothSameProhibited = function(result,us,them) {
        result.addText(`Both ${us} and ${them} go for the belt.`);
        return false;
    };
    att.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`They each get a grip on each other's belt.`);
    };
    att.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} grabs the belt of ${them}.`);
    };
    att.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} left the ring first after ${us} pushed ${them} all the way to the edge of the dohyo. ${us} wins by Yori-kiri.`);
        } else {
            result.addText(`${us} grabs the belt of ${them} and pushes ${them} all the way to the edge of the dohyo and out of it. ${us} wins by Yori-kiri.`);
        }
    };

    const ste = new Move("Step Aside",Rikishi.prototype.getAgility,Rikishi.prototype.getSpeed);
    ste.bothSameProhibited = function(result,us,them) {
        result.addText(`Both ${us} and ${them} attempt to step to the side of the other.`);
        return false;
    };
    ste.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi succeed, resulting in a circling of the dohyo.`);
    };
    ste.addSuccessfulDescription = function(result,us,them,theirSuccessfulMove) {
        if (theirSuccessfulMove === cha) {
            result.addText(`${us} steps to the side of ${them}.`);
        } else {
            result.addText(`${us} steps to the side and behind ${them}.`);
        }
    };
    ste.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} touched the ground first after both rikishi tumbled out of the dohyo. ${us} wins by Okuri-dashi.`);
        } else {
            result.addText(`${us} steps to the side and behind ${them} pushing ${them} out of the dohyo. ${us} wins by Okuri-dashi.`);
        }
    };

    const sla = new Move("Slap",Rikishi.prototype.getStrength,Rikishi.prototype.getWeight);
    sla.bothSameProhibited = function(result,us,them) {
        result.addText(`Both ${us} and ${them} come out slapping.`);
        return false;
    };
    sla.addBothTriedMoveAndSucceededDescription = function(_result) {
    };
    sla.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} slaps ${them}.`);
    };
    sla.addWinDescription = function(result,us,them,theirMove,afterMonoIi) {
        if (afterMonoIi) {
            if (theirMove === cha) {
                result.addText(`${us} slapped down ${them} just before leaving the dohyo, so ${us} wins by Hataki-komi.`);
            } else if (theirMove === sla) {
                result.addText(`${them} touched the dohyo first and so ${us} wins by Tsuki-dashi.`);
            }else {
                result.addText(`${us} slapped down ${them} winning by Tsuki-dashi.`);
            }
        } else {
            if (theirMove === cha) {
                result.addText(`${us} slaps down ${them} as ${them} charges, so ${us} wins by Hataki-komi.`);
            } else {
                result.addText(`${us} slaps down ${them} winning by Tsuki-dashi.`);
            }
        }
    };

    cha.addCounterMove(cha,13); cha.addCounterMove(att, 6); cha.addCounterMove(ste,18); cha.addCounterMove(sla, 9);
    att.addCounterMove(cha,18); att.addCounterMove(att,13); att.addCounterMove(ste, 6); att.addCounterMove(sla,12);
    ste.addCounterMove(cha, 6); ste.addCounterMove(att,18); ste.addCounterMove(ste,13); ste.addCounterMove(sla,15);
    sla.addCounterMove(cha,15); sla.addCounterMove(att,12); sla.addCounterMove(ste, 9); sla.addCounterMove(sla,13);
    self.addCommonMoves([cha,att,ste,sla]);
});

const Continuation = new State("Continuation", self => {
    const lev = new Move("Throw",Rikishi.prototype.getAgility,Rikishi.prototype.getStrength);
    lev.bothSameProhibited = function(_result,_us,_them) {
        return false;
    };
    lev.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi attempt to throw their opponent, but neither gains the advantage.`);
    };
    lev.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} tries to throw ${them}.`);
    };
    lev.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} touched the dohyo first so ${us} wins by Uwate-nage.`);
        } else {
            result.addText(`${us} throws ${them}. ${us} wins by Uwate-nage.`);
        }
    };

    const lif = new Move("Lift",Rikishi.prototype.getStrength,Rikishi.prototype.getWeight);
    lif.bothSameProhibited = function(_result,_us,_them) {
        return false;
    };
    lif.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi attempt to lift their opponent, but neither gains the advantage.`);
    };
    lif.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} attempts to lift ${them}.`);
    };
    lif.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} was lifted out of the dohyo before ${us} stepped outside and so ${us} wins by Yori-taoshi.`);
        } else {
            result.addText(`${them} is lifted out of the dohyo and ${us} wins by Yori-taoshi.`);
        }
    };

    const pus = new Move("Push",Rikishi.prototype.getWeight,Rikishi.prototype.getWeight);
    pus.bothSameProhibited = function(_result,_us,_them) {
        return false;
    };
    pus.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi attempt to push their opponent out of the dohyo, but neither gains the advantage.`);
    };
    pus.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} pushs ${them}.`);
    };
    pus.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} left the dohyo first and so ${us} wins by Oshi-dashi.`);
        } else {
            result.addText(`${them} is pushed out of the dohyo and so ${us} wins by Oshi-dashi.`);
        }
    };

    const pul = new Move("Pull",Rikishi.prototype.getStrength,Rikishi.prototype.getAgility);
    pul.bothSameProhibited = function(_result,_us,_them) {
        return false;
    };
    pul.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi attempt to pull their opponent to the ground, but neither gains the advantage.`);
    };
    pul.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} pulls on ${them}.`);
    };
    pul.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} was pulled to the clay first and so ${us} wins by Hiki-otoshi.`);
        } else {
            result.addText(`${them} is pulled to the clay and so ${us} wins by Hiki-otoshi.`);
        }
    };

    const tri = new Move("Trip",Rikishi.prototype.getSpeed,Rikishi.prototype.getAgility);
    tri.bothSameProhibited = function(_result,_us,_them) {
        return false;
    };
    tri.addBothTriedMoveAndSucceededDescription = function(result) {
        result.addText(`Both rikishi attempt to trip their opponent, but neither gains the advantage.`);
    };
    tri.addSuccessfulDescription = function(result,us,them,_theirSuccessfulMove) {
        result.addText(`${us} tries to trip ${them}.`);
    };
    tri.addWinDescription = function(result,us,them,_theirMove,afterMonoIi) {
        if (afterMonoIi) {
            result.addText(`${them} hit the clay first and so ${us} wins by Soto-gake.`);
        } else {
            result.addText(`${us} trips ${them} and so ${us} wins by Soto-gake.`);
        }
    };


    lev.addCounterMove(lev,13); lev.addCounterMove(lif,18); lev.addCounterMove(pus, 6); lev.addCounterMove(pul, 9); lev.addCounterMove(tri, 9);
    lif.addCounterMove(lev, 6); lif.addCounterMove(lif,13); lif.addCounterMove(pus,18); lif.addCounterMove(pul,12); lif.addCounterMove(tri, 6);
    pus.addCounterMove(lev,15); pus.addCounterMove(lif, 6); pus.addCounterMove(pus,13); pus.addCounterMove(pul,15); pus.addCounterMove(tri,15);
    pul.addCounterMove(lev,13); pul.addCounterMove(lif,12); pul.addCounterMove(pus, 9); pul.addCounterMove(pul,13); pul.addCounterMove(tri,13);
    tri.addCounterMove(lev,18); tri.addCounterMove(lif, 6); tri.addCounterMove(pus, 6); tri.addCounterMove(pul,18); tri.addCounterMove(tri,13);
    self.addCommonMoves([lev,lif,pus,pul,tri]);
});

const BoutComplete = new State("BoutComplete", _self => {});

Opening.bothSucceeded = function(_eastmove,_westmove) {
    return Continuation;
};
Opening.succeeded = function(_rik,_move) {
    return Continuation;
};
Opening.neitherSucceeded = function(result) {
    result.addText(`Both rikishi crash together from the tachi-ai and lock up in the middle of the dohyo with neither having the advantage.`);
    return Continuation;
};
Continuation.bothSucceeded = function(_eastmove,_westmove) {
    return Continuation;
};
Continuation.succeeded = function(_rik,_move) {
    return Continuation;
};
Continuation.neitherSucceeded = function(result) {
    result.addText(`Both rikishi remain locked up in the middle of the dohyo with neither having the advantage.`);
    return Continuation;
};

class Bout {
    constructor(east,west) {
        this._east = east;
        this._west = west;
        this._result = new Result();
    }
    resolve() {
        let time = 0;
        let state = Opening;
        while(state != BoutComplete) {
            state = this._resolveMoves(state,time++);
            this._result.newline(false);
        }
        return this._result;
    }
    _resolveMoves(state,time) {
        let eastmove, westmove, nextState;
        do {
            eastmove = this._east.getMove(this._west,state.getEastMoves(),time,this._result);
            westmove = this._west.getMove(this._east,state.getWestMoves(),time,this._result);
        }
        while(eastmove === westmove && eastmove.bothSameProhibited(this._result,this._east,this._west));
        this._result.newline(true);
        this._result.addText(`${state} ${this._east} ${eastmove} ${Math.floor(eastmove.getOdds(westmove,this._east,this._west))}%, ${this._west} ${westmove} ${Math.floor(westmove.getOdds(eastmove,this._west,this._east))}%,`);
        this._result.addText(`${this._east} saves ${Math.floor(this._east.getSaveChance(time))}%, ${this._west} saves ${Math.floor(this._west.getSaveChance(time))}%`);
        const eastroll = ROT.RNG.getPercentage();
        const westroll = ROT.RNG.getPercentage();
        this._result.addText(`${this._east} rolls ${eastroll}, ${this._west} rolls ${westroll}.`);
        const eastmovesuccess = (eastroll <= eastmove.getOdds(westmove,this._east,this._west));
        const westmovesuccess = (westroll <= westmove.getOdds(eastmove,this._west,this._east));
        const eastsaves = this._east.saves(time);
        const westsaves = this._west.saves(time);
        if (eastsaves) this._result.addText(` ${this._east} saves.`);
        if (westsaves) this._result.addText(` ${this._west} saves.`);
        this._result.newline(false);
        const eastposswin = eastmovesuccess && !westsaves;
        const westposswin = westmovesuccess && !eastsaves;
        if (eastposswin && westposswin) {
            // Both rikishi succeeded in their moves
            if (eastmove === westmove) {
                // Both made the same move
                eastmove.addBothTriedMoveAndSucceededDescription(this._result);
            } else {
                eastmove.addSuccessfulDescription(this._result,this._east,this._west,westmove);
                westmove.addSuccessfulDescription(this._result,this._west,this._east,eastmove);
            }
            // But both failed to save - Need to determine who was the fastest
            const eastspeed = ROT.RNG.getPercentage()+this._east.getBalance()/2+ROT.RNG.getUniform()/5;
            const westspeed = ROT.RNG.getPercentage()+this._west.getBalance()/2+ROT.RNG.getUniform()/5;
            this._result.newline(true);
            this._result.addText(`${this._east} speed is ${eastspeed}, ${this._west} speed is ${westspeed}.`);
            this._result.newline(false);
            if (Math.abs(eastspeed-westspeed) < 40) {
                this._result.addText(`A mono-ii is called and it is determined that`);
            }
            if (eastspeed > westspeed) {
                eastmove.addWinDescription(this._result,this._east,this._west,westmove,true);
                this._result.setWinner(this._east);
                this._result.setLoser(this._west);
                nextState = BoutComplete;
            } else {
                westmove.addWinDescription(this._result,this._west,this._east,eastmove,true);
                this._result.setWinner(this._west);
                this._result.setLoser(this._east);
                nextState = BoutComplete;
            }
        } else if (eastposswin) {
            eastmove.addWinDescription(this._result,this._east,this._west,this._westmove,false);
            this._result.setWinner(this._east);
            this._result.setLoser(this._west);
            nextState = BoutComplete;
        } else if (westposswin) {
            westmove.addWinDescription(this._result,this._west,this._east,eastmove,false);
            this._result.setWinner(this._west);
            this._result.setLoser(this._east);
            nextState = BoutComplete;
        } else if (eastmovesuccess && westmovesuccess) {
            // Both rikishi succeeded in their moves, but also both saved
            if (eastmove === westmove) {
                // Both made the same move
                eastmove.addBothTriedMoveAndSucceededDescription(this._result);
            } else {
                eastmove.addSuccessfulDescription(this._result,this._east,this._west,westmove);
                westmove.addSuccessfulDescription(this._result,this._west,this._east,eastmove);
            }
            nextState = state.bothSucceeded(eastmove,westmove);
        } else if (eastmovesuccess) {
            eastmove.addSuccessfulDescription(this._result,this._east,this._west,null);
            nextState = state.succeeded(this._east,eastmove);
        } else 	if (westmovesuccess) {
            westmove.addSuccessfulDescription(this._result,this._west,this._east,null);
            nextState = state.succeeded(this._west,westmove);
        } else {
            // Neither rikishi succeeded
            nextState = state.neitherSucceeded(this._result);
        }
        return nextState;
    }
}
