/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals Util, ShipDesignType, ShipDesign, Ship */
/* exported ShipDesigner */
var ShipDesigner = {
	_player: null,
	_designId: 1,
	_design: undefined,
	_callbackWhenDesignComplete: undefined,
	_techs: undefined,
	_selectedBlockType: undefined,
	_previousElement: undefined,
	init: function() {
		document.getElementById("shipDesignOk").onclick = ShipDesigner._designComplete;
		document.getElementById("shipDesignCancel").onclick = ShipDesigner._cancelDesign;

		document.getElementById("shipDesignClassNameInput").onchange = ShipDesigner._nameChangeEvent;

		document.getElementById("shipdesign-add-block-select").onchange = ShipDesigner._addBlockTypeSelectEvent;
		document.getElementById("shipdesign-add-block-subclause-select").onchange = ShipDesigner._addBlockSelectEvent;
		document.getElementById("shipdesign-add-one").onclick = function() { ShipDesigner.addBlocks(1); };
		document.getElementById("shipdesign-add-two").onclick = function() { ShipDesigner.addBlocks(2); };
		document.getElementById("shipdesign-add-three").onclick = function() { ShipDesigner.addBlocks(3); };
		document.getElementById("shipdesign-add-six").onclick = function() { ShipDesigner.addBlocks(6); };
		document.getElementById("shipdesign-add-twelve").onclick = function() { ShipDesigner.addBlocks(12); };
	},
	setPlayer: function(player) {
		ShipDesigner._player = player;
	},
	review: function(previousElementId,designOrShip,costArray) {
		const design = (designOrShip instanceof ShipDesign ? designOrShip : designOrShip.design);
		const ship = (designOrShip instanceof ShipDesign ? null : designOrShip);
		const blocks = designOrShip.getBlocks(); // Duck typing here :)
		// Hide the designer
		ShipDesigner._hideDesignerFields(true);
		ShipDesigner._previousElement = document.getElementById(previousElementId);

		ShipDesigner._previousElement.classList.add("hidden");
		document.getElementById("shipDesign").classList.remove("hidden");

		Util.replaceText("shipDesignClassName",design.name);
		if (designOrShip instanceof Ship) {
			Util.replaceText("shipDesignShipName","Ship Name: "+designOrShip.name);
			document.getElementById("shipDesignShipNameDiv").classList.remove("hidden");
		} else {
			document.getElementById("shipDesignShipNameDiv").classList.add("hidden");
		}
		ShipDesigner._populateDesignTable(design,ship,blocks,false,costArray);
		Util.replaceText("shipDesignOk","Return");

		ShipDesigner._design = undefined;
	},
	newDesign: function(designId,techs,hasBuiltAShip,callback) {
		// Show the designer fields (though we may decide to hide some of them again later :)
		ShipDesigner._hideDesignerFields(false);

		const designName = "Design-"+ShipDesigner._designId++;
		ShipDesigner._callbackWhenDesignComplete = callback;
		ShipDesigner._hasBuiltAShip = hasBuiltAShip;
		ShipDesigner._previousElement = document.getElementById("summary");

		document.getElementById("summary").classList.add("hidden");
		document.getElementById("shipDesignShipNameDiv").classList.add("hidden");
		document.getElementById("shipDesign").classList.remove("hidden");
		document.getElementById("shipDesignClassNameInput").value = designName;

		// Start with an empty design for a ship
		ShipDesigner._techs = techs;
		ShipDesigner._design = new ShipDesign(designName,ShipDesignType.Ship,designId);
		ShipDesigner._design.setFighterTechs(ShipDesigner._techs);

		ShipDesigner._populateDesignTable(ShipDesigner._design,null,ShipDesigner._design.getBlocks(),true,null);
		Util.replaceText("shipDesignOk","Add Design");
		ShipDesigner._showAvailableBlockTypes();
	},
	_showAvailableBlockTypes() {
		const typeselect = document.getElementById("shipdesign-add-block-select");
		typeselect.replaceChildren();
		Util.addOption(typeselect,"Bridge");
		Util.addOption(typeselect,"Power Plant");
		Util.addOption(typeselect,"Jump Drive");
		Util.addOption(typeselect,"Maneuver Drive");
		Util.addOption(typeselect,"Weaponary");
		Util.addOption(typeselect,"Ship Defences");
		Util.addOption(typeselect,"Marine Barracks");
		Util.addOption(typeselect,"Colonisation System");
		if (ShipDesigner._techs.arch > 1) Util.addOption(typeselect,"Neural Net");
		if (ShipDesigner._techs.arch > 2) Util.addOption(typeselect,"Jump Space Dredger");
		if (ShipDesigner._techs.arch > 3) Util.addOption(typeselect,"Tractor Beam");
		if (ShipDesigner._techs.arch > 4) Util.addOption(typeselect,"Statis Beam");
		typeselect.selectedIndex = 0;
		ShipDesigner._addBlockTypeSelectEvent(); // To force population of the subtype and enablement (as appropriate) for the add buttons
	},
	_hideDesignerFields(hide) {
		document.getElementById("shipDesignClassNameInput").classList.toggle("hidden",hide);
		document.getElementById("shipDesignClassName").classList.toggle("hidden",!hide);
		document.getElementById("shipdesign-block-adder").classList.toggle("hidden",hide);
		document.getElementById("shipDesignCancel").classList.toggle("hidden",hide);
		document.getElementById("shipDesignOk").disabled = false; // Always enable the OK button (it may get disabled later)
		document.getElementById("shipWarnings").classList.add("hidden"); // Always hidden (will be revealed if it is needed)
		document.getElementById("shipNoBridge").classList.add("hidden"); // Always hidden (will be revealed if it is needed)
		document.getElementById("ShipDesignTable").classList.toggle("review-design",hide);
	},
	_populateDesignTable(design,ship,blocks,designing,costArray) {
		const table = document.getElementById("ShipDesignTable");
		table.classList.toggle("design-is-base",design.type !== ShipDesignType.Ship && !designing); // Hide the jump energy for definite bases

		// Delete any existing data rows (i.e. all rows after the first)
		while(table.rows.length > 1) {
			table.deleteRow(1);
		}

		let idCell = null;
		let blockId = null;
		let blockSize = null;
		let blockCount = null;
		let countCell = null;
		let costCell = null;
		let costPerBlock = 0;
		let energyCell = null;
		let energyPerBlock = 0;
		let jumpCell = null;
		let jumpPerBlock = 0;

		const updateRow = function() {
			blockCount++;
			idCell.replaceChildren();
			if (blockSize*blockCount > 1) {
				const lastBlockIdEnd = blockId + blockSize * blockCount - 1;
				idCell.append(blockId+"-"+lastBlockIdEnd);
			} else {
				idCell.append(blockId);
			}
			countCell.replaceChildren();
			countCell.append(blockCount);
			costCell.replaceChildren();
			costCell.append(costPerBlock*blockCount);

			const eps  = energyPerBlock * blockCount;
			const epsj = jumpPerBlock   * blockCount;
			let energy = "";
			let jump = "";
			if (eps > 0) {
				energy = eps+" EPs generated";
				jump = energy;
			} else if (eps < 0) {
				energy = (-eps)+" EPs";
			} else if (epsj < 0) {
				jump = (-epsj)+" EPs";
			}

			energyCell.replaceChildren(); energyCell.append(energy);
			jumpCell.replaceChildren();   jumpCell.append(jump);
		};

		const addRow = function(blocktype,startid,size,description,title,cost,epb,jpb,designing) {
			const newrow = document.createElement("tr");
			idCell = document.createElement("td"); newrow.append(idCell);
			countCell = document.createElement("td"); newrow.append(countCell);
			const descr = document.createElement("td");  newrow.append(descr);
			descr.append(description);
			descr.title = title;
			costCell = document.createElement("td"); newrow.append(costCell);
			energyCell = document.createElement("td"); newrow.append(energyCell);
			jumpCell = document.createElement("td"); newrow.append(jumpCell);
			const buttonCell = document.createElement("td");
			if (designing && blocktype) {
				const button = document.createElement("button");
				button.append("Delete one block"); buttonCell.append(button);
				button.title = "Delete a single "+description+" from this design";
				button.onclick = function() { ShipDesigner._deleteBlock(blocktype); };
			}
			newrow.append(buttonCell);
			table.append(newrow);

			if (size) {
				blockId = startid; blockSize = size;
				blockCount = 0; // Will be incremented to one by updateRow
				costPerBlock = cost;
				energyPerBlock = epb;
				jumpPerBlock = jpb;
				updateRow();
			} else {
				// Special case for 0-size "blocks" like the hull
				if (cost) costCell.append(cost);
				if (epb) energyCell.append(epb);
				if (jpb) jumpCell.append(jpb);
			}
		};

		addRow(null,null,0,"Hull","",design.hullCost()-10,null,null,false);

		let blocknumber = 1;
		let lastname = null;
		for(let block of blocks) {
			const full = block.getFullName(designing || Boolean(ship));
			if (lastname && full == lastname) {
				updateRow();
			} else {
				lastname = full;
				addRow(block.key,blocknumber,block.size,lastname,block.description,block.cost,block.eps,block.epsj,designing);
			}
			blocknumber += block.size;
		}

		// Stats
		if (costArray) {
			const [cost,firstOfDesign] =costArray;
			let firstPrice = cost, nextPrice = cost;
			if (firstOfDesign) nextPrice /= 2;
			Util.replaceText("shipdesignNextPrice",nextPrice);
			Util.replaceText("shipdesignFirstPrice",firstPrice);
		} else {
			let cost = design.totalCost();
			if (ShipDesigner._hasBuiltAShip && design.numberOfClassBuilt(ShipDesigner._player) == 0) {
				Util.replaceText("shipdesignNextPrice",cost);
				Util.replaceText("shipdesignFirstPrice",cost*2);
			} else  { // It's either the very first ship or a subsequent one
				Util.replaceText("shipdesignNextPrice",cost);
				Util.replaceText("shipdesignFirstPrice",cost);
			}
		}

		const fighterSpec = design.fighterSpec();
		document.getElementById("fighterDesc").classList.toggle("hidden",(fighterSpec == null));
		document.getElementById("designHasFighters").classList.toggle("hidden",(fighterSpec == null));
		if (fighterSpec) {
			Util.replaceText("fighterDescCombatSpeed",fighterSpec.speed);
			Util.replaceText("fighterDescWeapon",fighterSpec.weaponDesc);
			Util.replaceText("fighterDescCost",fighterSpec.cost);
		}

		if (design.getJumpRange() > 0) {
			document.getElementById("designCanJump").classList.remove("hidden");
			document.getElementById("designCantJump").classList.add("hidden");
			Util.replaceText("shipdesignJump",design.getJumpRange());
			if (design.getJumpRange() == 1) {
				Util.replaceText("designJumpUnit","hex");
			} else {
				Util.replaceText("designJumpUnit","hexes");
			}
			if (ship && (ship.getJumpRange() !== design.getJumpRange())) {
				document.getElementById("designJumpDamaged").classList.remove("hidden");
				Util.replaceText("designJumpDamaged",` (However in its current damaged state it can only achieve a jump of ${ship.getJumpRange()} hex${ship.getJumpRange()==1?"":"es"})`);
			} else {
				document.getElementById("designJumpDamaged").classList.add("hidden");
			}
		} else {
			document.getElementById("designCanJump").classList.add("hidden");
			document.getElementById("designCantJump").classList.remove("hidden");
		}

		if (design.hasWeapons()) {
			document.getElementById("designHasWeapons").classList.remove("hidden");
			const hits = design.getAverageHits();
			Util.replaceText("shipDesignHitCount",hits);
			Util.replaceText("shipDesignHitUnit",hits==1 ? "hit" : "hits");
		} else {
			document.getElementById("designHasWeapons").classList.add("hidden");
		}

		const [minSurvive,avgSurvive,maxSurvive] = design.getEstimatedDurability();
		Util.replaceText("shipDesignMinHitsSurvived",minSurvive);
		Util.replaceText("shipDesignAverageHitsSurvived",avgSurvive);
		Util.replaceText("shipDesignAverageHitsSurvivedUnit",avgSurvive==1 ? "hit" : "hits");
		Util.replaceText("shipDesignMaxHitsSurvived",maxSurvive);
		Util.replaceText("shipDesignMaxHitsSurvivedUnit",maxSurvive==1 ? "hit" : "hits");

		Util.replaceText("shipdesignSpeed",design.getCombatSpeed());
		if (ship && (ship.getCombatSpeed() !== design.getCombatSpeed())) {
			document.getElementById("designSpeedDamaged").classList.remove("hidden");
			Util.replaceText("designSpeedDamaged",` (However in its current damaged state it can only achieve a combat speed of ${ship.getCombatSpeed()})`);
		} else {
			document.getElementById("designSpeedDamaged").classList.add("hidden");
		}

		// Warnings (and no bridge error)
		if (designing) {
			const noBridge = !ShipDesigner._design.hasBridge();
			const manSpeed = ShipDesigner._design.getManeuverSpeed();
			const jump = ShipDesigner._design.getJumpRange();
			document.getElementById("shipNoBridge").classList.toggle("hidden",!noBridge);
			if (manSpeed == 0 && jump == 0 && ShipDesigner._techs.planet > 2) {
				if (ShipDesigner._techs.planet == 3) {
					ShipDesigner._design.type = ShipDesignType.SpaceBase;
				} else {
					ShipDesigner._design.type = ShipDesignType.StarBase;
				}
			} else {
				ShipDesigner._design.type = ShipDesignType.Ship;
			}
			document.getElementById("shipDesignOk").disabled = noBridge;
			ShipDesigner._enableAddButtons();
			document.getElementById("spaceBaseEligible").classList.toggle("hidden",ShipDesigner._design.type !== ShipDesignType.SpaceBase);
			document.getElementById("starBaseEligible").classList.toggle("hidden",ShipDesigner._design.type !== ShipDesignType.StarBase);
		} else {
			document.getElementById("spaceBaseEligible").classList.toggle("hidden",design.type !== ShipDesignType.SpaceBase);
			document.getElementById("starBaseEligible").classList.toggle("hidden",design.type !== ShipDesignType.StarBase);
		}
	},
	_addBlockTypeSelectEvent: function() {
		const typeselect = document.getElementById("shipdesign-add-block-select");
		const subselect = document.getElementById("shipdesign-add-block-subclause-select");
		subselect.replaceChildren();
		const addOptions = function(max,opts,types) {
			for(let i=0; i<max; i++) {
				Util.addOption(subselect,opts[i],{block: types[i]});
			}
			subselect.selectedIndex = max-1;
			ShipDesigner._selectedBlockType = types[max-1];
		};
		switch(typeselect.options[typeselect.selectedIndex].text) {
			case "Bridge":
				addOptions(ShipDesigner._techs.computer+1,
					["with no computer","with TL1 computer","with TL2 computer","with TL3 computer","with TL4 computer","with TL5 computer","with TL6 computer","with TL7 computer","with TL8 computer"],
					["_b0","_b1","_b2","_b3","_b4","_b5","_b6","_b7","_b8","_b9"]
				);
				break;
			case "Power Plant":
				addOptions(ShipDesigner._techs.power,
					["(Chemical)","(Atomic)","(Fusion)","(Large Fusion)","(Anti-matter)","(Large Anti-matter)","(Ultimate)"],
					["_p1","_p2","_p3","_p4","_p5","_p6","_p7"]
				);
				break;
			case "Jump Drive":
				addOptions(ShipDesigner._techs.jump,
					["-A","-B","-C","-D","-E","-F","-G","-H"],
					["_ja","_jb","_jc","_jd","_je","_jf","_jg","_jh"]
				);
				break;
			case "Maneuver Drive":
				addOptions(ShipDesigner._techs.man,
					["-A","-B","-C","-D","-E","-F","-G","-H"],
					["_ma","_mb","_mc","_md","_me","_mf","_mg","_mh"]
				);
				break;
			case "Weaponary":
				addOptions(ShipDesigner._techs.weapons,
					["Laser Battery","Phaser Battery","Plasma Gun","Fusion Gun","Particle Accelerator Spinal Mount","Particle Acclerator","Meson Gun Spinal Mount","Meson Gun","Anti-matter Spinal Mount","Anti-matter Gun","Planet Buster"],
					["_wl","_wp","_wa","_wf","_wr","_wc","_ws","_wm","_wt","_wx","_wb"]
				);
				break;
			case "Ship Defences":
				addOptions(ShipDesigner._techs.defences,
					["Armour","Damage Control","Shield","Fighter","Improved Shield","Regenerating Shield","Reflecting Shield","Invulnerability Shield"],
					["_da","_dc","_ds","_df","_di","_dg","_dr","_di"]
				);
				break;
			case "Marine Barracks":
				ShipDesigner._selectedBlockType = "_bb";
				break;
			case "Colonisation System":
				ShipDesigner._selectedBlockType = "_cb";
				break;
			case "Neural Net":
				ShipDesigner._selectedBlockType = "_an";
				break;
			case "Jump Space Dredger":
				ShipDesigner._selectedBlockType = "_ad";
				break;
			case "Tractor Beam":
				ShipDesigner._selectedBlockType = "_at";
				break;
			case "Statis Beam":
				ShipDesigner._selectedBlockType = "_as";
				break;
		}

		document.getElementById("shipdesign-add-block-subclause-div").classList.toggle("hidden",subselect.options.length == 0);
		ShipDesigner._enableAddButtons();
	},
	_enableAddButtons: function() {
		const currentSize = ShipDesigner._design.totalSize();
		const blocksize = ShipDesign.SHIPBLOCKSIZES[ShipDesigner._selectedBlockType];
		const buttons = ["shipdesign-add-one","shipdesign-add-two","shipdesign-add-three","shipdesign-add-six","shipdesign-add-twelve"];
		const counts = [1,2,3,6,12];
		const max = [10,20,30,40,60,80,100,150,1e9][ShipDesigner._techs.size-1];
		Util.replaceText("tooLargeWarningMaxSize",max);
		for(let i=0; i<buttons.length; i++) {
			document.getElementById(buttons[i]).disabled = currentSize+blocksize*counts[i] > max;
		}
		const tooLarge = (currentSize == max);
		document.getElementById("tooLargeWarning").classList.toggle("hidden", !tooLarge );
		const spaceBaseDredger = (ShipDesigner._design.type == ShipDesignType.SpaceBase && ShipDesigner._design._ad > 0);
		document.getElementById("spaceBaseDredger").classList.toggle("hidden", !spaceBaseDredger);
		document.getElementById("shipWarnings").classList.toggle("hidden",(!tooLarge && !spaceBaseDredger));
	},
	_addBlockSelectEvent: function() {
		const subselect = document.getElementById("shipdesign-add-block-subclause-select");
		const option = subselect.options[subselect.selectedIndex];
		ShipDesigner._selectedBlockType = option.dataset.block;
		ShipDesigner._enableAddButtons();
	},
	addBlocks: function(amount) {
		for(let i=0; i<amount; i++) {
			ShipDesigner._design.addBlock(ShipDesigner._selectedBlockType);
		}
		ShipDesigner._populateDesignTable(ShipDesigner._design,null,ShipDesigner._design.getBlocks(),true);
	},
	_deleteBlock: function(blocktype) {
		ShipDesigner._design.deleteBlock(blocktype);
		ShipDesigner._populateDesignTable(ShipDesigner._design,null,ShipDesigner._design.getBlocks(),true);
	},
	_nameChangeEvent: function() {
		ShipDesigner._design.name = document.getElementById("shipDesignClassNameInput").value;
	},
	_cancelDesign: function() {
		document.getElementById("summary").classList.remove("hidden");
		document.getElementById("shipDesign").classList.add("hidden");
		ShipDesigner._callbackWhenDesignComplete(null); // Design was cancelled
	},
	_designComplete: function() {
		ShipDesigner._previousElement.classList.remove("hidden");
		document.getElementById("shipDesign").classList.add("hidden");
		if (ShipDesigner._design) {
			ShipDesigner._callbackWhenDesignComplete(ShipDesigner._design);
		}
	}
};