/*!
 * © 2021, 2022 David Vines
 *
 * sumoOyakata is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * sumoOyakata is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with sumoOyakata. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */

/* globals Names, Rikishi, Bout, ROT */
var Game =  {
	MAKUUCHI_SIZE: 42,
	_gameSpeed: 1000,
	_emptyList: function(id) {
		const element = document.getElementById(id);
		while (element.firstChild) {
			element.removeChild(element.firstChild);
		}
		return element;
	},
	_replaceText: function(id,text) {
		let node = document.getElementById(id);
		node.replaceChild(document.createTextNode(text),node.firstChild);
	},
	_sortTable: function(champion) {
		Game.rikishi.sort(function(a,b) {
			if (champion && a === champion) return -1; // To ensure the champ (if any) is at the top of the table
			if (champion && b === champion) return 1;  // To ensure the champ (if any) is at the top of the table
			if (a.getSortOrder() > b.getSortOrder()) return -1;
			if (b.getSortOrder() > a.getSortOrder()) return 1;
			return a.toString().localeCompare(b.toString());
		});
	},
	_updateTable: function(champion) {
		Game._sortTable(champion);
		const selfGame = Game;
			const modal = document.getElementById("rikishiData");
			const list = document.getElementById("rikishiDataList");
		for(let i=0; i<Game.MAKUUCHI_SIZE; i++) {
			const rik = Game.rikishi[i];
			Game._replaceText("r"+i+"name" ,rik.toString());
			Game._replaceText("r"+i+"rank" ,rik.getRank());
			Game._replaceText("r"+i+"int"  ,rik.getIntelligence());
			Game._replaceText("r"+i+"spd"  ,rik.getSpeed());
			Game._replaceText("r"+i+"agl"  ,rik.getAgility());
			Game._replaceText("r"+i+"str"  ,rik.getStrength());
			Game._replaceText("r"+i+"wgt"  ,rik.getWeight());
			Game._replaceText("r"+i+"bal"  ,rik.getBalance());
			Game._replaceText("r"+i+"end"  ,rik.getEndurance());
			Game._replaceText("r"+i+"score",rik.getScore());
			const score = document.getElementById("r"+i+"score");
			if (rik.getWins() > 7) {
				score.className = "Kachikoshi";
			} else if (rik.getLosses() > 7) {
				score.className = "Makekoshi";
			} else {
				score.className = "";
			}
			const nameElement = document.getElementById("r"+i+"name");
			nameElement.onclick = function() {
				selfGame._replaceText("rikishiDataName",rik.getRank()+" "+rik.toString());
				list.replaceChildren();
				for(let item of rik.getHistory()) {
					const term = document.createElement("dt");
					term.append(item.basho);
					list.append(term);
					const def = document.createElement("dd");
					if (item.rank) def.append("Started as "+item.rank+".");
					if (item.rank && item.wins) def.append(" ");
					if (item.wins) def.append("Result: "+item.wins+"-"+(15-item.wins));
					if (item.champion) {
						def.append(" ");
						const bold = document.createElement("b");
						bold.append("Champion");
						def.append(bold);
					}
					list.append(def);
				}
				modal.open();
			};
		}
	},
	_combatantText: function(east,west) {
		return east.getRank()+" "+east.toString()+" ("+east.getScore()+") and "+west.getRank()+" "+west.toString()+" ("+west.getScore()+")";
	},
	_setupDay: function(day) {
		const cDay = day;
		let tryWindow = 16-day; /* Initial window is quite narrow at the end of the basho, wide at the start */
		Game._todaysBouts = [];
		while(Game._todaysBouts.length === 0) {
			let rikishiOrder = ROT.RNG.shuffle(Game.rikishi.slice());
			while(rikishiOrder.length != 0) {
				const firstRikishi = rikishiOrder.shift();
				// clone the array, remove those rikishi first has already played and then choose one of the others (if any)
				const orsm = rikishiOrder.filter(function(r) { return !r.hasPlayed(firstRikishi); } );
				if (orsm.length !== 0) {
					const winSortOrder = function(a,b) {
						const score = function(r) {
							const sortOrder = Math.abs(r.getRankScore() - firstRikishi.getRankScore());
							const winOrder = Math.abs(r.getWins() - firstRikishi.getWins()) * 40; // To scale up to 600 for 15 wins
							const factor = (cDay > 10 ? 0 : (11-cDay)/10);
							return sortOrder*factor + winOrder*(1-factor);
						};
						return score(a) - score(b);
					};
					const orsmo = orsm.sort(winSortOrder);
					const orsmos = orsmo.slice(0,tryWindow);
					const otherRikishi = ROT.RNG.getItem(orsmos);
					Game._todaysBouts.push([firstRikishi,otherRikishi]);
					rikishiOrder = rikishiOrder.filter(function(r) { return r !== otherRikishi; } );
				} else {
					// Abort this attempt since no other unplayed rikishi left
					Game._todaysBouts = [];
					rikishiOrder = [];
					tryWindow++; // Widen the window to allow to choose from another rikishi
				}
			}
		}
		// But rikishi with the lowest wins totals first (and lowest rankscore on ties)
		const gameSortOrder = function(a,b) {
			return (a[0].getWins()+a[1].getWins()+a[0].getRankScore()/1000+a[1].getRankScore()/1000) - (b[0].getWins()+b[1].getWins()+b[0].getRankScore()/1000+b[1].getRankScore()/1000);
		};
		Game._todaysBouts = Game._todaysBouts.sort(gameSortOrder);
		for(let bout of Game._todaysBouts)
		{
			const firstRikishi = bout[0];
			const otherRikishi = bout[1];
			firstRikishi.played(otherRikishi);
			otherRikishi.played(firstRikishi);
		}
	},
	_buttons : function(state) {
		document.getElementById("next").disabled = state;
		document.getElementById("save").disabled = state;
		document.getElementById("load").disabled = state;
	},
	_enableButtons : function() {
		Game._lastROTRNGstate = ROT.RNG.getState(); // To allow save/load to restore it
		Game._buttons(false);
	},
	_disableButtons : function() {
		Game._buttons(true);
	},
	_setupBout: function(east,west,boutname) {
		Game._setupBoutArgs = [east,west,boutname]; // To be saved and restored
		Game._emptyList("bout");
		east = east || Game._todaysBouts[Game._bout][0];
		west = west || Game._todaysBouts[Game._bout][1];
		boutname = boutname || Game._bout+1;
		Game._result = (new Bout(east,west)).resolve();
		Game._replaceText("boutno",boutname);
		Game._replaceText("eastname",(east.getRank() ? east.getRank()+" " : "")+east.toString()+" ("+east.getScore()+")");
		Game._replaceText("westname",(west.getRank() ? west.getRank()+" " : "")+west.toString()+" ("+west.getScore()+")");
		Game._resultLinesShown = 0;
		Game._replaceText("next","Next");
		Game._disableButtons();
		Game._showNextResultLine();
	},
	_endBasho: function(first) {
		if (Game._winners.length == 1) {
			Game._completeBasho();
		} else if (Game._winners.length === 2) {
			Game._replaceText("next","Tie-break between "+Game._winners[0].toString()+" and "+Game._winners[1].toString());
			Game._enableButtons();
			document.getElementById("next").onclick = function() { Game._setupBout(Game._winners[0],Game._winners[1],"Tie-breaker"); };
		} else if (Game._winners.length === 3) {
			Game._prevWinner = null;
			Game._replaceText("next",first+" Tie-break match between "+Game._winners[0].toString()+" and "+Game._winners[1].toString());
			Game._enableButtons();
			document.getElementById("next").onclick = function() { Game._setupBout(Game._winners[0],Game._winners[1],"Tie-breaker"); };
		} else {
			Game._nextRound = [];
			Game._winners = ROT.RNG.shuffle(Game._winners);
			if (Game._winners.length%2 === 1 ) {
				Game._nextRound.push(Game._winners[Game._winners.length-1]);
			}
			Game._replaceText("next",first+" Tie-break match between "+Game._winners[0].toString()+" and "+Game._winners[1].toString());
			Game._nextIndex = 2;
			Game._enableButtons();
			document.getElementById("next").onclick = function() { Game._setupBout(Game._winners[0],Game._winners[1],"Tie-breaker"); };
		}
	},
	_addLineToList: function(list,line,style) {
		const element = document.createElement("li");
		if (style) element.classList.add(style);
		const textnode = document.createTextNode(line);
		element.appendChild(textnode);
		list.appendChild(element);
	},
	_showNextResultLine: function() {
		if (Game._resultLinesShown < Game._result.getText().length) {
			do {
				Game._addLineToList(document.getElementById("bout"),Game._result.getText()[Game._resultLinesShown],Game._result.getStyle()[Game._resultLinesShown]);
				Game._resultLinesShown++;
			} while(Game._resultLinesShown < Game._result.getText().length && Game._result.getStyle()[Game._resultLinesShown] !== "");
			window.setTimeout(Game._showNextResultLine, Game._gameSpeed);
		} else {
			if (Game._winners === null) {
				// Regular bout completed
				Game._result.getWinner().addWin();
				Game._result.getLoser().addLoss();
				Game._updateTable();
				if (Game._bout < Game.MAKUUCHI_SIZE/2-1) {
					Game._bout++;
					Game._replaceText("next","Next Bout is between "+Game._combatantText(Game._todaysBouts[Game._bout][0],Game._todaysBouts[Game._bout][1]));
					Game._enableButtons();
					document.getElementById("next").onclick = function() { Game._setupBout(); };
				} else if (Game._day < 14) {
					Game._day++;
					Game._setupDay(Game._day);
					Game._bout = 0;
					Game._replaceText("next","Next Day. First bout is between "+Game._combatantText(Game._todaysBouts[Game._bout][0],Game._todaysBouts[Game._bout][1]));
					Game._enableButtons();
					document.getElementById("next").onclick = function() {
						Game._replaceText("day",Game._day+1);
						Game._setupBout();
					};
				} else {
					Game._winners = [Game.rikishi[0]];
					for(let i=1; (Game.rikishi[i].getWins() == Game.rikishi[0].getWins()) && i<Game.MAKUUCHI_SIZE; i++) {
						Game._winners.push(Game.rikishi[i]);
					}
					Game._endBasho("First");
				}
			} else if (Game._winners.length === 1) {
				// All done
				throw new Error("Got into _showNextResult while Game._winners.length === 1");
			} else if (Game._winners.length === 2) {
				Game._winners = [Game._result.getWinner()];
				Game._completeBasho();
			} else if (Game._winners.length === 3) {
				if (Game._result.getWinner() === Game._prevWinner) {
					Game._winners = [Game._result.getWinner()];
					Game._completeBasho();
				} else {
					Game._prevWinner = Game._result.getWinner();
					const east = Game._prevWinner;
					let west = Game._winners[0];
					if (west === Game._result.getWinner() || west === Game._result.getLoser()) west = Game._winners[1];
					if (west === Game._result.getWinner() || west === Game._result.getLoser()) west = Game._winners[2];
					Game._replaceText("next","Next Tie-break match between "+east+" and "+west);
					Game._enableButtons();
					document.getElementById("next").onclick = function() { Game._setupBout(east,west,"Tie-breaker"); };
				}
			} else {
				// Game._winners.length > 3
				Game._nextRound.push(Game._result.getWinner());
				if (Game._nextIndex >= Game._winners.length-1) {
					Game._winners = Game._nextRound;
					Game._endBasho("Next");
				} else {
					Game._replaceText("next","Next Tie-break match between "+Game._winners[Game._nextIndex].toString()+" and "+Game._winners[Game._nextIndex+1].toString());
					Game._enableButtons();
					const east = Game._winners[Game._nextIndex];
					const west = Game._winners[Game._nextIndex+1];
					document.getElementById("next").onclick = function() { Game._setupBout(east,west,"Tie-breaker"); };
					Game._nextIndex += 2;
				}
			}
		}
	},
	_completeBasho() {
		const para = document.getElementById("para");
		Game._prevChampion = Game._champion;
		Game._prevChampionWasOzeki = Game._championWasOzeki;
		Game._champion = Game._winners[0];
		Game._championWasOzeki = Game._champion.getRank().startsWith("Ozeki"); // Note: needs to be recorded before promotions
		para.className = para.className.replace(/hidden/,"visible");
		Game._replaceText("winner",Game._champion.toString());
		Game._sortTable(Game._champion);
		const events = Game._emptyList("events");
		events.className = para.className.replace(/hidden/,"visible");

		for(let rikishi of Game.rikishi) {
			rikishi.recordBasho(rikishi === Game._champion);
		}

		let east = true;
		let rank = 1;
		let sekiwakeCount = 0;
		let komusubiCount = 0;
		let leavers = []; // Also those demoted out of Makunochi
		if (Game._season === 0 && Game._year === Game._firstYear) {
			for(let rikishi of Game.rikishi) {
				if (rikishi === Game._champion) {
					rikishi.setRank("Sekiwake");
					sekiwakeCount++;
				} else if (rikishi.getWins() === Game._champion.getWins()) {
					if (sekiwakeCount <2) {
						rikishi.setRank("Sekiwake");
						sekiwakeCount++;
					} else {
						rikishi.setRank("Komusubi");
						komusubiCount++;
					}
				} else {
					if (sekiwakeCount < 2) {
						rikishi.setRank("Sekiwake");
						sekiwakeCount++;
					} else if (komusubiCount < 2) {
						rikishi.setRank("Komusubi");
						komusubiCount++;
					} else {
						rikishi.setRank("Maegashira "+rank);
						if (!east) rank++;
						east = !east;
					}
				}
			}
		} else {
			// Scan the rikishi looking for promotions and demotions based their current rank
			let maega = []; // Maegeshira ranks
			let fillMaega = function(rank,r) {
				rank = rank*2+15; // To allow two ranks at the same-ish level) and to allow good high rank Maegeshira to leapfrog poor Komusubi and Sekiwake
				if (rank<0) { rank=0; } // Avoid negatives (the above *should* have avoided this, this is belt and braces stuff)
				let unfilled = true;
				while(unfilled) {
					if (!maega[rank]) {
						maega[rank] = r; unfilled = false;
					} else {
						rank++;
					}
				}
			};
			let remainingRikishi = Game.rikishi.slice();
			let nextRikishi = [];
			for(let rikishi of remainingRikishi) {
				if (rikishi.getRank() === "Yokozuna") {
					if (rikishi.getWins() <8) {
						Game._addLineToList(events,"Yokozuna "+rikishi.toString()+" retires following his poor showing at this Basho");
						leavers.push(rikishi);
					}
				} else {
					nextRikishi.push(rikishi);
				}
			}
			remainingRikishi = nextRikishi;
			nextRikishi = [];
			for(let rikishi of remainingRikishi) {
				if (rikishi.getRank() === "Ozeki") {
					if (rikishi === Game._champion &&  Game._champion === Game._prevChampion && Game._prevChampionWasOzeki) {
						Game._addLineToList(events,rikishi.toString()+" is promoted to Yokozuna following his two successive Basho championships");
						rikishi.setRank("Yokozuna");
					} else if (rikishi.getWins() < 8) {
						rikishi.setRank("Ozeki (±)");
						Game._addLineToList(events,rikishi.toString()+" is now a Kadoban Ozeki");
					}
				} else if (rikishi.getRank() === "Ozeki (±)") {
					// Cannot have won the previous basho since only seven wins (at best) cannot win the basho
					if (rikishi === Game._champion) {
						Game._prevOzekiChampion = Game._champion;
						rikishi.setRank("Ozeki");
						Game._addLineToList(events,rikishi.toString()+" is no longer a Kadoban Ozeki");
					} else if (rikishi.getWins() < 8) {
						Game._addLineToList(events,rikishi.toString()+" is demoted to Sekiwake");
						rikishi.setRank("Sekiwake");
						sekiwakeCount++;
					} else {
						rikishi.setRank("Ozeki");
						Game._addLineToList(events,rikishi.toString()+" is no longer a Kadoban Ozeki");
					}
				} else {
					nextRikishi.push(rikishi);
				}
			}
			remainingRikishi = nextRikishi;
			nextRikishi = [];
			for(let rikishi of remainingRikishi) {
				if (rikishi.getRank() === "Sekiwake") {
					if (rikishi.getLastBashoWins() > 32) {
						// Fudge the decision making process - 33 wins is enough in all cases for this game
						Game._addLineToList(events,rikishi.toString()+" is promoted to Ozeki after winning "+rikishi.getLastBashoWins()+" bouts in the last three Bashos");
						rikishi.setRank("Ozeki");
					} else if (rikishi.getWins() < 8) {
						rikishi.setRank("Komusubi");
						Game._addLineToList(events,rikishi.toString()+" is demoted to Komusubi");
						komusubiCount++;
					} else {
						sekiwakeCount++;
					}
				} else {
					nextRikishi.push(rikishi);
				}
			}
			remainingRikishi = nextRikishi;
			nextRikishi = [];
			for(let rikishi of remainingRikishi) {
				if (rikishi.getRank() === "Komusubi") {
					if (rikishi.getLastBashoWins() > 32) {
						// Fudge the decision making process - 33 wins is enough in all cases for this game
						Game._addLineToList(events,rikishi.toString()+" is promoted to Ozeki after winning "+rikishi.getLastBashoWins()+" bouts in the last three Bashos");
						rikishi.setRank("Ozeki");
					} else if (rikishi.getWins() < 8) {
						Game._addLineToList(events,rikishi.toString()+" is demoted to Maegashira");
						fillMaega(1,rikishi);
					} else if (rikishi.getWins() > 9 || sekiwakeCount < 2) {
						rikishi.setRank("Sekiwake");
						Game._addLineToList(events,rikishi.toString()+" is promoted to Sekiwake");
						sekiwakeCount++;
					} else {
						komusubiCount++;
					}
				} else {
					nextRikishi.push(rikishi);
				}
			}
			remainingRikishi = nextRikishi;
			for(let rikishi of remainingRikishi) {
				if (sekiwakeCount <2) {
					rikishi.setRank("Sekiwake");
					Game._addLineToList(events,rikishi.toString()+" is promoted to Sekiwake to bring the total number to two");
					sekiwakeCount++;
				} else if (komusubiCount < 2) {
					rikishi.setRank("Komusubi");
					Game._addLineToList(events,rikishi.toString()+" is promoted to Komusubi to bring the total number to two");
					komusubiCount++;
				} else if (rikishi.getWins() > 11) {
					rikishi.setRank("Komusubi");
					Game._addLineToList(events,rikishi.toString()+" is promoted to Komusubi");
					komusubiCount++;
				} else {
					// Must be a Maegeshira
					const previously = Number(rikishi.getRank().slice(11));
					const diff = rikishi.getLosses()-rikishi.getWins();
					const newrank = previously + diff;
					const maxRank = 20; // Have to fall significantly out of the ranking to be demoted
					if (newrank <= maxRank || rikishi.getWins() > 7) {
						fillMaega(newrank,rikishi);
					} else {
						rikishi.setRank("Juryo");
						Game._addLineToList(events,rikishi.toString()+" is demoted to Juryo");
						leavers.push(rikishi);
					}

				}
			}
			// Now need to set the Maegeshira
			for(let rikishi of maega) {
				if (rikishi) {
					rikishi.setRank("Maegashira "+rank);
					if (!east) rank++;
					east = !east;
				}
			}
		}

		Game._updateTable(Game._champion);

		Game._joiners = [];
		for(let [index,rikishi] of Game.rikishi.entries()) {
			const thisrik = rikishi;
			if (leavers.some(rik => rik === thisrik)) {
				let r = null;
				while(!r) {
					const newrik = new Rikishi();
					if (!Game.rikishi.some(rik => newrik.toString() === rik.toString())) {
						r = newrik;
					}
				}
				Game._addLineToList(events,r.toString()+" is promoted from Juryo");
				Game._joiners.push(r);
				r.setRank("Maegashira "+rank);
				if (!east) rank++;
				east = !east;
				Game.rikishi[index] = r;
			}
		}

		Game._betweenBashos = true;
		Game._replaceText("next","Next Basho");
		Game._enableButtons();
		document.getElementById("next").onclick = function() { Game._updateInfo(); Game._prepareBasho(); };
	},
	_updateInfo() {
		let champion = Game._champion;
		if (Game._betweenBashos && Game._prevChampion) {
			champion = Game._prevChampion;
		}
		if (champion) {
			Game._replaceText("prevChampion",champion.toString());
		} else {
			Game._replaceText("prevChampion"," ");
		}
		const lasttwo = Game._emptyList("lasttwowins");
		const map = new Map();
		for(let rik of Game.rikishi) {
			if (rik.getRank() === "Sekiwake" | rik.getRank() === "Komusubi") {
				map.set(rik.toString(),rik.getLastBashoWins());
			}
		}
		const sortedmap = new Map([...map].sort((a,b) => b[1] - a[1] ));
		for(let [name,wins] of sortedmap) {
			Game._addLineToList(lasttwo,wins+" "+name);
		}
		const joiners = Game._emptyList("newfromJuryo");
		if (Game._joiners) {
			for(let rik of Game._joiners) {
				Game._addLineToList(joiners,rik.toString());
			}
		}
	},
	_prepareBasho() {
		const para = document.getElementById("para");
		para.className = para.className.replace(/visible/,"hidden");
		const events = document.getElementById("events");
		events.className = para.className.replace(/visible/,"hidden");
		Game._season++; if (Game._season === Game._SEASONS.length) { Game._season = 0; Game._year++;}
		Game._replaceText("season",Game._SEASONS[Game._season]);
		Game._replaceText("year",Game._year);
		Game._winners = null;
		Game._betweenBashos = false;
		const bashoName = Game._SEASONS[Game._season] + " " + Game._year;
		for(let rikishi of Game.rikishi) {
			rikishi.prepBasho(bashoName);
		}
		Game._updateTable();
		Game._replaceText("day",1);
		Game._setupDay(1);
		Game._day = 0;
		Game._bout = 0;
		Game._winners = null;
		Game._prevWinner = null;
		Game._nextRound = null;
		Game._nextIndex = null;
		Game._updateInfo();
		Game._setupBout();
	},
	init: function() {
		Names.init();
		Game.rikishi = [new Rikishi()];
		for(let i=1; i<Game.MAKUUCHI_SIZE; i++) {
			let r = null;
			while(!r) {
				const newrik = new Rikishi();
				if (!Game.rikishi.some(rik => newrik.toString() === rik.toString())) {
					r = newrik;
				}
			}
			Game.rikishi.push(r);
		}
		Game._SEASONS = ["New Year","Spring","Summer","Nagoya","Autumn","Kyushu"];
		Game._firstYear = new Date().getFullYear();
		Game._year = Game._firstYear;
		Game._season = -1;
		Game._betweenBashos = true;
		Game._prepareBasho();

		document.getElementById("save").onclick = function() { localStorage.sumoOyakata = Game.stringify(); Game._checkLoadButton(); };

		document.getElementById("debugbutton").onclick = function() {
			document.getElementById("bout").classList.remove("debughide","debugshow");
			document.getElementById("bout").classList.add(document.getElementById("debugbutton").checked ? "debugshow" : "debughide");
		};
		document.getElementById("debugbutton").onclick();

		// Game speeds
		document.getElementById("normal").onclick  = function() { Game._gameSpeed = 1000; };
		document.getElementById("fast").onclick    = function() { Game._gameSpeed = 500;  };
		document.getElementById("quick").onclick   = function() { Game._gameSpeed = 100;  };
		document.getElementById("instant").onclick = function() { Game._gameSpeed = 1;    };

		Game._checkLoadButton();
	},
	stringify: function() {
		let data = {};
		data._saved = new Date();
		data._version = 4;
		data._bout = Game._bout;
		data._day = Game._day;
		data._firstYear = Game._firstYear;
		data._season = Game._season;
		data._todaysBouts = [];
		for(let bout of Game._todaysBouts) {
			data._todaysBouts.push([bout[0].toString(),bout[1].toString()]);
		}
		if (Game._winners) {
			data._winners = [];
			for (let winner of Game._winners) {
				data._winners.push(winner.toString());
			}
		}
		if (Game._prevWinner) {
			data._prevWinner = Game._prevWinner.toString();
		}
		if (Game._nextRound) {
			data._nextRound = [];
			for(let rounder of Game._nextRound) {
				data._nextRound.push(rounder.toString());
			}
		}
		if (Game._nextIndex) {
			data._nextIndex = Game._nextIndex;
		}
		data._year = Game._year;
		data.rikishi = [];
		for(let r of Game.rikishi) {
			data.rikishi.push(r.flattened());
		}
		data.ROTRNGstate = JSON.stringify(Game._lastROTRNGstate);
		data._betweenBashos = Game._betweenBashos;
		data._setupBoutArgs = Game._setupBoutArgs.slice();
		for(let i=0; i<2; i++) {
			if (data._setupBoutArgs[i]) {
				data._setupBoutArgs[i] = data._setupBoutArgs[i].toString();
			}
		}
		if (Game._prevChampion) {
			data._prevChampion = Game._prevChampion.toString();
		}
		if (Game._prevChampionWasOzeki) {
			data._prevChampionWasOzeki = Game._prevChampionWasOzeki;
		}
		if (Game._champion) {
			data._champion = Game._champion.toString();
		}
		if (Game._championWasOzeki) {
			data._championWasOzeki = Game._championWasOzeki;
		}
		if (Game._joiners) {
			data._joiners = [];
			for(let r of Game._joiners) {
				data._joiners.push(r.toString());
			}
		}
		data._buttontext = document.getElementById("next").firstChild.data;
		return JSON.stringify(data);
	},
	_showLoadButton: function() {
		const savedata = JSON.parse(localStorage.sumoOyakata);
		const loadButton = document.getElementById("load");
		const saved = new Date(savedata._saved);
		loadButton.className = loadButton.className.replace(/hidden/,"visible");
		Game._replaceText("loaddate",Game._time_diff(saved));
		const nextUpdate = Game._next_time_diff(saved);
		if (nextUpdate) {
			window.setTimeout(Game._showLoadButton, nextUpdate*1000 );
		}
		return savedata; // To avoid a reparse in checkLoadButton
	},
	_checkLoadButton: function() {
		if (localStorage.sumoOyakata) {
			const savedata = Game._showLoadButton();
			if (savedata && savedata._version >= 4) {
				document.getElementById("load").onclick = function() {
					ROT.RNG.setState(JSON.parse(savedata.ROTRNGstate));
					Game.rikishi = [];
					let rikishiTable = {};
					Game._year = savedata._year;
					Game._season = savedata._season;
					for(let r of savedata.rikishi) {
						const rik = new Rikishi(r);
						rikishiTable[r._name] = rik;
						Game.rikishi.push(rik);
					}
					// Must wait until all rikishi created before recording who has played who
					for(let r of savedata.rikishi) {
						const rik = rikishiTable[r._name];
						for(let opp of r._played) {
							rik.played(rikishiTable[opp]);
						}
					}
					if (savedata._winners) {
						Game._winners = [];
						for (let winner of savedata._winners) {
							Game._winners.push(rikishiTable[winner]);
						}
					} else {
						Game._winners = null;
					}
					if (savedata._prevWinner) {
						Game._prevWinner = rikishiTable[savedata._prevWinner];
					} else {
						Game._prevWinner = null;
					}
					if (savedata._nextRound) {
						Game._nextRound = [];
						for (let rounder of savedata._nextRound) {
							Game._nextRound.push(rikishiTable[rounder]);
						}
					}
					if (savedata._nextIndex) {
						Game._nextIndex = savedata.nextIndex;
					}
					if (savedata._champion) {
						Game._champion = rikishiTable[savedata._champion];
					} else {
						Game._champion = null;
					}
					if (savedata._prevChampion) {
						Game._prevChampion = rikishiTable[savedata._prevChampion];
					} else {
						Game._prevChampion = null;
					}
					if (savedata._joiners) {
						Game._joiners = [];
						for (let r of savedata._joiners) {
							Game._joiners.push(rikishiTable[r]);
						}
					} else {
						Game._joiners = null;
					}
					Game._championWasOzeki = savedata._championWasOzeki || false;
					Game._prevChampionWasOzeki = savedata.prevChampionWasOzeki || false;
					Game._todaysBouts = [];
					for(let bout of savedata._todaysBouts) {
						Game._todaysBouts.push([rikishiTable[bout[0]],rikishiTable[bout[1]]]);
					}
					Game._firstYear = savedata._firstYear;
					Game._day = savedata._day;
					Game._bout = savedata._bout;
					Game._replaceText("season",Game._SEASONS[Game._season]);
					Game._replaceText("year",Game._year);
					Game._replaceText("day",Game._day+1);
					Game._betweenBashos = savedata._betweenBashos || false;
					Game._updateTable(Game._betweenBashos ? Game._champion : null);
					Game._updateInfo();
					if (savedata._version === 1) {
						Game._setupBoutArgs = [undefined,undefined,undefined];
					} else {
						Game._setupBoutArgs = savedata._setupBoutArgs;
						if (Game._setupBoutArgs[0]) {
							Game._setupBoutArgs[0] = rikishiTable[Game._setupBoutArgs[0]];
						}
						if (Game._setupBoutArgs[1]) {
							Game._setupBoutArgs[1] = rikishiTable[Game._setupBoutArgs[1]];
						}
					}
					savedata._buttontext = savedata._buttontext || "Next";
					Game._replaceText("next",savedata._buttontext);
					document.getElementById("next").onclick = function() {
						if (!Game._betweenBashos) {
							Game._setupBout(...Game._setupBoutArgs);
						} else {
							Game._updateInfo();
							Game._prepareBasho();
						}
					};
				};
			} else {
				// Not a good version
				Game._replaceText("loaddate","an earlier version is not possible");
				document.getElementById("load").disabled = true;
			}
		}
	},
	_time_diff: function(from) {
		const diff = Math.floor(Math.abs(new Date().getTime() - from.getTime())/1000); // In seconds
		if (diff < 60) return "just now";
		if (diff < 120) return "a minute ago";
		if (diff < 3600) return Math.floor(diff/60)+" minutes ago";
		if (diff < 7200) return "an hour ago";
		if (diff < 86400) return Math.floor(diff/3600)+" hours ago";
		if (diff < 86400*2) return "yesterday";
		if (diff < 86400*6) return Intl.DateTimeFormat(undefined,{weekday:"long"}).format(from);
		if (new Date().getFullYear() === from.getFullYear()) return Intl.DateTimeFormat(undefined,{day:"numeric",month:"short"}).format(from);
		return Intl.DateTimeFormat(undefined,{day:"numeric",month:"short", year:"numeric"}).format(from);
	},
	_next_time_diff: function(from) {
		const diff = Math.floor(Math.abs(new Date().getTime() - from.getTime())/1000); // In seconds
		if (diff < 3660) return 60;
		if (diff < 86400) return 3600;
		return null;
	}

};

window.onload = function() { Game.init(); };