//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals TrainFactory, BellCode, Subclass, Invocation */
/* exported Timetables */

function Trains(controller,model,kimbridgeBox,eastleighBox,redbridgeBox) {
    var t;
    t = new TrainFactory(controller,model,"Salisbury to Southampton Train","Salisbury","Southampton");
    t.setActions([t.updateLocation("Kimbridge"),
                  t.startTrain(kimbridgeBox,BellCode.OfferPassenger),
                  t.updateLocation("BetweenKimbridgeAndRomsey"), t.expected(),
                  t.updateLocation("A"),  t.traverse(),
                  t.waitAtSignal(20),     t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("B"),  t.traverse(),
                  t.checkPoints(14,false),
                  t.updateLocation("P1"), t.atPlatform(120),
                  t.waitAtSignal(16),     t.pause(5),
                  t.updateLocation("C"),
                  t.checkPoints(12,false),
                  t.checkPoints(10,false),
                  t.checkPoints(8,true),
                  t.traverse(),
                  t.updateLocation("J"),  t.leftViewOf(kimbridgeBox), t.traverse(),
                  t.updateLocation("D"),
                  t.waitAtSignal(13),     t.traverse(),
                  t.updateLocation("BetweenRomseyAndRedbridge"), t.expected(), t.arrived(redbridgeBox),
                  t.updateLocation("Redbridge"),
                  t.stopTrain(redbridgeBox),
                  t.endTrain()
                ]);
    this.SAL_TO_SOU = t;
    t = new TrainFactory(controller,model,"Southampton to Salisbury Train","Southampton","Salisbury");
    t.setActions([t.updateLocation("Redbridge"),
                  t.startTrain(redbridgeBox,BellCode.OfferPassenger), 
                  t.updateLocation("BetweenRedbridgeAndRomsey"), t.expected(),
                  t.updateLocation("BeforeSignal5"), t.traverse(),
                  t.waitAtSignal(5),      t.pause(5),
                  t.updateLocation("H"),  t.traverse(),
                  t.checkPoints(7,false),
                  t.updateLocation("K"),  t.leftViewOf(redbridgeBox), t.traverse(),
                  t.updateLocation("L"),  t.traverse(),
                  t.updateLocation("P2"), t.atPlatform(120),
                  t.waitAtSignal(3),      t.pause(10),
                  t.checkPoints(15,false),
                  t.updateLocation("BetweenRomseyAndKimbridge"), t.expected(), t.arrived(kimbridgeBox),
                  t.updateLocation("Kimbridge"),
                  t.stopTrain(kimbridgeBox),
                  t.endTrain()
                ]);
    this.SOU_TO_SAL = t;
    t = new TrainFactory(controller,model,"Salisbury to Eastleigh","Salisbury","Eastleigh");
    t.setActions([t.updateLocation("Kimbridge"),
                  t.startTrain(kimbridgeBox,BellCode.OfferPassenger), 
                  t.updateLocation("BetweenKimbridgeAndRomsey"), t.expected(),
                  t.updateLocation("A"),  t.traverse(),
                  t.waitAtSignal(20),     t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("B"),  t.traverse(),
                  t.checkPoints(14,false),
                  t.updateLocation("P1"), t.atPlatform(120),
                  t.waitAtSignal(18),     t.pause(5),
                  t.updateLocation("C"),  
                  t.checkPoints(12,false),
                  t.checkPoints(10,false),
                  t.checkPoints(8,false),
                  t.leftViewOf(kimbridgeBox), t.traverse(),
                  t.updateLocation("UZ"), t.traverse(),
                  t.waitAtSignal(17),     t.pause(5),
                  t.updateLocation("PG"), t.traverse(),
                  t.checkPoints(11,true),
                  t.updateLocation("PF"), t.traverse(),
                  t.updateLocation("PE"), t.traverse(),
                  t.updateLocation("PD"), t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,false), t.traverse(0.5),
                  t.updateLocation("PC"), t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,true), t.traverse(0.5),
                  t.updateLocation("Eastleigh"), t.expected(), t.arrived(eastleighBox),
                  t.setEastleighSignals(false,false),
                  t.stopTrain(eastleighBox),
                  t.endTrain()
                 ]);
    this.SAL_TO_ESL = t;
    t = new TrainFactory(controller,model,"Eastleigh to Salisbury","Eastleigh","Salisbury");
    t.setActions([t.updateLocation("Eastleigh"),
                  t.setEastleighPoint(true),
                  t.startTrain(eastleighBox,BellCode.OfferPassenger), 
                  t.updateLocation("BetweenEastleighAndPC"), t.expected(),
                  t.updateLocation("PC"),     t.traverse(0.5,0.5),
                  t.setEastleighPoint(false), t.traverse(0.5),
                  t.updateLocation("PD"),     t.traverse(),
                  t.updateLocation("PE"),     t.traverse(),
                  t.updateLocation("PF"),     t.traverse(),
                  t.waitAtSignal(1),          t.pause(5),
                  t.checkPoints(11,false),
                  t.updateLocation("PH"),     t.traverse(),
                  t.waitAtSignal(2),          t.pause(5),
                  t.updateLocation("J"),      t.traverse(),
                  t.checkPoints(7,true),
                  t.updateLocation("K"),      t.leftViewOf(eastleighBox), t.traverse(),
                  t.checkPoints(10,false),
                  t.updateLocation("L"),      t.traverse(),
                  t.updateLocation("P2"),     t.atPlatform(120),
                  t.waitAtSignal(3),          t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("BetweenRomseyAndKimbridge"), t.expected(), t.arrived(kimbridgeBox),
                  t.updateLocation("Kimbridge"),
                  t.stopTrain(kimbridgeBox),
                  t.endTrain()
                ]);
    this.ESL_TO_SAL = t;
    t = new TrainFactory(controller,model,"Salisbury to Eastleigh Freight","Salisbury","Eastleigh");
    t.setActions([t.updateLocation("Kimbridge"),
                  t.startTrain(kimbridgeBox,BellCode.OfferExpressFreight), 
                  t.updateLocation("BetweenKimbridgeAndRomsey"), t.expected(),
                  t.updateLocation("A"),  t.traverse(),
                  t.waitAtSignal(20),     t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("B"),  t.traverse(),
                  t.checkPoints(14,false),
                  t.updateLocation("P1"), t.traverse(),
                  t.waitAtSignal(18),     t.pause(5),
                  t.updateLocation("C"),  
                  t.checkPoints(12,false),
                  t.checkPoints(10,false),
                  t.checkPoints(8,false),
                  t.leftViewOf(kimbridgeBox), t.traverse(),
                  t.updateLocation("UZ"), t.traverse(),
                  t.waitAtSignal(17),     t.pause(5),
                  t.updateLocation("PG"), t.traverse(),
                  t.checkPoints(11,true),
                  t.updateLocation("PF"), t.traverse(),
                  t.updateLocation("PE"), t.traverse(),
                  t.updateLocation("PD"), t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,false), t.traverse(0.5),
                  t.updateLocation("PC"), t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,true), t.traverse(0.5),
                  t.updateLocation("Eastleigh"), t.expected(), t.arrived(eastleighBox),
                  t.setEastleighSignals(false,false),
                  t.stopTrain(eastleighBox),
                  t.endTrain()
                 ]);
    this.SAL_TO_ESL_freight = t;
    t = new TrainFactory(controller,model,"Eastleigh to Salisbury Freight","Eastleigh","Salisbury");
    t.setActions([t.updateLocation("Eastleigh"),
                  t.setEastleighPoint(true),
                  t.startTrain(eastleighBox,BellCode.OfferExpressFreight), 
                  t.updateLocation("BetweenEastleighAndPC"), t.expected(),
                  t.updateLocation("PC"),     t.traverse(0.5,0.5),
                  t.setEastleighPoint(false), t.traverse(0.5),
                  t.updateLocation("PD"),     t.traverse(),
                  t.updateLocation("PE"),     t.traverse(),
                  t.updateLocation("PF"),     t.traverse(),
                  t.waitAtSignal(1),          t.pause(5),
                  t.checkPoints(11,false),
                  t.updateLocation("PH"),     t.traverse(),
                  t.waitAtSignal(2),          t.pause(5),
                  t.updateLocation("J"),      t.traverse(),
                  t.checkPoints(7,true),
                  t.updateLocation("K"),      t.leftViewOf(eastleighBox), t.traverse(),
                  t.checkPoints(10,false),
                  t.updateLocation("L"),      t.traverse(),
                  t.updateLocation("P2"),     t.traverse(),
                  t.waitAtSignal(3),          t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("BetweenRomseyAndKimbridge"), t.expected(), t.arrived(kimbridgeBox),
                  t.updateLocation("Kimbridge"),
                  t.stopTrain(kimbridgeBox),
                  t.endTrain()
                ]);
    this.ESL_TO_SAL_freight = t;
    t = new TrainFactory(controller,model,"Eastleigh to Eastleigh Passenger","Eastleigh","Romsey");
    t.setActions([t.updateLocation("Eastleigh"),
                  t.setEastleighPoint(true), 
                  t.startTrain(eastleighBox,BellCode.OfferPassenger), 
                  t.updateLocation("BetweenEastleighAndPC"), t.expected(),
                  t.updateLocation("PC"),            t.traverse(0.5,0.5),
                  t.setEastleighPoint(false),        t.pause(0.5),
                  t.updateLocation("PD"),            t.traverse(),
                  t.updateLocation("PE"),            t.traverse(),
                  t.updateLocation("PF"),            t.traverse(),
                  t.waitAtSignal(1),                 t.pause(5),
                  t.checkPoints(11,false),
                  t.updateLocation("PH"),            t.traverse(),
                  t.waitAtSignal(2),                 t.pause(5),
                  t.updateLocation("J"),             t.traverse(),
                  t.checkPoints(7,true),
                  t.updateLocation("K"),             t.leftViewOf(eastleighBox), t.traverse(),
                  t.checkPoints(10,false),
                  t.updateLocation("L"),             t.traverse(),
                  t.updateLocation("P2"),            t.atPlatform(150),
                  t.change("Eastleigh"),             t.atPlatform(150),
                  t.updateLocation("L"),             t.traverse(),
                  t.waitAtSignal(22),                t.pause(5),
                  t.checkPoints(10,true),
                  t.checkPoints(8,false),
                  t.updateLocation("C"),             t.traverse(),
                  t.updateLocation("UZ"),            t.traverse(),
                  t.waitAtSignal(17),                t.pause(5),
                  t.updateLocation("PG"),            t.traverse(),
                  t.checkPoints(11,true),
                  t.updateLocation("PF"),            t.traverse(),
                  t.updateLocation("PE"),            t.traverse(),
                  t.updateLocation("PD"),            t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,false), t.traverse(0.5),
                  t.updateLocation("PC"),            t.traverse(0.5,0.5),
                  t.setEastleighSignals(true,true),  t.traverse(0.5),
                  t.updateLocation("Eastleigh"),     t.expected(), t.arrived(eastleighBox),
                  t.setEastleighSignals(false,false),t.traverse(),
                  t.stopTrain(eastleighBox),
                  t.endTrain()
                ]);
    this.ESL_TO_ESL = t;
    t = new TrainFactory(controller,model,"Salisbury to Southampton Freight","5Y14","Salisbury","Southampton");
    t.setActions([t.updateLocation("Kimbridge"),
                  t.startTrain(kimbridgeBox,BellCode.OfferExpressFreight), 
                  t.updateLocation("BetweenKimbridgeAndRomsey"), t.expected(),
                  t.updateLocation("A"),  t.traverse(),
                  t.waitAtSignal(20),     t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("B"),  t.traverse(),
                  t.checkPoints(14,false),
                  t.updateLocation("P1"), t.traverse(),
                  t.waitAtSignal(16),     t.pause(5),
                  t.updateLocation("C"),
                  t.checkPoints(10,false),
                  t.checkPoints(8,true),
                  t.traverse(),
                  t.updateLocation("J"),  t.leftViewOf(kimbridgeBox), t.traverse(),
                  t.updateLocation("D"),
                  t.waitAtSignal(13),     t.traverse(),
                  t.updateLocation("BetweenRomseyAndRedbridge"), t.expected(), t.arrived(redbridgeBox),
                  t.updateLocation("Redbridge"),
                  t.stopTrain(redbridgeBox),
                  t.endTrain()
                ]);
    this.SAL_TO_SOU_freight = t;
    t = new TrainFactory(controller,model,"Southampton to Salisbury Freight","Southampton","Salisbury");
    t.setActions([t.updateLocation("Redbridge"),
                  t.startTrain(redbridgeBox,BellCode.OfferExpressFreight),
                  t.updateLocation("BetweenRedbridgeAndRomsey"), t.expected(),
                  t.updateLocation("BeforeSignal5"), t.traverse(),
                  t.waitAtSignal(5),      t.pause(5),
                  t.updateLocation("H"),  t.traverse(),
                  t.checkPoints(7,false),
                  t.updateLocation("K"),  t.leftViewOf(redbridgeBox), t.traverse(),
                  t.checkPoints(10,false),
                  t.updateLocation("L"),  t.traverse(),
                  t.updateLocation("P2"), t.traverse(),
                  t.waitAtSignal(3),      t.pause(10),
                  t.checkPoints(15,false),
                  t.updateLocation("BetweenRomseyAndKimbridge"), t.expected(), t.arrived(kimbridgeBox),
                  t.updateLocation("Kimbridge"),
                  t.stopTrain(kimbridgeBox),
                  t.endTrain()
                ]);
    this.SOU_TO_SAL_freight = t;
    t = new TrainFactory(controller,model,"Salisbury to Salisbury Passenger","Salisbury","Romsey");
    t.setActions([t.updateLocation("Kimbridge"),
                  t.startTrain(kimbridgeBox,BellCode.OfferPassenger),
                  t.updateLocation("BetweenKimbridgeAndRomsey"), t.expected(),
                  t.updateLocation("A"),  t.traverse(),
                  t.waitAtSignal(20),     t.pause(5),
                  t.checkPoints(15,false),
                  t.updateLocation("B"),  t.traverse(),
                  t.checkPoints(14,false),
                  t.updateLocation("P1"), t.atPlatform(540),
                  t.change("Signal 1"),
                  t.waitAtSignal(18),     t.pause(5),
                  t.updateLocation("C"),  
                  t.checkPoints(12,false),
                  t.checkPoints(10,false),
                  t.checkPoints(8,false),
                  t.leftViewOf(kimbridgeBox), t.traverse(),
                  t.updateLocation("UZ"), t.traverse(),
                  t.waitAtSignal(17),     t.pause(5),
                  t.updateLocation("PG"), t.traverse(),
                  t.checkPoints(11,true),
                  t.updateLocation("PF"), t.traverse(),
                  t.change("Salisbury"), t.pause(900),
                  t.waitAtSignal(1),          t.pause(5),
                  t.checkPoints(11,false),
                  t.updateLocation("PH"),     t.traverse(),
                  t.waitAtSignal(2),          t.pause(5),
                  t.updateLocation("J"),      t.traverse(),
                  t.checkPoints(7,true),
                  t.updateLocation("K"),      t.leftViewOf(eastleighBox), t.traverse(),
                  t.checkPoints(10,false),
                  t.updateLocation("L"),      t.traverse(),
                  t.updateLocation("P2"), t.atPlatform(120),
                  t.waitAtSignal(3),      t.pause(10),
                  t.checkPoints(15,false),
                  t.updateLocation("BetweenRomseyAndKimbridge"), t.expected(), t.arrived(kimbridgeBox),
                  t.updateLocation("Kimbridge"),
                  t.stopTrain(kimbridgeBox),
                  t.endTrain()
                ]);
    this.SAL_TO_SAL = t;
}

function TrainRecord(offset,description,factory,codes,view) {
    this._offset = offset;
    this._description = description;
    this._factory = factory;
    this._rows = [];
    this._view = view;
    this._rowIndex = 0;
    this._location = "";
    this._status = "";
    this._train = null;
    this._baseCode = [].concat(codes);
    this._codeIndex = 1;
}
TrainRecord.prototype = {
    getOffset: function() { return this._offset; },
    _listener: function() {
        const self = this; // Capture this
        return { start: function() { self.start(); },
                 updateLocation: function(location) { self.updateLocation(location); },
                 change: function() { self.change(); },
                 updateStatus: function(status) { self.updateStatus(status); },
                 completed: function() { self.completed("Left the simulation"); }
               };
    },
    prepare: function() {
        const codes = [];
        for(let i=0; i<this._factory.getDestinationsCount(); i++) {
            codes.push(this._baseCode[i].slice(0,2) + this._codeIndex.toString() + this._baseCode[i].slice(3));
        }
        this._codeIndex++; if (this._codeIndex === 10) { this._codeIndex = 0; }
        this._train = this._factory.make(codes);
        for(let i=0, o=this._factory.getOrigin(), d=this._factory.getDestination(i); i<this._factory.getDestinationsCount();i++, o=d, d=this._factory.getDestination(i)) {
            this._rows.push(this._view.addTrainToTimetable(codes[i],
                                                           o,
                                                           d,
                                                           (i===0 ? this._description : "Starts here (using train "+codes[0]+")")
                                                          ));
        }
        this._train.addListener(this._listener());
        return this._train;
    },
    start: function() {
        this._view.showTrain(this._rows[this._rowIndex]);
    },
    updateLocation: function(location) {
        this._location = location;
        this._showLocation();
    },
    _showLocation: function() {
        this._view.updateLocation(this._rows[this._rowIndex],this._location);
    },
    change: function() {
        this.completed("Terminated here");
        this._rowIndex++;
        this.start();
        this._train.setCurrentTrainCode(this._train.getTrainCode(this._rowIndex));
        this._showLocation();
        this._showStatus();
    },
    updateStatus: function(status) {
        this._status = status;
        this._showStatus();
    },
    _showStatus: function() {
        this._view.updateStatus(this._rows[this._rowIndex],this._status);
    },
    completed: function(reason) {
        this._view.completeTrain(this._rows[this._rowIndex],reason);
    }
};

function Timetable(scheduler,name,view) {
    this._scheduler = scheduler;
    this._name = name;
    this._view = view;
}
Timetable.prototype = {
    getName: function() { return this._name; },
    getView: function() { return this._view; },
    run: function(_clock) {
        throw new Error("run not overriden on "+this.constructor.name);
    }
};

function SingleTrainTimetable(scheduler,factory,code,view) {
    Timetable.call(this,scheduler,factory.getName(),view);
    this._factory = factory;
    this._code = code;
}
Subclass.make(SingleTrainTimetable,Timetable);
SingleTrainTimetable.prototype.run = function(_clock) {
    this.getView().showTimetable();
    const record = new TrainRecord(0,this._factory.getName(),this._factory,this._code,this.getView());
    const train = record.prepare();
    this._scheduler.addRealTimeEvent(1,new Invocation(train.doNextStep,train));
};

function MultipleTrainTimetable(scheduler,name,view,starthh,startmm,startss) {
    Timetable.call(this,scheduler,name,view);
    this._starthh = starthh;
    this._startmm = startmm;
    this._startss = startss;
    this._trains = [];
}
Subclass.make(MultipleTrainTimetable,Timetable);
MultipleTrainTimetable.prototype.addTrain = function(offset,description,factory,code) {
    this._trains.push(new TrainRecord(offset,description,factory,code,this._view));
};
MultipleTrainTimetable.prototype.run = function(clock) {
    this.getView().showTimetable();
    clock.setClock(this._starthh,this._startmm,this._startss);
    for(var i=0; i<this._trains.length; i++) {
        var train = this._trains[i].prepare();
        this._scheduler.addGameTimeEvent(this._trains[i].getOffset(),new Invocation(train.doNextStep,train));
    }
};

function Timetables(controller,model,view,kimbridgeBox,eastleighBox,redbridgeBox) {
    const trains = new Trains(controller,model,kimbridgeBox,eastleighBox,redbridgeBox);
    const scheduler = controller.getScheduler();
    const salToSou = new SingleTrainTimetable(scheduler,trains.SAL_TO_SOU,"1F11",view);
    const souToSal = new SingleTrainTimetable(scheduler,trains.SOU_TO_SAL,"1F16",view);
    const salToEsl = new SingleTrainTimetable(scheduler,trains.SAL_TO_ESL_freight,"6X41",view);
    const eslToSal = new SingleTrainTimetable(scheduler,trains.ESL_TO_SAL_freight,"6F11",view);
    const eslToEsl = new SingleTrainTimetable(scheduler,trains.ESL_TO_ESL,["2S32","2S35"],view);
    
    const jan242019 = new MultipleTrainTimetable(scheduler,"11:55-13:00 January 24 2019",view,11,55,0);
    jan242019.addTrain(1,"GWR to Great Malvern due at 1201",trains.SAL_TO_SOU,"1F11");
    jan242019.addTrain(50,"SWR circular from Eastleigh due at 1207",trains.ESL_TO_ESL,["2S32","2S35"]);
    jan242019.addTrain(725,"Freight to Southampton Docks due at 1210",trains.SAL_TO_SOU_freight,"742A");
    jan242019.addTrain(1185,"GWR to Cardiff Central due at 1220",trains.SOU_TO_SAL,"1F16");
    jan242019.addTrain(1605,"SWR circular from Salisbury due at 1227",trains.SAL_TO_SOU,"2S31");
    jan242019.addTrain(1860,"Freight to Whatley Quarry (via Salisbury) due at 1233",trains.ESL_TO_SAL_freight,"6F11");
    jan242019.addTrain(2265,"GWR to Southampton due at 1238",trains.SOU_TO_SAL,"1F13");
    jan242019.addTrain(3030,"SWR circular to Salisbury due at 1250",trains.SOU_TO_SAL,"2S35");
    jan242019.addTrain(3045,"GWR to Portsmouth Harbour due at 1250",trains.SAL_TO_SOU,"1F17");
    
    const apr092020 = new MultipleTrainTimetable(scheduler,"11:50-12:30 April 09 2020",view,11,50,0);
    apr092020.addTrain(1,"GWR to Portsmouth Harbour due at 1155",trains.SAL_TO_SOU,"1F11");
    apr092020.addTrain(325,"SWR circular from Eastleigh due at 1207",trains.ESL_TO_ESL,["2S31","2S35"]);
    apr092020.addTrain(1080,"Freight to Southampton Docks due at 1215",trains.SAL_TO_SOU_freight,"742A");
    apr092020.addTrain(1740,"GWR to Cardiff Central due at 1223",trains.SOU_TO_SAL,"1F16");
    
    const feb192021 = new MultipleTrainTimetable(scheduler,"08:07-09:00 February 19 2021",view,8,07,0);
    feb192021.addTrain(1,"GWR Portsmouth to Cardiff (via Eastleigh) due at 08:19",trains.ESL_TO_SAL,"1F08");
    feb192021.addTrain(300,"SWR Salisbury to Salisbury due at 08:27 (then at 08:49)",trains.SAL_TO_SAL,["2R26","5S15","2S15"]);
    feb192021.addTrain(720,"Freight to Liverpool (via Eastleigh) due at 08:55",trains.ESL_TO_SAL_freight,"061A");
    feb192021.addTrain(840,"Engineering Train from Southampton to Westbury due at 08:30",trains.SOU_TO_SAL_freight,"406U");
    feb192021.addTrain(1620,"Freight to Southampton Docks (via Eastleigh) due at 08:36",trains.SAL_TO_ESL_freight,"019A");
    feb192021.addTrain(2460,"GWR Cardiff to Portsmouth (via Eastleigh) due at 08:51",trains.SAL_TO_ESL,"1F05");
    
    this.list = [souToSal,salToSou,salToEsl,eslToSal,eslToEsl,apr092020,jan242019,feb192021];
}