//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals SignalBox, Subclass, Invocation */

// The basic TrainAction
function TrainAction(){}
TrainAction.prototype = {
    doAction: function(controller,train,invocation) {
        const scheduler = controller.getScheduler();
        const done = this.tryAction(controller,train,invocation);
        
        if (done.retry) {
            var repeatInvocation = new Invocation(this.doAction,this,controller,train,invocation);
            
            if (done.realTime !== undefined) {
                scheduler.addRealTimeEvent(done.realTime,repeatInvocation);
            } else {
                scheduler.addGameTimeEvent(done.gameTime,repeatInvocation);
            }
        } else {
            if (done.scheduleDone) {
                if (!!!done.realTime) { done.realTime = 0; }
                if (!!!done.gameTime) { done.gameTime = 0; }
                if (done.realTime === 0 && done.gameTime === 0) {
                    // We don't call it immediately as train doesn't expect the call back on this call stack :)
                    scheduler.addRealTimeEvent(0,invocation);
                } else {
                    if (done.gameTime === 0) {
                        scheduler.addRealTimeEvent(done.realTime,invocation);
                    } else {
                        scheduler.addGameTimeEvent(done.gameTime,invocation);
                    }
                }
                if (done.alsoInvoke) {
                    scheduler.addGameTimeEvent(done.gameTime+done.whenToAlsoInvoke,done.alsoInvoke);
                }
            }
        }
    },
    tryAction: function(_controller,_train,_doneInvocation) {
        throw new TypeError("TrainAction.tryAction not overriden on "+this.constructor.name);
    },
    getStatus: function() {
        return "";
    }
};

// Start a new train
function StartTrainAction(startLocation,bellcode) {
    TrainAction.call(this);
    this._startLocation = startLocation;
    this._bellcode = bellcode;
}
Subclass.make(StartTrainAction,TrainAction);
StartTrainAction.prototype.tryAction = function(_controller,train,doneInvocation) {
    if (this._startLocation === "Romsey") {
        // Easy, it's here! - but starting at a speed of zero :)
        train.setSpeed(0);
        return { retry: false, scheduleDone: true };
    }
    if (this._startLocation instanceof SignalBox) {
        var accepted = this._startLocation.offerTrain(train.getCurrentTrainCode(),this._bellcode,doneInvocation);
        if (accepted) {
            train.setSpeed(train.getMaxSpeed());
            return { retry: false, scheduleDone: false };
        } else {
            return { retry: true, realTime: 1 };
        }
    }
    throw new Error("_startLocation of "+this._startLocation+" not recognised when attempting to start a train");
};
StartTrainAction.prototype.getStatus = function() {
    if (this._startLocation instanceof SignalBox) {
        return "Waiting to be signalled to Romsey";
    }
};

// Pause for a fixed period of time
function PauseTrainAction(duration) {
    TrainAction.call(this);
    this._duration = duration;
}
Subclass.make(PauseTrainAction,TrainAction);
PauseTrainAction.prototype.tryAction = function(_controller,_train,_doneInvocation) {
    return { retry:false, scheduleDone: true, gameTime: this._duration };
};

//Traverse the current location
function TraverseLocationTrainAction(location,fraction,remaining) {
    TrainAction.call(this);
    this._fraction = (fraction ? fraction : 1);
    this._remaining = (remaining ? remaining : 0);
    this._location = location;
}
Subclass.make(TraverseLocationTrainAction,TrainAction);
TraverseLocationTrainAction.prototype.tryAction = function(controller,train,_doneInvocation) {
    // Should the train use max speed instead of current?
    const length = this._location.getLength()*this._fraction;
    const remainingLength = this._location.getLength()*this._remaining;
    if (length >= 1760) {
        // More than a mile in length - we'll accelerate to top speed
        train.setSpeed(train.getMaxSpeed());
    }
    const time = Math.ceil(length*25/(11*train.getSpeed()));
    const totalTime = Math.ceil((length+remainingLength)*25/(11*train.getSpeed()));
    this._at = controller.getScheduler().getClock().getClockAsStringAfterXMilliseconds(totalTime*1000); // total used for the status text
    return { retry:false, scheduleDone: true, gameTime: time }; // but just the time for the time to the next event
};
TraverseLocationTrainAction.prototype.getStatus = function() {
    return "Expected to leave section at "+this._at;
};

// Traverse the current location and give a status saying when the train is expected to leave the location
function ExpectedTrainAction(location,fraction) {
    TraverseLocationTrainAction.call(this,location,fraction);
}
Subclass.make(ExpectedTrainAction,TraverseLocationTrainAction);
ExpectedTrainAction.prototype.getStatus = function() {
    return "Expected "+this._at;
};


// Train arrived at a remote signal box
function ArrivedTrainAction(box) {
    this._box = box;
}
Subclass.make(ArrivedTrainAction,TrainAction);
ArrivedTrainAction.prototype.tryAction = function(_controller,train,_doneInvocation) {
    var result = {retry: false, scheduleDone: true };
    result.alsoInvoke = new Invocation(this._box.trainJustArrived,this._box,train.getCurrentTrainCode());
    result.whenToAlsoInvoke = 5;
    return result;
};

// Train left a section monitored by a remote signal box
function LeftViewOfTrainAction(box) {
    this._box = box;
}
Subclass.make(LeftViewOfTrainAction,TrainAction);
LeftViewOfTrainAction.prototype.tryAction = function(_controller,train,_doneInvocation) {
    var result = { retry: false, scheduleDone: true };
    result.alsoInvoke = new Invocation(this._box.trainDepartedSection,this._box,train.getCurrentTrainCode());
    result.whenToAlsoInvoke = 60; // Give a minute to send the out of section
    return result;
};

// Pause at a platform (for passengers to leave/board) - variant of pause
function AtPlatformTrainAction(duration) {
    PauseTrainAction.call(this,duration);
}
Subclass.make(AtPlatformTrainAction,PauseTrainAction);
AtPlatformTrainAction.prototype.tryAction = function(controller,train,doneInvocation) {
    this._at = controller.getScheduler().getClock().getClockAsStringAfterXMilliseconds(this._duration*1000);
    return PauseTrainAction.prototype.tryAction.apply(this,controller,train,doneInvocation);
};
AtPlatformTrainAction.prototype.getStatus = function() {
    return "Stopped at platform until "+this._at;
};

// Update the location of a train
function UpdateLocationTrainAction(oldPosition,position) {
    TrainAction.call(this);
    this._oldPosition = oldPosition;
    this._position = position;
    this._posName = (position ? position.getDescriptionWithPreposition() : "");
    this._complained = false;
}
Subclass.make(UpdateLocationTrainAction,TrainAction);
UpdateLocationTrainAction.prototype.tryAction = function(controller,train,_doneInvocation) {
    if (this._position) {
        if (this._position.allowsMultipleTrains() || this._position.getOccupier() === null) {
            var result = { retry: false, scheduleDone: true, realTime: 0 };
            if (this._oldPosition) {
                // Should the train accelerate
                if (train.getSpeed() < train.getMaxSpeed()) {
                    var newSpeed = train.getSpeed()+10;
                    if (newSpeed > train.getMaxSpeed()) { newSpeed = train.getMaxSpeed(); }
                    train.setSpeed(newSpeed);
                }
                result.alsoInvoke = new Invocation(this._oldPosition.occupy,this._oldPosition,null);
                result.whenToAlsoInvoke = this._oldPosition.getOverlap()*25/(11*train.getSpeed()); // time to clear the overlap
            }
            this._position.occupy(train.getCurrentTrainCode());
            return result;
        } else {
            train.setSpeed(0); // Grind to a halt
            if (!!!this._complained) {
                this._complained = true;
                controller.getMessageBox().showMessage("Driver of train "+train.getCurrentTrainCode(),"This is the driver of train "+
                        train.getCurrentTrainCode()+". I should be entering track section "+this._position.getDescription()+", but "+
                        "train "+this._position.getOccupier()+" is already there! This is a serious irregularity that I will need to "+
                        "report since this could have been a serious accident. I will wait here at "+
                        this._oldPosition.getDescriptionWithPreposition()+"until that location is clear.");
            }
            return { retry: true, scheduleDone: false, gameTime: 3};
        }
    } else {
        // The train is about to end
        if (this._oldPosition) {
            this._oldPosition.occupy(null);
        }
        return { retry: false, scheduleDone: true, realTime: 0 };
    }
    
};
UpdateLocationTrainAction.prototype.getLocationName = function() {
    return this._posName;
};

// Wait at signal
function WaitAtSignalTrainAction(signal) {
    TrainAction.call(this);
    this._signal = signal;
    this._lastChecked = null;
    this._delay = 60;
}
Subclass.make(WaitAtSignalTrainAction,TrainAction);
WaitAtSignalTrainAction.prototype.tryAction = function(controller,train,_doneInvocation) {
    if (this._signal.isPulled()) {
        return { retry: false, scheduleDone: true, realTime: 0};
    } else {
        if (this._lastChecked === null) {
            this._lastChecked = controller.getScheduler().getClock().getTicks();
        } else if (controller.getScheduler().getClock().getTicks() > this._lastChecked+this._delay) {
                this._lastChecked = controller.getScheduler().getClock().getTicks();
                this._delay *= 2;
                if (this._delay === 120) {
                    controller.getMessageBox().showMessage("Driver of train "+train.getCurrentTrainCode(),"This is the driver of train "+
                            train.getCurrentTrainCode()+". I'm waiting at signal "+this._signal.getName()+".");
                } else {
                    controller.getMessageBox().showMessage("Driver of train "+train.getCurrentTrainCode(),"This is the driver of train "+
                            train.getCurrentTrainCode()+". I'm still waiting at signal "+this._signal.getName()+".");
                }
        }
        train.setSpeed(0);
        return { retry: true, scheduleDone: false, gameTime: 3};
    }
};
WaitAtSignalTrainAction.prototype.getStatus = function() {
    return "Waiting for signal "+this._signal.getName()+" to be pulled";
};

// Set Eastleigh signals
function SetEastleighSignalsTrainAction(model,zw30,next) {
    TrainAction.call(this);
    this._model = model;
    this._zw30 = zw30;
    this._next = next;
}
Subclass.make(SetEastleighSignalsTrainAction,TrainAction);
SetEastleighSignalsTrainAction.prototype.tryAction = function(_controller,_train,_doneInvocation) {
    this._model.getEastleigh().setSignals(this._zw30,this._next);
    return { retry: false, scheduleDone: true, realTime: 0 };
};

//Set Eastleigh point
function SetEastleighPointTrainAction(model,state) {
    TrainAction.call(this);
    this._model = model;
    this._state = state;
}
Subclass.make(SetEastleighPointTrainAction,TrainAction);
SetEastleighPointTrainAction.prototype.tryAction = function(_controller,_train,_doneInvocation) {
    this._model.getEastleigh().setPoint(this._state);
    return { retry: false, scheduleDone: true, realTime: 0 };
};

// Stop Train
function StopTrainAction(location) {
    TrainAction.call(this);
    this._location = location;
}
Subclass.make(StopTrainAction,TrainAction);
StopTrainAction.prototype.tryAction = function(_controller,_train,doneInvocation) {
    if (this._location === "Romsey") {
        // Easy, it's here!
        return { retry: false, scheduleDone: true };
    }
    if (this._location instanceof SignalBox) {
        var accepted = this._location.exitTrain(doneInvocation);
        if (accepted) {
            return { retry: false, scheduleDone: false };
        } else {
            return { retry: true, realTime: 1 };
        }
    }
    throw new Error("_location of "+this._location+" not recognised when attempting to stop a train");

};

// Change a train code, origin and destination
function ChangeTrainAction() {
    TrainAction.call(this);
}
Subclass.make(ChangeTrainAction,TrainAction);
ChangeTrainAction.prototype.tryAction = function(_controller,_train,_doneInvocation) {
    // As an action this is basically a no-op
    return { retry: false, scheduleDone: true };
};

// Check points and complain if the points are mis set
function CheckPointsTrainAction(lever,desiredState) {
    TrainAction.call(this);
    this._lever = lever;
    this._desiredState = desiredState;
    this._lastChecked = null;
}
Subclass.make(CheckPointsTrainAction,TrainAction);
CheckPointsTrainAction.prototype.tryAction = function(controller,train,_doneInvocation) {
    if (this._lever.isPulled() !== this._desiredState) {
        train.setSpeed(0); // Grind to a halt!
        if ( (!!!this._lastChecked) || (controller.getScheduler().getClock().getTicks() > this._lastChecked+300) ) {
            this._lastChecked = controller.getScheduler().getClock().getTicks();
            controller.getMessageBox().showMessage("Driver of train "+train.getCurrentTrainCode(),"This is the driver of train "+
                    train.getCurrentTrainCode()+". "+this._lever.getDescription()+" in front of my train are not set correctly "+
                    "for my route. Please correct the setting so that I can proceed.");
        }
        return { retry: true, scheduleDone: false, gameTime: 3};
    } else {
        if (train.getSpeed() === 0) {
            train.setSpeed(10);
        }
        return { retry: false, scheduleDone: true };
    }
};