//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals BellCode, Invocation */
/* globals module, exports */
function BoxState(name,description) {
    this._name = name;
    this._description = description;
}
BoxState.prototype = {
    getTooltip: function() { return this._description; },
    toString: function() { return this._name; }
};

BoxState.allStates = [];
BoxState._addState = function(name,description) {
    BoxState[name] = new BoxState(name,description);
    BoxState.allStates.push(BoxState[name]);
};

BoxState._addState("NONE","Idle");
BoxState._addState("RECEIVING","Expecting a bell code");
BoxState._addState("AWAITING_TRAIN_ENTERING","Waiting for a 'Train Entering Section' bell code");
BoxState._addState("AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT","Waiting for either 'Train Entering Section' or 'Train Out Of Section' bell code");
BoxState._addState("AWAITING_ATTN_ACK","Waiting for the acknowledgement of an 'Attention' bell code (so they can offer a train)");
BoxState._addState("AWAITING_OFFER_ACK","Waiting for an acknowledgment of their offer of a train");
BoxState._addState("AWAITING_THEIR_LINE_CLEAR","Waiting for the line to Romsey to be LINE CLEAR");
BoxState._addState("PAUSE_TRAIN_START","Arranging for a train to signalled through to Romsey");
BoxState._addState("PAUSE_AFTER_EXIT_ACK","Pausing after acknowledging a 'Train Out Of Section' bell code");
BoxState._addState("EXPECTING_TRAIN","Expecting a train to arrive from Romsey");
BoxState._addState("AWAITING_ENTERING_ACK","Waiting for an acknowledgement of 'Train Entering Section' bell code");
BoxState._addState("AWAITING_THEIR_TRAIN_ON_LINE","Waiting for the line to Romsey to be TRAIN_ON_LINE");
BoxState._addState("AWAITING_EXIT_ATTN_ACK","Waiting for an acknowledgement of an ' Attention' bell code (so they can send 'Train Out Of Section')");
BoxState._addState("AWAITING_OUT_OF_SECTION_ACK","Waiting for an acknowledgement of their 'Train Out Of Section' bell code");
BoxState._addState("AWAITING_SHUNT_ATTN","Waiting for an attn that will signal end of 'Shunt' bell code");
BoxState._addState("AWAITING_END_SHUNT","Waiting for an 'End Shunt' bell code");

function SignalBox(controller, bell) {
    this._controller = controller;
    this._scheduler = controller.getScheduler();
    this._messagebox = controller.getMessageBox();
    this._name = bell.getName();
    this._bell = bell;
    this._state = BoxState.NONE;
    this._preOfferState = BoxState.NONE;
    this._bell.setTooltip(this._state.getTooltip());
    this._offeredTrains = []; // The list of trains offered to the next box, but not yet out of section
    this._trainsIntoOurSection = 0;
    this._trainsLeftOurSection = 0;
    this._idleStarted = this._scheduler.getClock().getTicks();
    
    this._changeAfterPauseInvocation = new Invocation(this.changeAfterPause,this);
    this._trainIntoSectionInvocation = new Invocation(this.trainIntoSection,this);
    this._sendOutOfSectionInvocation = new Invocation(this.sendOutOfOurSection,this);
    this._timeoutAttnWhenIdleInvocation             = new Invocation(this.timeoutAttnWhenIdle,this);
    this._timeoutOfferAckInvocation                 = new Invocation(this.timeoutOfferAck,this);
    this._timeoutAttnAckInvocation                  = new Invocation(this.timeoutAttnAck,this);
    this._timeoutEnteringAckInvocation              = new Invocation(this.timeoutEnteringAck,this);
    this._timeoutAwaitingTheirTrainOnLineInvocation = new Invocation(this.timeoutAwaitingTheirTrainOnLine,this);
    this._timeoutAwaitingOutOfSectionAckInvocation  = new Invocation(this.timeoutAwaitingOutOfSectionAck,this);
    this._changeStateToNONE                    = new Invocation(this.changeState,this,BoxState.NONE);
    this._changeStateToRECEIVING               = new Invocation(this.changeState,this,BoxState.RECEIVING);
    this._changeStateToAWAITING_TRAIN_ENTERING = new Invocation(this.changeState,this,BoxState.AWAITING_TRAIN_ENTERING);
    this._changeStateToAWAITING_SHUNT_ATTN     = new Invocation(this.changeState,this,BoxState.AWAITING_SHUNT_ATTN);

    this._receiveTimeout = null;
    this._offerAckTimeout = null;
    this._attnAckTimeout = null;
    this._attnExitAckTimeout = null;
    this._enteringAckTimeout = null;
    this._awaitingTheirTrainOnlineTimeout = null;
    this._awaitingOutOfSectionAckTimeout = null;
    this._startLineClearCheck = 0;
    
    const self = this;
    this._echoBellAndChangeState = function(state) {
        this.changeState(state);
        return this._bell.echoBellCode();
    };
    this._echoBellAndChangeStateFunction = function(state) {
        return function() {
            self._echoBellAndChangeState(state);
        };
    };
    
    // Create the action arrays
    this._action = [];
    for(let i=0; i<BoxState.allStates.length; i++) {
        this._action[BoxState.allStates[i]] = [];
    }
    
    // Standard actions
    this._action[BoxState.NONE][BellCode.Attn] = function() {
        self._receiveTimeout = self._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,self._timeoutAttnWhenIdleInvocation);
        self.changeState(BoxState.RECEIVING);
        self._bell.echoBellCode();
    };
    this._action[BoxState.PAUSE_AFTER_EXIT_ACK] = this._action[BoxState.NONE][BellCode.Attn];
    this._action[BoxState.RECEIVING][BellCode.OutOfSection] = function() {
        if (self._receiveTimeout) {
            self._receiveTimeout.cancel();
            self._receiveTimeout = null;
        }
        if (self._offeredTrains.length > 0) {
            const bellLength = self._echoBellAndChangeState(BoxState.PAUSE_AFTER_EXIT_ACK);
            self._trainOnTrack = false;
            self._offeredTrains.shift();
            self._scheduler.addGameTimeEvent(bellLength+5,self._changeAfterPauseInvocation);
        } else {
            self.changeState(BoxState.NONE);
        }
    };
    this._action[BoxState.RECEIVING][BellCode.Cancel] = function() {
        if (self._receiveTimeout) {
            self._receiveTimeout.cancel();
            self._receiveTimeout = null;
        }
        self._echoBellAndChangeState(BoxState.PAUSE_AFTER_EXIT_ACK);
    };
    this._action_correct_offer_ack = function() {
        delete self._action[BoxState.AWAITING_OFFER_ACK][self._trainToOffer];
        if (self._offerAckTimeout) {
            self._offerAckTimeout.cancel();
            self._offerAckTimeout = null;
        }
        self.changeState(BoxState.AWAITING_THEIR_LINE_CLEAR);
        self.scheduleCheckLineEmpty();
        self._startLineClearCheck = self._scheduler.getClock().getTicks();
    };
    
    this._action[BoxState.AWAITING_ENTERING_ACK][BellCode.EnteringSection] = function() {
        if (self._enteringAckTimeout) {
            self._enteringAckTimeout.cancel();
            self._enteringAckTimeout = null;
        }
        self.changeState(BoxState.AWAITING_THEIR_TRAIN_ON_LINE);
        self.scheduleCheckLineOccupied();
        self._awaitingTheirTrainOnlineTimeout = 
            self._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,self._timeoutAwaitingTheirTrainOnLineInvocation);
    };
    this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.EnteringSection] = function() {
        const bellLength = self._bell.echoBellCode();
        const newState = self.doEnterSectionReturningNewState(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._trainIntoSectionInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.02,new Invocation(self.changeState,self,newState));
    };
    this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.WronglyDescribed] = function() {
        const bellLength = self._bell.echoBellCode();
        self.cancelEnterSection(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToRECEIVING);
    };
    this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.Cancel] = function() {
        const bellLength = self._bell.echoBellCode();
        self.cancelEnterSection(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToNONE);
    };
    this._action[BoxState.AWAITING_OUT_OF_SECTION_ACK][BellCode.OutOfSection] = function() {
        if (self._awaitingOutOfSectionAckTimeout) {
            self._awaitingOutOfSectionAckTimeout.cancel();
            self._awaitingOutOfSectionAckTimeout = null;
        }
        self.doTrainOutOfSection();
        self.changeState(BoxState.PAUSE_AFTER_EXIT_ACK);
        self._scheduler.addGameTimeEvent(5,self._changeAfterPauseInvocation);
        self.makeTrainExitedCall();
    };
    this._action[BoxState.AWAITING_ATTN_ACK][BellCode.Attn] = function() {
        if (self._attnAckTimeout) {
            self._attnAckTimeout.cancel();
            self._attnAckTimeout = null;
        }
        self.changeState(BoxState.AWAITING_OFFER_ACK);
        self._bell.scheduleBellCode(1,self._trainToOffer);
        self._offerAckTimeout = self._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,self._timeoutOfferAckInvocation);
    };
    this._action[BoxState.AWAITING_SHUNT_ATTN][BellCode.Attn] = this._echoBellAndChangeStateFunction(BoxState.AWAITING_END_SHUNT);
    this._action[BoxState.AWAITING_EXIT_ATTN_ACK][BellCode.Attn] = function() {
        if (self._attnExitAckTimeout) {
            self._attnExitAckTimeout.cancel();
            self._attnExitAckTimeout = null;
        }
        self.changeState(BoxState.AWAITING_OUT_OF_SECTION_ACK);
        self._bell.scheduleBellCode(1,BellCode.OutOfSection);
        self._awaitingOutOfSectionAckTimeout = self._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,self._timeoutAwaitingOutOfSectionAckInvocation);
    };
    this._action[BoxState.AWAITING_END_SHUNT][BellCode.EndShunt] = function() {
        const bellLength = self._bell.echoBellCode();
        self.cancelEnterSection(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToNONE);
    };
}

SignalBox.pauseTrainDelay = 5;
SignalBox.timeoutDelay = 60;

SignalBox.prototype = {
    changeState: function(state) {
        if (state !== this._state) {
            this._state = state;
            this._bell.setTooltip(state.getTooltip());
        }
    },
    changeAfterPause: function() {
        if (this._state === BoxState.PAUSE_AFTER_EXIT_ACK) {
            this.changeState(BoxState.NONE);
        }
    },
    startIdleTimer: function() {
        this._idleStarted = this._scheduler.getClock().getTicks();
    },
    trainIntoSection: function() {
        this._trainsIntoOurSection++;
//      console.log("Train into section for "+this._name+", Intocount="+this._trainsIntoOurSection+" LeftCount="+this._trainsLeftOurSection);
    },
    trainLeftSection: function() {
        this._trainsLeftOurSection++;
//      console.log("Train left section for "+this._name+", Intocount="+this._trainsIntoOurSection+" LeftCount="+this._trainsLeftOurSection);
    },
    changePrevState: function(state) {
        this._prevState = state;
    },
    timeoutAttnWhenIdle: function() {
        if (this._bell.isInBellCode()) {
            this._scheduler.rescheduleCurrentEvent(5); // Try us again in 5 (game since we're a game time event) seconds if we haven't been cancelled
        } else {
            this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. You called attention on the bell, but" +
                    " haven't then offered a train. I'm going to assume that this was a miscall, so if you need to offer a train you will" +
                    " need to call attention on the bell again.");
            this.changeState(BoxState.NONE);
        }
    },
    timeoutOfferAck: function() {
        if (this._bell.isInBellCode()) {
            this._scheduler.rescheduleCurrentEvent(5); // Try us again in 5 (game) seconds if we haven't been cancelled
        } else {
            this.changeState(BoxState.AWAITING_ATTN_ACK);
            this._bell.scheduleBellCode(0,BellCode.Attn);
            this._attnAckTimeout = this._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,this._timeoutAttnAckInvocation);
        }
    },
    timeoutAttnAck: function() {
        if (this._bell.isInBellCode()) {
            this._scheduler.rescheduleCurrentEvent(5); // Try us again in 5 (game) seconds if we haven't been cancelled
        } else {
            this._bell.scheduleBellCode(0,BellCode.Attn);
            this._scheduler.rescheduleCurrentEvent(SignalBox.timeoutDelay); // Try us again in 60 (game) seconds if we haven't been cancelled
        }
    },
    timeoutEnteringAck: function() {
        if (this._bell.isInBellCode()) {
            this._scheduler.rescheduleCurrentEvent(5); // Try us again in 5 (game) seconds if we haven't been cancelled
        } else {
            this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. You accepted my offer of " +
                    "a train, haven't acknowledged my Entering Section bell code. I'll resend that Entering Section " +
                    "code now, and I'll continue to expect an acknowledgement.");
            this._bell.scheduleBellCode(2,BellCode.EnteringSection);
            this._scheduler.rescheduleCurrentEvent(SignalBox.timeoutDelay); // Try us again in 60 (game) seconds if we haven't been cancelled
        }
    },
    timeoutAwaitingTheirTrainOnLine: function() {
        this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. You acknowledged my Train Entering " +
                "Section bell code, but haven't marked the train has being on line. Please do so now so that we're correctly showing " +
                "the train's status.");
        this._scheduler.rescheduleCurrentEvent(SignalBox.timeoutDelay); // Try us again in 60 (game) seconds if we haven't been cancelled
    },
    timeoutAwaitingOutOfSectionAck: function() {
        if (this._bell.isInBellCode()) {
            this._scheduler.rescheduleCurrentEvent(5); // Try us again in 5 (game) seconds if we haven't been cancelled
        } else {
            this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. You haven't acknowledged my Out Of Section" +
                "bell code (after acknowledging my call attention). I'll resend that Out Of Section code now, and " +
                "I'll continue to expect an acknowledgement.");
            this._bell.scheduleBellCode(2,BellCode.OutOfSection);
            this._scheduler.rescheduleCurrentEvent(SignalBox.timeoutDelay); // Try us again in 60 (game) seconds if we haven't been cancelled
        }
    },
    sendOutOfOurSection: function() {
        if (this._state === BoxState.PAUSE_TRAIN_START) {
            this._bell.scheduleBellCode(0,BellCode.EnteringSection);
            this.changeState(BoxState.AWAITING_ENTERING_ACK);
            this._enteringAckTimeout = this._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,this._timeoutEnteringAckInvocation);
        }
    },
    getName: function() {
        return this._name;
    },
    getBell: function() {
        return this._bell;
    },
    getState: function() {
        return this._state;
    },
    offerTrain: function(traincode,bellcode,invocation) {
        if (this.canTrainBeOffered(bellcode)) { // Note: this method must be provided by the actual signal box objects once their construction is complete
            this._preOfferState = this._state;
            this.changeState(BoxState.AWAITING_ATTN_ACK);
            this._trainToOffer = bellcode;
            this._action[BoxState.AWAITING_OFFER_ACK][bellcode] = this._action_correct_offer_ack;
            this._attnAckTimeout = this._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,this._timeoutAttnAckInvocation);
            this._invocationOnceTrainOnLine = invocation;
            this._bell.scheduleBellCode(1,BellCode.Attn);
            this._offeredTrains.push(traincode);
            return true;
        } else {
            return false;
        }
    },
    makeTrainOnLineCall: function() {
        if (this._awaitingTheirTrainOnlineTimeout) {
            this._awaitingTheirTrainOnlineTimeout.cancel();
            this._awaitingTheirTrainOnlineTimeout = null;
        }
        this._scheduler.addRealTimeEvent(0.1,this._invocationOnceTrainOnLine);
    },
    makeTrainExitedCall: function() {
        this._scheduler.addRealTimeEvent(0.1,this._invocationOnceTrainExited);
    },
    actOnBellPush: function() {
        // Invoke the action for our current state and the bell code (if it there isn't one, we ignore the bell push)
        if (this._action[this._state] && this._action[this._state][this._bell.getBellCode()]) {
            this._action[this._state][this._bell.getBellCode()]();
        }
    },
    exitTrain: function(invocation) {
        if (this.canTrainExit()) {
            this._invocationOnceTrainExited = invocation;
            this.changeState(BoxState.AWAITING_EXIT_ATTN_ACK);
            this._bell.scheduleBellCode(1,BellCode.Attn);
            this._attnExitAckTimeout = this._scheduler.addGameTimeEvent(SignalBox.timeoutDelay,this._timeoutAttnAckInvocation);
            return true;
        } else {
            return false;
        }
    },
    _trainCheckCounts: function(traincode) {
        if (this._trainsIntoOurSection !== this._trainsLeftOurSection) {
            this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. Train "+traincode+" has just arrived at my "+
                    "box, but I didn't receive a 'Train Entering Section' bell code from you. If you can send that bell code we can get "+
                    "back to normal operation.");
        }
        this._scheduler.addGameTimeEvent(300,new Invocation(this._trainCheckCounts,this,traincode));
    },
    trainJustArrived: function(traincode) {
        this.trainLeftSection();
        this._trainCheckCounts(traincode);
    },
    trainDepartedSection(traincode) {
        var complained = false;
        for(var i=0; i<this._offeredTrains.length; i++) {
            if (traincode === this._offeredTrains[i]) {
                this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. I've not had a 'Train Out "+
                        "Of Section' bell code from you for train "+traincode+". If the train has arrived can you send that bell code so "+
                        "we can get back to normal operation");
            }
            complained = true;
        }
        if (complained) {
            this._scheduler.addGameTimeEvent(300,new Invocation(this.trainDepartedSection,this,traincode));
        }
    },
    canTrainBeOffered: function(_bellcode) {
        throw new TypeError("canTrainBeOffered not overridden on "+this.constructor.name);
    },
    canTrainExit: function() {
        throw new TypeError("canTrainBeOffered not overridden "+this.constructor.name);
    },
    doEnterSectionReturningNewState: function(_bellLength) {
        throw new TypeError("doEnterSection not overriden on"+this.constructor.name);
    },
    cancelEnterSection: function(_bellLength) {
        throw new TypeError("cancelEnterSection not overriden on "+this.constructor.name);
    },
    scheduleCheckLineEmpty: function() {
        throw new TypeError("scheduleCheckLineEmpty not overriden on "+this.constructor.name);
    },
    scheduleCheckLineOccupied: function() {
        throw new TypeError("scheduleCheckLineOccupied not overriden on "+this.constructor.name);
    },
    doTrainOutOfSection: function() {
        throw new TypeError("doTrainOutOfSection not overriden on "+this.constructor.name);
    }
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.SignalBox = SignalBox;
    exports.BoxState  = BoxState;
}