//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals Subclass, Invocation */
/* jshint debug: true */

function TimeEvent(scheduler, invocation) {
    if (!!!invocation) {
        throw new Error("No invocation object passed when creating an event");
    }
    if (typeof invocation.invoke !== "function") {
        throw new Error("Invocation object passed does not have an invoke method");
    }
    this._scheduler = scheduler;
    this._invocation = invocation;
    this._id = this._nextEventId++;
}
TimeEvent.prototype = {
    doEvent: function() {
        this._invocation.invoke(this._scheduler._id);
    },
    isReady: function() {
        return (this.getRemainingMillis() < 0);
    },
    getRemainingMillis: function() {
        throw new Error("getRemainingTime not overriden on "+this.constructor.name);
    },
    setNewWhen: function() {
        throw new Error("setNewWhen not overriden on "+this.constructor.name);
    },
    cancel: function() {
        this._scheduler.cancelEvent(this);
    },
    isEarlierThan: function(other) {
        const ourRemainingTime = this.getRemainingMillis();
        const otherRemainingTime = other.getRemainingMillis();
        if (ourRemainingTime < otherRemainingTime) {
            return true;
        } else if (ourRemainingTime > otherRemainingTime) {
            return false;
        } else {
            // Same time, so the first registered wins
            return (this._id < other._id);
        }
    },
    _nextEventId: 1
};

function RealTimeEvent(scheduler, when, invocation) {
    TimeEvent.call(this, scheduler, invocation);
    if (!when && when !== 0) {
        throw new Error("no 'when' provided for "+this.constructor.name);
    }
    this._when = when*1000 + Date.now();
}
Subclass.make(RealTimeEvent,TimeEvent);
RealTimeEvent.prototype.getRemainingMillis = function() {
    return (this._when - Date.now());
};
RealTimeEvent.prototype.setNewWhen = function(when) {
    this._when = this._when = when*1000 + Date.now();
};
function GameTimeEvent(scheduler, when, invocation) {
    TimeEvent.call(this, scheduler, invocation);
    if (!when && when !== 0) {
        throw new Error("no 'when' provided for "+this.constructor.name);
    }
    this._when = when + this._scheduler._clock.getTicks();
}
Subclass.make(GameTimeEvent,TimeEvent);
GameTimeEvent.prototype.getRemainingMillis = function() {
    return (this._when-this._scheduler._clock.getTicks())*1000/this._scheduler._warpFactor;
};
GameTimeEvent.prototype.setNewWhen = function(when) {
    this._when = when + this._scheduler._clock.getTicks();
};

function Scheduler(window,id) {
    this._window = window;
    this._id = id;
    this._events = [];
    this._warpFactor = 1;
    this._clock = {getTicks: function() { return 0; }}; // Initialise to an 'null' clock
    this._eventTimer = null; // No timeout initially set
    this._paused = false;
    this._currentEvent = null;
}

Scheduler.prototype = {
    setClock : function(clock) {
        this._clock = clock;
    },
    getClock: function() {
        return this._clock;
    },
    setWarpFactor: function(factor) {
        this._warpFactor = factor;
        if (!!!this._paused) {
            this.scheduleNextEvent(); // This removes the currently scheduled next event and replaces it with the (now) next event
        }
    },
    pause: function() {
        if (!!!this._paused) {
            // Cancel the currently scheduled next event
            this._paused = true;
            this.cancelNextEvent(); // Note that we don't schedule a replacement. That happens on the unpause
        }
    },
    unpause: function() {
        if (this.paused) {
            if (this._warpFactor === 0) {
                throw new Error("Attempting to unpause when warp factor is zero");
            }
            this._paused = false;
            this.scheduleNextEvent();
        }
    },
    getWarpFactor: function() {
        return this._warpFactor;
    },
    isEmpty: function() {
        return this._events.length === 0;
    },
    executeEvents: function() {
        var doneAnEvent;
        do {
            doneAnEvent = this.executeFirstEvent();
        } while(doneAnEvent);
        this.scheduleNextEvent();
    },
    executeFirstEvent: function() {
        if (this.isEmpty()) {
            return false;
        }
        this._moveNextEventToFront();
        if (this._events[0].isReady()) {
            this._currentEvent = this._events.shift();
            this._currentEvent.doEvent();
            this._currentEvent = null;
            return true;
        } else {
            return false;
        }
    },
    getCurrentEvent: function() {
        return this._currentEvent;
    },
    rescheduleCurrentEvent: function(when) {
        if (this._currentEvent) {
            this._currentEvent.setNewWhen(when);
            this._addEvent(this._currentEvent);
        } else {
            throw new Error("No current Event to reschedule!");
        }
    },
    addRealTimeEvent: function(when,invocation) {
        if (!!!Invocation.isNumber(when)) {
            debugger;
            throw new Error("No offset provided when adding creating a real time event");
        }
        return this._addEvent(new RealTimeEvent(this,when,invocation));
    },
    addGameTimeEvent: function(when,invocation) {
        if (!!!Invocation.isNumber(when)) {
            debugger;
            throw new Error("No offset provided when adding creating a game time event");
        }
        return this._addEvent(new GameTimeEvent(this,when,invocation));
    },
    _addEvent: function(event){
        this._events.push(event);
        this.scheduleNextEvent();
        return event;
    },
    cancelEvent: function(event) {
        for(var i=0; i<this._events.length; i++) {
            if (this._events[i] === event) {
                this._events.splice(i,1);
                break;
            }
        }
        this.scheduleNextEvent();
    },
    _moveNextEventToFront: function() {
        var earliestEventIndex = 0; 
        for (var i=1; i<this._events.length; i++) {
            if (this._events[i].isEarlierThan(this._events[earliestEventIndex])) {
                earliestEventIndex = i;
            }
        }
        if (earliestEventIndex !== 0) {
            const temp = this._events[0];
            this._events[0] = this._events[earliestEventIndex];
            this._events[earliestEventIndex] = temp;
        }
    },
    cancelNextEvent: function() {
        if (this._eventTimer) {
            this._window.clearTimeout(this._eventTimer);
            this._eventTimer = null;
        }
    },
    scheduleNextEvent: function() {
        this.cancelNextEvent();
        if (!!!this.isEmpty()) {
            this._moveNextEventToFront();
            const when = (this._events[0].getRemainingMillis()<0 ? 0 : this._events[0].getRemainingMillis());
            const self = this; // Capture this
            this._eventTimer = this._window.setTimeout(function() { self.executeEvents(); },when);
        }
    }
};