//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals CommutatorState: true, Subclass, module, exports */
/* exported CommutatorState */
function Position(name,preposition,description,allowsMultipleTrains,length,overlap) {
    this._name = name;
    this._preposition = (preposition ? preposition : "at");
    this._description = (description ? description : name);
    this._allowsMultipleTrains = allowsMultipleTrains;
    this._occupier = null;
    this._listeners = [];
    this._length = length;
    this._overlap = overlap;
}

Position.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _fireListeners: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i]();
        }
    },
    getName: function() { return this._name; },
    getDescription : function() { return this._description; },
    getDescriptionWithPreposition: function() { return this._preposition+" "+this._description; },
    getOccupier: function() { return this._occupier; },
    getLength: function() { return this._length; },
    getOverlap: function() { return this._overlap; },
    occupy: function(occupier) { this._occupier = occupier; this._fireListeners(); },
    allowsMultipleTrains: function() { return this._allowsMultipleTrains; }
};

function Eastleigh() {
    this._zw30 = false;
    this._next = false;
    this._pointPulled = false;
    this._listeners = [];
}
Eastleigh.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _fireListeners: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i]();
        }
    },
    getZw30 : function () { return this._zw30; },
    getZw30Next : function() { return this._next; },
    getPointPulled: function() { return this._pointPulled; },
    setSignals: function(zw30,next) {
        this._zw30 = zw30;
        this._next = next;
        this._fireListeners();
    },
    setPoint: function(pulled) { this._pointPulled = pulled; this._fireListeners(); },
};

var CommutatorState = {
    TRAIN_ON_LINE: 51,
    NORMAL: 144,
    LINE_CLEAR: 224
};

function Commutator(name) {
    this._name = name;
    this._state = CommutatorState.NORMAL;
    this._listeners = [];
}

Commutator.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _fireListeners: function(oldState) {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i](oldState);
        }
    },
    getName: function() { return this._name; },
    getState: function() { return this._state; },
    setState: function(state) {
        const oldState = this._state;
        this._state = state; 
        this._fireListeners(oldState);
    }
};

function Lever(name,description,initially) {
    this._name = name;
    this._description = description;
    this._pulled = initially;
    this._lockCause = null;
    this._listeners = [];
}
Lever.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _fireListeners: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i]();
        }
    },
    getName: function() { return this._name; },
    getDescription: function() { return this._description; },
    isLocked: function() { return this._lockCause !== null; },
    getLockCause: function() { return this._lockCause ? this._lockCause : ""; },
    unlock: function() { this._lockCause = null; }, // Doesn't fire the listeners since it's called from computeLocks (which is called via a listener itself)
    setLocked: function(cause) { this._lockCause = cause; }, // Doesn't fire the listeners since it's called from computeLocks (which is called via a listener itself)
    isPulled: function() { return this._pulled; },
    pull: function() {
        if (!this.isLocked()) {
            this._pulled = true;
            this._fireListeners();
            return true;
        } else {
            return false; 
        }
    },
    push: function() {
        if (!this.isLocked()) {
            this._pulled = false;
            this._fireListeners();
            return true;
         } else {
             return false;
         }
    }
};

function OnePullLever(name,description,commutator) {
    Lever.call(this,name,description,false);
    this._commutator = commutator;
    this._commutator.addListener(this.commutatorListener());
    this._onePullPermitted = false;
}
Subclass.make(OnePullLever,Lever);
OnePullLever.prototype.pull = function() {
    if (Lever.prototype.pull.apply(this)) {
        this._onePullPermitted = false;
    }
};
OnePullLever.prototype.isLocked = function() {
    const superResult = Lever.prototype.isLocked.apply(this);
    if (superResult) return superResult;
    if (this._pulled) return false;
    if (this._onePullPermitted) return false;
    return true; // If pulled it is unlocked (so it can be pushed), otherwise it is locked unless one pull is permitted
};
OnePullLever.prototype.getLockCause = function() {
    const superResult = Lever.prototype.getLockCause.apply(this);
    if (superResult) return superResult;
    if (this._pulled || this._onePullPermitted) {
        return "";
    } else {
        return "This lever cannot be pulled a second time without the commutator being reset";
    }
};
OnePullLever.prototype.commutatorListener = function() {
    const self = this;
    return function(oldState) {
        if ( (self._commutator.getState() === CommutatorState.LINE_CLEAR) && (oldState !== CommutatorState.LINE_CLEAR)) {
            self._onePullPermitted = true;
            self._fireListeners();
        }
    };
};

function Light() {
    this._state = false;
    this._listeners = [];
}
Light.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _fireListeners: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i]();
        }
    },
    isFree: function() {
        return this._state;
    },
    lock: function() {
        this._state = false; this._fireListeners();
    },
    release: function() {
        this._state = true; this._fireListeners();
    }
};

function RomseyBoxModel() {
    const self = this;
    var computeLockListener = function() {
        return function() {
            self.computeLocks();
        };
    }();

    this._redbridgeUp   = new Commutator("Redbridge Up Line");
    this._redbridgeDown = new Commutator("Redbridge Down Line");
    this._kimbridgeUp   = new Commutator("Kimbridge Up Line");
    this._kimbridgeDown = new Commutator("Kimbridge Down Line");
    this._eastleigh = new Eastleigh();
    this._lever = [];
    this._lever[1 ] = new Lever(1, "Down Main Outer Home & Distant", false);
    this._lever[2 ] = new Lever(2, "Down Main Inner Home", false);
    this._lever[3 ] = new OnePullLever(3, "Down Main Starting", this._kimbridgeDown);
    this._lever[4 ] = new Lever(4, "Down Branch Distant", false);
    this._lever[5 ] = new Lever(5, "Down Branch Home", false);
    this._lever[6 ] = new Lever(6, "Direction Lever From Eastleigh", false);
    this._lever[7 ] = new Lever(7, "Down Main Points", false);
    this._lever[8 ] = new Lever(8, "Up Branch Points", false);
    this._lever[9 ] = new Lever(9, "FPL on No.8", true);
    this._lever[10] = new Lever(10,"Cross over Points East", false);
    this._lever[11] = new Lever(11,"Up Main Points", false);
    this._lever[12] = new Lever(12,"Up Siding No. 1 Points", false);
    this._lever[13] = new OnePullLever(13,"Up Branch Starting", this._redbridgeUp);
    this._lever[14] = new Lever(14,"Up Siding No.2 Points Release", false);
    this._lever[15] = new Lever(15,"Cross-over Points West Release", false);
    this._lever[16] = new Lever(16,"Up Branch Inner Home", false);
    this._lever[17] = new Lever(17,"Up Main Advance Starting", false);
    this._lever[18] = new Lever(18,"Up Main Inner Home", false);
    this._lever[19] = new Lever(19,"Down Main Draw Ahead", false);
    this._lever[20] = new Lever(20,"Up Main Outer Home & Distant", false);
    this._lever[21] = new Lever(21,"From Up Main Shunt", false);
    this._lever[22] = new Lever(22,"From Down Main Shunt", false);
    this._lever[23] = new Lever(23,"From Up Siding No.1 Shunt", false);
    for(var i=1; i<=23; i++) {
        this._lever[i].addListener(computeLockListener);
    }

    this._light17 = new Light();
    this._light17.addListener(computeLockListener);
    
    this._positions = {
            //                name preposition descr multiple leng over
            PC : new Position("PC",undefined,undefined,false, 7000,   0), // In front of signal ZW30
            PD : new Position("PD",undefined,undefined,false, 2000,   0), // In front of signal 1R
            PE : new Position("PE",undefined,undefined,false,  670,   0), // Between Halterworth Crossing and PF
            PF : new Position("PF",undefined,undefined,false,  200, 300), // In front of Signal 1
            PG : new Position("PG",undefined,undefined,false,  300, 200), // In front of Points 11
            PH : new Position("PH",undefined,undefined,false,  300, 200), // In front of Signal 2
            UZ : new Position("UZ",undefined,undefined,false,  580, 200), // In front of Signal 17
            C  : new Position("C" ,undefined,undefined,false,  220, 200),  // Immediately after Points 8 on up main (overlap 200yds)
            J  : new Position("J" ,undefined,undefined,false,  200, 200),  // In front of Points 7 (overlap 200 yds)
            D  : new Position("D" ,undefined,undefined,false,  360, 100),  // In front of Signal 13
            H  : new Position("H" ,undefined,undefined,false,  150, 100),  // After signal 5 (overlap 200 yds)
            K  : new Position("K" ,undefined,undefined,false,  200, 100),  // In front of Points 10 on down main (overlap 200 yds)
            L  : new Position("L" ,undefined,undefined,false,  100, 200),  // Between Points 10 and Platform 2
            B  : new Position("B" ,undefined,undefined,false,  300, 100),  // Between Up siding 2 and Crossover Points West
            A  : new Position("A" ,undefined,undefined,false,  250, 440),  // In front of signal 20
            P1 : new Position("P1",undefined,"Platform 1 (up line)",false,100,100),
            P2 : new Position("P2",undefined,"Platform 2 (down line)",false,100,100),
            Kimbridge : new Position("Kimbridge",undefined,undefined,true,10,0),
            Redbridge : new Position("Redbridge",undefined,undefined,true,10,0),
            Eastleigh : new Position("Eastleigh",undefined,undefined,true,10,0),
            BeforeSignal5 : new Position("BeforeSignal5","Before","Signal 5",false,700,0),
            BetweenKimbridgeAndRomsey : new Position("BetweenKimbridgeAndRomsey","Between","Kimbridge and Romsey",true,5600,0),
            BetweenRomseyAndRedbridge : new Position("BetweenRomseyAndRedbridge","Between","Romsey and Redbridge",true,9000,0),
            BetweenRedbridgeAndRomsey : new Position("BetweenRedbridgeAndRomsey","Between","Redbridge and Romsey",false,8300,0),
            BetweenRomseyAndKimbridge : new Position("BetweenRomseyAndKimbridge","Between","Romsey and Kimbridge",false,5600,0),
            BetweenEastleighAndPC     : new Position("BetweenEastleighAndPC","Between","Eastleigh and location PC",false,440,0)
    };
    for(var p in this._positions) {
        if (!!!this._positions.hasOwnProperty(p)) { continue; }
        this._positions[p].addListener(computeLockListener);
    }
    this.computeLocks();
    this._listeners = [];

    // And add computeLocks as listeners for the commutators too
    this._redbridgeUp.addListener(computeLockListener);
    this._redbridgeDown.addListener(computeLockListener);
    this._kimbridgeUp.addListener(computeLockListener);
    this._kimbridgeDown.addListener(computeLockListener);
}

RomseyBoxModel.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
        for(var i=1; i<this._lever.length; i++) {
            this._lever[i].addListener(listener);
        }
        for(var index in this._positions) {
            if (this._positions.hasOwnProperty(index)) {
                this._positions[index].addListener(listener);
            }
        }
        this._light17.addListener(listener);
        this._redbridgeUp.addListener(listener);
        this._redbridgeDown.addListener(listener);
        this._kimbridgeUp.addListener(listener);
        this._kimbridgeDown.addListener(listener);
        this._eastleigh.addListener(listener);
    },
    _fireListeners : function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i]();
        }
    },
    getLever : function(i) { return this._lever[i]; },
    getRedbridgeUpCommutator   : function() { return this._redbridgeUp; },
    getRedbridgeDownCommutator : function() { return this._redbridgeDown; },
    getKimbridgeUpCommutator   : function() { return this._kimbridgeUp; },
    getKimbridgeDownCommutator : function() { return this._kimbridgeDown; },
    getLight17: function() { return this._light17; },
    getEastleigh : function() { return this._eastleigh; },
    getPosition: function(name) { return this._positions[name]; },
    computeLocks : function() {
        var locks = function(lever, target) {
            if (lever.isPulled()) { target.setLocked("Locked by "+lever.getName()+" being pulled"); }
        };
        var releasedBySingle = function(target,lever) {
            if (!!!lever.isPulled()) {target.setLocked("Locked by "+lever.getName()+" being unpulled"); }
            locks(target,lever);
        };
        var releasedByBoth = function(target, lever1, lever2) {
            if (!!!lever1.isPulled() && !!!lever2.isPulled()) { target.setLocked("Locked by both "+lever1.getName()+" and "+lever2.getName() +" being unpulled. Both must be pulled to release this lever"); }
            else if (!!!lever1.isPulled()) { target.setLocked("Locked by "+lever1.getName()+" being unpulled"); }
            else if (!!!lever2.isPulled()) { target.setLocked("Locked by "+lever2.getName()+" being unpulled"); }
            locks(target,lever1);
            locks(target,lever2);
        };
        var releasedByEither = function(target, lever1, lever2) {
            if (!!!lever1.isPulled() && !!!lever2.isPulled()) { target.setLocked("Locked by both "+lever1.getName()+" and "+lever2.getName() +" being unpulled"); }
            locks(target,lever1);
            locks(target,lever2);
        };
        var locksN = function(lever, condition, target) {
            if (lever.isPulled() && !!!condition.isPulled()) { target.setLocked("Locked by "+lever.getName()+" being pulled and "+condition.getName()+" being unpulled"); }
        };
        var locksR = function(lever, condition, target) {
            if (lever.isPulled() && condition.isPulled()) { target.setLocked("Locked by both "+lever.getName()+" and "+condition.getName()+" being pulled"); }
        };
        var locksBW = locks;

        var posLock = function(position, target) {
            if (position.getOccupier() !== null && !!!target.isPulled()) { target.setLocked("Locked by position "+position.getName()+" being occupied and this lever not already pulled"); }
        };
        var posLoPL = function(position, target) {
            if (position.getOccupier() !== null && target.isPulled()) { target.setLocked("Locked by position "+position.getName()+" being occupied and this lever is pulled"); }
        };
        var posLoBW = function(position, target) {
            if (position.getOccupier() !== null) { target.setLocked("Locked by position "+position.getName()+" being occupied"); }
        };

        // Stage 0 assume unlocked
        this._lever.forEach(function(_v,i,a) { a[i].unlock(); });

        // Stage 1 - Apply track circuit locks (so that the lever lock explanations win if they also lock the lever)
        // Note: Signals can be moved to on when the circuit is occupied
        //       Points cannot be changed while the circuit is occupied
        // Signal  1 locked by PH, PG
        // Signal  2 locked by J, K, L, P2
        // Signal  5 locked by H, K, L, P2
        // Points  7 locked by H, J
        // Points 14 locked by B
        // Signal 16 locked by J, D
        // Signal 17 locked by PG, PF, PE, PD, PC
        // Signal 18 locked by C, UZ
        // Signal 20 locked by B, P1
        // Signal 21 locked by L, P2
        // Signal 22 locked by C, UZ
        // Signal 23 locked by if 8 J, D else C, UZ
        posLock(this._positions.PH,this._lever[1] );
        posLock(this._positions.PG,this._lever[1] );
        posLock(this._positions.J ,this._lever[2] );
        posLock(this._positions.K ,this._lever[2] );
        posLock(this._positions.L ,this._lever[2] );
        posLock(this._positions.P2,this._lever[2] );
        posLock(this._positions.H ,this._lever[5] );
        posLock(this._positions.K ,this._lever[5] );
        posLock(this._positions.L ,this._lever[5] );
        posLock(this._positions.P2,this._lever[5] );
        posLoBW(this._positions.PC,this._lever[6] );
        posLoBW(this._positions.PD,this._lever[6] );
        posLoBW(this._positions.PE,this._lever[6] );
        posLoBW(this._positions.PF,this._lever[6] );
        posLoBW(this._positions.H ,this._lever[7] );
        posLoBW(this._positions.J ,this._lever[7] );
        posLoPL(this._positions.PG,this._lever[11]);
        posLoBW(this._positions.B ,this._lever[14]);
        posLock(this._positions.J ,this._lever[16]);
        posLock(this._positions.D ,this._lever[16]);
        posLock(this._positions.PG,this._lever[17]);
        posLock(this._positions.PF,this._lever[17]);
        posLock(this._positions.PE,this._lever[17]);
        posLock(this._positions.PD,this._lever[17]);
        posLock(this._positions.PC,this._lever[17]);
        posLock(this._positions.C ,this._lever[18]);
        posLock(this._positions.UZ,this._lever[18]);
        posLock(this._positions.B ,this._lever[20]);
        posLock(this._positions.P1,this._lever[20]);
        posLock(this._positions.L ,this._lever[21]);
        posLock(this._positions.P2,this._lever[21]);
        posLock(this._positions.C ,this._lever[22]);
        posLock(this._positions.UZ,this._lever[22]);
        if (this._lever[8].isPulled()) {
          posLock(this._positions.J ,this._lever[23]);
          posLock(this._positions.D ,this._lever[23]);
        } else {
          posLock(this._positions.C ,this._lever[23]);
          posLock(this._positions.UZ,this._lever[23]);
        }

        // Lever  1 Released by          Locks 11
        // Lever  2 Released by 7        Locks 15
        // Lever  3 Released by BLOCK    Locks 7(B/w), 10(B/W), 15, 19, 22
        // Lever  4 Released by 3, 5     Locks
        // Lever  5 Released by          Locks 7, 10, 15
        // Lever  6 Released by          Locks (Lever 11 locks 6 electrically without a reverse lock applied (i.e. 6 does not lock 11)
        // Lever  7 Released by          Locks 5, 8, 10
        // Lever  8 Released by          Locks 7, 18
        // Lever  9 Released by          Locks 8(B/W)
        // Lever 10 Released by          Locks 5, 7, 12, 15, 16, 18
        // Lever 11 Released by          Locks 1, 6 (see lever 6)
        // Lever 12 Released by          Locks 10, 16, 18
        // Lever 13 Released by BLOCK    Locks 10(B/W), 12(B/W), (21 when 8R)
        // Lever 14 Released by          Locks 15, 16, 18, 20
        // Lever 15 Released by          Locks 2, 3, 5, 10, 14, 16, 18, 19, 20
        // Lever 16 Released by 8, 9     Locks 10, 12, 14, 15
        // Lever 17 Released by 11       Locks 10(B/W), 12(B/W) (21 when 8N) (also requires 17 free light lit)
        // Lever 18 Released by 9        Locks 8, 10, 12, 14, 15
        // Lever 19 Released by BLOCK    Locks 3, 7(B/W), 10(B/W), 15, 22
        // Lever 20 Released by 9        Locks 14, 15
        // Lever 21 Released by 10 or 12 Locks 8(B/W), (13 when 8R), (17 when 8N), 22, 23
        // Lever 22 Released by 9, 10    Locks 3, 19, 21
        // Lever 23 Released by 9, 12    Locks 21

        // Stage 2 Apply released by rules
        releasedBySingle(this._lever[2],this._lever[7]);
        releasedByBoth(this._lever[4],this._lever[3],this._lever[5]);
        releasedByBoth(this._lever[16],this._lever[8],this._lever[9]);
        releasedBySingle(this._lever[17],this._lever[11]);
        releasedBySingle(this._lever[18],this._lever[9]);
        releasedBySingle(this._lever[20],this._lever[9]);
        releasedByEither(this._lever[21],this._lever[10],this._lever[12]);
        releasedByBoth(this._lever[22],this._lever[9],this._lever[10]);
        releasedByBoth(this._lever[23],this._lever[9],this._lever[12]);

        // Stage 3 standard locks - if the lever is pulled, the target must be held normal - note that the chart has the reciprocal locks
        // defined (so we don't need to check target is at normal - if it get pulled, lever must be held normal)
        locks(this._lever[ 1],this._lever[11]);
        locks(this._lever[ 2],this._lever[15]);
        locks(this._lever[ 3],this._lever[15]);
        locks(this._lever[ 3],this._lever[19]);
        locks(this._lever[ 3],this._lever[22]);

        locks(this._lever[ 5],this._lever[ 7]);
        locks(this._lever[ 5],this._lever[10]);
        locks(this._lever[ 5],this._lever[15]);

        locks(this._lever[ 7],this._lever[ 5]);
        locks(this._lever[ 7],this._lever[ 8]);
        locks(this._lever[ 7],this._lever[10]);
        locks(this._lever[ 8],this._lever[ 7]);
        locks(this._lever[ 8],this._lever[18]);
        locks(this._lever[10],this._lever[ 5]);
        locks(this._lever[10],this._lever[ 7]);
        locks(this._lever[10],this._lever[12]);
        locks(this._lever[10],this._lever[15]);
        locks(this._lever[10],this._lever[16]);
        locks(this._lever[10],this._lever[18]);
        locks(this._lever[11],this._lever[ 1]);
        locks(this._lever[12],this._lever[10]);
        locks(this._lever[12],this._lever[16]);
        locks(this._lever[12],this._lever[18]);

        locks(this._lever[14],this._lever[15]);
        locks(this._lever[14],this._lever[16]);
        locks(this._lever[14],this._lever[16]);
        locks(this._lever[14],this._lever[18]);
        locks(this._lever[14],this._lever[20]);
        locks(this._lever[15],this._lever[ 2]);
        locks(this._lever[15],this._lever[ 3]);
        locks(this._lever[15],this._lever[ 5]);
        locks(this._lever[15],this._lever[10]);
        locks(this._lever[15],this._lever[14]);
        locks(this._lever[15],this._lever[16]);
        locks(this._lever[15],this._lever[18]);
        locks(this._lever[15],this._lever[19]);
        locks(this._lever[15],this._lever[20]);
        locks(this._lever[16],this._lever[10]);
        locks(this._lever[16],this._lever[12]);
        locks(this._lever[16],this._lever[14]);
        locks(this._lever[16],this._lever[15]);

        locks(this._lever[18],this._lever[ 8]);
        locks(this._lever[18],this._lever[10]);
        locks(this._lever[18],this._lever[12]);
        locks(this._lever[18],this._lever[14]);
        locks(this._lever[18],this._lever[15]);
        locks(this._lever[19],this._lever[ 3]);
        locks(this._lever[19],this._lever[15]);
        locks(this._lever[19],this._lever[22]);
        locks(this._lever[20],this._lever[14]);
        locks(this._lever[20],this._lever[15]);
        locks(this._lever[21],this._lever[22]);
        locks(this._lever[21],this._lever[23]);
        locks(this._lever[22],this._lever[ 3]);
        locks(this._lever[22],this._lever[19]);
        locks(this._lever[22],this._lever[21]);
        locks(this._lever[23],this._lever[21]);

        // Stage 4 N locks - if the lever if pulled and the conditional is not, the target must be held normal (again reciposity in the chart)
        locksN(this._lever[17],this._lever[8],this._lever[21]);
        locksN(this._lever[21],this._lever[8],this._lever[17]);

        // Stage 5 R locks - if the lever is pulled and the conditional is also pulled, the target must be held normal (reciprocal chart)
        locksR(this._lever[13],this._lever[8],this._lever[21]);
        locksR(this._lever[21],this._lever[8],this._lever[13]);

        // Stage 6 B/W locks - if the lever is pulled, the target is locked (implementation same as for standard locks - the lack of reciposity in
        // the chart is what gives the B/W bahaviour
        locksBW(this._lever[ 3],this._lever[ 7]);
        locksBW(this._lever[ 3],this._lever[10]);
        locksBW(this._lever[ 9],this._lever[ 8]);
        locksBW(this._lever[13],this._lever[10]);
        locksBW(this._lever[13],this._lever[12]);
        locksBW(this._lever[17],this._lever[10]);
        locksBW(this._lever[17],this._lever[12]);
        locksBW(this._lever[19],this._lever[ 7]);
        locksBW(this._lever[19],this._lever[10]);
        locksBW(this._lever[21],this._lever[ 8]);

        // Stage 7 Note 1 - Lever 11 locks 6 electrically without a reverse lock (i.e. 6 does not lock 11)
        locks(this._lever[11],this._lever[ 6]);

        // Stage 8 apply the absolute block instrument rules
        if (this._redbridgeUp.getState() !== CommutatorState.LINE_CLEAR && !!!this._lever[13].isPulled()) {
            this._lever[13].setLocked("Locked because Redbridge Up Line is not LINE CLEAR and this lever is not already pulled");
        }
        if (this._kimbridgeDown.getState() !== CommutatorState.LINE_CLEAR && !!!this._lever[3].isPulled()) {
            this._lever[3].setLocked("Locked because Kimbridge Down Line is not LINE CLEAR and this lever is not already pulled");
        }
        if (this._kimbridgeDown.getState() !== CommutatorState.TRAIN_ON_LINE && !!!this._lever[19].isPulled()) {
            this._lever[19].setLocked("Locked because Kimbridge Down Line is not TRAIN_ON_LINE and this lever is not already pulled");
        }

        // Stage 9 apply single line working rules
        if (!!!this._light17.isFree() && !!!this._lever[17].isPulled()) {
            this._lever[17].setLocked("Locked because the eastleigh box has not released the line and this lever is not already pulled");
        }
    }
};

//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
  exports.CommutatorState = CommutatorState;
  exports.RomseyBoxModel = RomseyBoxModel;
}