//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals RomseyBoxModel, RomseyBoxView, CommutatorState, Scheduler, Bell, Capture, DoubleLineBox, SingleLineBox, Timetables, Clock, RomseyMessageBox */
function RomseyBoxController(window,id) {
    this._model = new RomseyBoxModel();
    this._id = id;
    this._scheduler = new Scheduler(window,id);
    this._view = new RomseyBoxView(window,id,this._model);
    this._clock = new Clock(this._scheduler,this._view);
    this._scheduler.setClock(this._clock);
    this._messageBox = new RomseyMessageBox(this._clock,this._id);

    this._view.render();
    this._oldWarpFactor = null;
    
    const dosignalclick = function(thelever) {
        return function() {
            if (!!!thelever.isLocked()) {
                if (thelever.isPulled()) {
                    thelever.push();
                } else {
                    thelever.pull();
                }
            }
        };
    };
    
    const docommclick = function(commutator,lockLocations,state) {
        return function() {
            if (lockLocations.every(function(x) { return (x.getOccupier() === null); })) {
                commutator.setState(state);
            }
        };
    };
    
    const doLocationListener = function(lockLocations,elements) {
        return function() {
            var occupiedLocation = null;
            var occupier = null;
            var i;
            for(i=0; i<lockLocations.length; i++) {
                if (lockLocations[i].getOccupier() !== null) {
                    occupier = lockLocations[i].getOccupier();
                    occupiedLocation = lockLocations[i].getDescriptionWithPreposition();
                }
            }
            for(i=0; i<elements.length; i++) {
                if (occupiedLocation) {
                    elements[i].classList.add("locked");
                    elements[i].title = "Locked because train "+occupier+" is "+occupiedLocation;
                } else {
                    elements[i].classList.remove("locked");
                    elements[i].title = "Unlocked";
                }
            }
        };
    };

    const dorender = function(view) {
        return function() {
            view.render();
        };
    };
    
    // Attach the render listener to the model - it gets fired whenever the model changes
    this._model.addListener(dorender(this._view));
    
    // Set up the lever listeners
    var i;
    for(i=1; i<=23; i++) {
        var signal = document.getElementById("djvRomseySignalBox-"+this._id+"-lever-"+i);
        var lever = this._model.getLever(i);
        
        signal.onclick = this.safely(dosignalclick(lever));
    }
    
    // Set up the bells
    const redbridgeTing = document.getElementById("djvRomseySignalBox-"+this._id+"-bell-sound-effect-ding");
    const eastleighTing = document.getElementById("djvRomseySignalBox-"+this._id+"-ding");
    const kimbridgeTing = document.getElementById("djvRomseySignalBox-"+this._id+"-metal-ding-sound-effect");
    const redbridgeBell = new Bell(this._scheduler,"Redbridge",redbridgeTing,document.getElementById("djvRomseySignalBox-"+this._id+"-redbridgebell"),this._clock,this._messageBox,this._view);
    const eastleighBell = new Bell(this._scheduler,"Eastleigh",eastleighTing,document.getElementById("djvRomseySignalBox-"+this._id+"-eastleighbell"),this._clock,this._messageBox,this._view);
    const kimbridgeBell = new Bell(this._scheduler,"Kimbridge",kimbridgeTing,document.getElementById("djvRomseySignalBox-"+this._id+"-kimbridgebell"),this._clock,this._messageBox,this._view);
    
    // Set up the signal boxes
    const redbridgeBox = new DoubleLineBox(this, redbridgeBell, this._model.getRedbridgeUpCommutator(), this._model.getRedbridgeDownCommutator());
    const eastleighBox = new SingleLineBox(this, eastleighBell, this._model.getLight17(), this._model.getLever(6));
    const kimbridgeBox = new DoubleLineBox(this, kimbridgeBell, this._model.getKimbridgeDownCommutator(), this._model.getKimbridgeUpCommutator());
    
    // Set up commutator buttons
    const setupcommbuttons = function(safely,id,boxname,commutator,locklocations) {
        const buttons = [{name: "trainonline", state: CommutatorState.TRAIN_ON_LINE},
                         {name: "lineclear",   state: CommutatorState.LINE_CLEAR},
                         {name: "normal",      state: CommutatorState.NORMAL}
                        ];
        var elements = [];
        for(var i=0; i<buttons.length; i++) {
            var div = document.getElementById("djvRomseySignalBox-"+id+"-"+boxname+"-switch-"+buttons[i].name);
            var button = document.getElementById("djvRomseySignalBox-"+id+"-"+boxname+"-switch-"+buttons[i].name+"-button");
            div.title = "Unlocked";
            button.onclick = safely(docommclick(commutator,locklocations,buttons[i].state));
            elements.push(div);
        }
        const loclist = doLocationListener(locklocations,elements);
        for(i=0; i< locklocations.length; i++) {
            locklocations[i].addListener(loclist);
        }
    };
    setupcommbuttons(this.safely,this._id,"redbridge",this._model.getRedbridgeDownCommutator(),
                     [this._model.getPosition("BetweenRedbridgeAndRomsey"),
                      this._model.getPosition("BeforeSignal5"),
                      this._model.getPosition("H")]
                    );
    setupcommbuttons(this.safely,this._id,"kimbridge",this._model.getKimbridgeUpCommutator(),
                     [this._model.getPosition("BetweenKimbridgeAndRomsey"),
                      this._model.getPosition("A"),
                      this._model.getPosition("B")]
                    );
    
    
    // Set up the plungers
    const thud = document.getElementById("djvRomseySignalBox-"+this._id+"-button-click-sound");
    document.getElementById("djvRomseySignalBox-"+this._id+"-redbridge-pushbutton").onclick =
        this.safely(this.doplungerpush(redbridgeBox,thud,"Redbridge"));
    document.getElementById("djvRomseySignalBox-"+this._id+"-eastleigh-pushbutton").onclick =
        this.safely(this.doplungerpush(eastleighBox,thud,"Eastleigh"));
    document.getElementById("djvRomseySignalBox-"+this._id+"-kimbridge-pushbutton").onclick =
        this.safely(this.doplungerpush(kimbridgeBox,thud,"Kimbridge"));
    
    // Set up the speed button
    document.getElementById("djvRomseySignalBox-"+this._id+"-pause").onclick =
        this.safely(this.doWarpFactor(0));
    document.getElementById("djvRomseySignalBox-"+this._id+"-x1").onclick =
        this.safely(this.doWarpFactor(1));
    document.getElementById("djvRomseySignalBox-"+this._id+"-x1").checked = true; // This is our default
    document.getElementById("djvRomseySignalBox-"+this._id+"-x2").onclick =
        this.safely(this.doWarpFactor(2));
    document.getElementById("djvRomseySignalBox-"+this._id+"-x5").onclick =
        this.safely(this.doWarpFactor(5));
    
    // Set up adding of trains
    const timetables = new Timetables(this,this._model,this._view,kimbridgeBox,eastleighBox,redbridgeBox);
    for(i=0; i<timetables.list.length; i++) {
        document.getElementById("djvRomseySignalBox-"+this._id+"-trains")
            .appendChild(new Option(timetables.list[i].getName(),i,false,false));
    }
    document.getElementById("djvRomseySignalBox-"+this._id+"-addtrain").onclick =
        this.safely(this.doaddtrain(timetables));
    
    // Start the clock
    this._clock.scheduleTick();
}

RomseyBoxController.prototype = {
    safely: function(f) {
        const self = this;
        return function() {
            Capture.exception(self._id, f, self);
        };
    },
    onCompletedBellPush: function(box) {
        const self = this;
        return function() {
            self._view.addBellPush(self._clock.getClockAsString(),"to "+box.getName(),box.getBell().getBellCode());
            box.actOnBellPush();
        };
    },
    doplungerpush: function(box,sound) {
        const self = this;
        return function() {
            if (sound) {
                sound.pause();
                sound.currentTime = 0;
                sound.play();
            }
            box.getBell().bellPushed(self.onCompletedBellPush(box));
        };
    },
    doWarpFactor: function(factor) {
        const self = this;
        return function() {
            self._clock.setWarpFactor(factor);
        };
    },
    doaddtrain: function(timetables) {
        const self = this;
        return function() {
            const choice =  document.getElementById("djvRomseySignalBox-"+self._id+"-trains").selectedIndex;
            if (choice !== -1) {
                timetables.list[choice].run(self._clock);
            }
        };
    },
    getScheduler: function() {
        return this._scheduler;
    },
    getMessageBox: function() {
        return this._messageBox;
    }
};