//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals BellCode, Invocation */
function Bell(scheduler, name, audio, element, clock, messageBox, view) {
    this._scheduler = scheduler;
    this._name = name;
    this._audio = audio;
    this._element = element;
    this._clock = clock;
    this._messageBox = messageBox;
    this._view = view;

    this._inGap = false;
    this._bellCode = null;
    this._startOfGapTime = 0;
    this._callWhenBellCodeComplete = null;
    
    this._markRingingInvocation = new Invocation(this.markRinging,this);
    this._playAudioInvocation = new Invocation(this._playAudio,this);
    this._unmarkRingingInvocation = new Invocation(this.unmarkRinging,this);
    this._checkEndOfBellCodeInvocation = new Invocation(this.checkEndOfBellCode,this);
}

Bell.prototype = {
    shortGap: 0.8,
    longGapMax: 2,
    longGap: 1.6,
    getName: function() {
        return this._name;
    },
    markRinging: function() {
        this._element.classList.add("ringing");
    },
    unmarkRinging: function() {
        this._element.classList.remove("ringing");
    },
    _playAudio: function() {
        this._audio.pause();
        this._audio.currentTime = 0;
        this._audio.play();
    },
    scheduleBellCode : function(delay, bellCode) {
        const gaps = bellCode.getGaps();
        const ding = function(when,self) {
            self._scheduler.addRealTimeEvent(when    ,self._markRingingInvocation);
            self._scheduler.addRealTimeEvent(when    ,self._playAudioInvocation);
            self._scheduler.addRealTimeEvent(when+0.5,self._unmarkRingingInvocation);
        };
        var when = delay;
        ding(when,this);
        for(var j=0; j< gaps.length; j++) {
            if (gaps[j] === "-") {
                when += this.longGap;
            } else {
                when += this.shortGap;
            }
            ding(when,this);
        }
        
        this._scheduler.addRealTimeEvent(when+this.longGapMax-this.shortGap,
                                         new Invocation(this._view.addBellPush,
                                                        this._view,
                                                        this._clock.getClockAsStringAfterXMilliseconds(when),"from "+this._name,bellCode
                                                       )
                                        );
        return when-delay;
    },
    bellPushed : function(toCallWhenComplete) {
        if (this._inGap) {
            this._gaps.push( (Date.now() - this._startOfGapTime) / 1000 );
        } else {
            this._gaps = [];
            this._inGap = true;
            this._callWhenBellCodeComplete = toCallWhenComplete;
        }
        this._startOfGapTime = Date.now();
        this._scheduler.addRealTimeEvent(this.longGapMax+0.001,this._checkEndOfBellCodeInvocation);
    },
    checkEndOfBellCode: function() {
        const sizeOfGap = (Date.now() - this._startOfGapTime) / 1000;
        if (sizeOfGap > this.longGapMax) {
            this._inGap = false;
            this._bellCode = BellCode.recognise(this._gaps);
            var sequence = "[ ";
            for(var i=0; i<this._gaps.length; i++) {
                if (i !== 0) {
                    sequence += ", ";
                }
                sequence += this._gaps[i];
            }
            sequence += " ]";
            if (!this._bellCode) {
                throw new Error("BellCode.recognise returned a falsey answer for "+sequence);
            } else if (this._bellCode !== BellCode.Unrecognised) {
                this._callWhenBellCodeComplete();
            } else {
                this._messageBox.showMessage("Supervisor","This is your supervisor. That bell code was not a recognised bell code. Please try to tap "+
                                             "your desired bell code again.",
                                             "Debugging details: The gaps between the bell pushes were: "+sequence
                                            );
            }
        }
        // else there's been another bellPush and hence another call to this function has been scheduled and we'll check again then!
    },
    isInBellCode: function() {
        return this._inGap;
    },
    getBellCode : function() {
        return this._bellCode;
    },
    echoBellCode: function() {
        return this.scheduleBellCode(0,this._bellCode)-1;
    },
    setTooltip: function(tip) {
        this._element.title = tip;
    }
};