/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals console, GameMap, ROT, TECHS, RepairShipOrder, ScrapShipOrder , Fighter, ShipDesignType */
/* exported TurnProcessor */
var TurnProcessor = {
	resetPlayerReports: function(players) {
		for(let player of players) {
			player.resetReports();
		}
	},
	resetPlayerCommsGrids: function(players) {
		for(let player of players) {
			player.resetCommsGrid();
		}
	},
	moveShips: function(players,workingMap) {
		for(let player of players) {
			if (player.active) {
				for(let ship of player.ships) {
					ship.retreatingTo = null;
					ship.makeJump(ship.order.jump);
					workingMap.addObject(ship.location, {ship: ship, owner:player});
				}
			}
		}
	},

	processTurn: function(turn,players,baseMap,orders) {
		const debug = location.hostname == 'virtual.internal';
		const events = [];  // Major events for the game history
		const shipsToBeDestroyed = [];

		// Step 0. Prep the players for the results of the turn
		const s2p = {};
		for(let player of players) {
			s2p[player.stylename] = player;
		}
		TurnProcessor.resetPlayerReports(players);

		// Ship orders
		// Step 1. create a new map and place all ships in their new locations
		const workingMap = new GameMap(baseMap);
		TurnProcessor.moveShips(players,workingMap);

		// Step 2. Resolve what happening in each hex
		workingMap.forEachHex(hex => {
			// Step 2a. Work out what is in the hex
			const bystyle = TurnProcessor.getShipsByStyle(hex);	// Each element of bystyle (when present) is itself a sparse array showing how many ships of that size are present
			const troopsByStyle = TurnProcessor.getTroopsByStyle(hex);

			// Step 2b. Look for non-dredger units
			TurnProcessor.updateShipStatus(hex);

			// Step 2c. Resolve combat
			if (Object.keys(bystyle).length > 1) {
				TurnProcessor.resolvePossibleCombat(s2p,
					hex,
					bystyle,
					debug,
					events,
					shipsToBeDestroyed); // Will also remove destroyed ships from hex.objects
			}

			// Step 2d. Resolve planet busting (if any)
			const thereWasAPlanet = Boolean(hex.planet);
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					TurnProcessor.processPlanetBusting(debug,events,baseMap,s2p,hex,object.ship);
				}
			}

			// Step 2e. Give the colony (if any) a report - also mark the hex as interdicted or not
			if (hex.owner && hex.level) {
				const player = s2p[hex.owner];
				const report = TurnProcessor.getReportOnOtherShips(s2p,hex,hex.owner);
				player.map.setColonyReport(hex.id,report);
				player.map.setLastScanned(hex.id,turn,report);
				const interdict = TurnProcessor.anyHostileShips(hex,hex.owner);
				baseMap.setInterdicted(hex.id,interdict);
				if (interdict) {
					player.addTurnNote("interdicted",hex.id);
				}
			} else {
				baseMap.setInterdicted(hex.id,false);
			}

			// Step 2f. And then for each ship determine what it does
			ROT.RNG.shuffle(hex.objects); // Shuffle the ships so that each colony ship has the same chance to be the first to colonise
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					// processShipOrder returns true is the ship is to be kept, false if it is to be destroyed
					if (!TurnProcessor.processShipOrder(debug,events,baseMap,s2p,hex,object.ship,object.owner,troopsByStyle,thereWasAPlanet)) {
						shipsToBeDestroyed.push(object);
					}
				}
			}

			// Step 2g. Resolve ground combat (if any)
			if (Object.keys(troopsByStyle).length > 1) {
				// processInvasion returns the 'winner' if the planet has suicide planet busted
				const winnerButBusted = TurnProcessor.processInvasion(debug,events,baseMap,s2p,hex,troopsByStyle);
				if (winnerButBusted) {
					TurnProcessor.suicidePlanetBust(debug,events,baseMap,s2p,hex,winnerButBusted,shipsToBeDestroyed);
				}
			} else if (Object.keys(troopsByStyle).length == 1) {
				baseMap.setMarines(hex.id,troopsByStyle[hex.owner]); // Put the marines (plus any additions) back on the planet
				hex.marines = troopsByStyle[hex.owner];
			}

			// Step 2h. Finally generate ship reports (after colonisation and ground combat) and note presence of other ships
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					TurnProcessor.updateReports(turn,s2p,hex,object.ship,object.owner,thereWasAPlanet);
				}
			}
		});

		// Step 3. Remove those ship no longer in existence
		for(let object of shipsToBeDestroyed) {
			if (!object.fighter) {
				object.owner.addTurnNote("shipDestroyed",object.ship.location, {name: object.ship.name});
				object.owner.destroyShip(object.ship);
				if (object.ship.report) {
					object.owner.addTurnUpdate(object.ship.report); // Since there's no other way the player will see the report!
				}
				if (object.ship.design.type === ShipDesignType.StarBase) {
					baseMap.removeStarBase(object.ship.location);
				}
			}
		}

		// Step 4. Apply non-ship orders if the player still has their capital (if not - they're eliminated)
		for(let i=0; i<players.length; i++) {
			const player = players[i];
			const stylename = player.stylename;
			const capitalHex = baseMap.hex(player.capital);
			if (capitalHex.planet && capitalHex.owner == stylename) {
				for(let order of orders[stylename]) {
					let update = order.apply(players[i],baseMap,turn,s2p);
					if (update) {
						players[i].addTurnUpdate(update);
					}
					if (update && debug) {
						console.info(`${stylename}: ${update}`);
					}
				}
			} else if (player.active){
				const elimText = (i==0 ? "You've been eliminated from the game as you have lost your capital!" : `A player, ${player.fullname}, has been eliminated from the game as they have lost their capital!`);
				const theBaseMap = baseMap;
				player.addTurnUpdate(elimText);
				events.push(elimText);
				player.deactivate();
				theBaseMap.forEachHex(hex => {
					if (hex.owner == stylename) {
						theBaseMap.removeColony(hex.id,true); // Also remove any starbases
					}
				});
			}
		}

		// Step 5. Get the players to recompute their comms grid following all the colonisation, invasions and destructions
		TurnProcessor.resetPlayerCommsGrids(players);

		return events;
	},
	getShipsByStyle: function(hex) {
		let bystyle = {}; // (re)build the mapping from stylename to a sparse array of ship counts (of a given ship size) present in the hex
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const name = object.owner.stylename;
				bystyle[name] = bystyle[name] ? bystyle[name] : [];
				bystyle[name][0] = bystyle[name][0] ? bystyle[name][0] : 1; // Use the 0 entry to count all ships
				bystyle[name][object.ship.size] = bystyle[name][object.ship.size] ? bystyle[name][object.ship.size]++ : 1;
			}
		}
		return bystyle;
	},
	getTroopsByStyle: function(hex) {
		let troopsByStyle = {};
		if (hex.planet && hex.marines) troopsByStyle[hex.owner] = hex.marines;
		return troopsByStyle;
	},
	updateShipStatus: function(hex) {
		let nonDredgerUnits = Boolean(hex.planet);
		for(let object of hex.objects) {
			if (!nonDredgerUnits) {
				nonDredgerUnits = (!Boolean(object.ship)) || object.ship.getDredgers() == 0;
			}
		}
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const ship = object.ship;
				// Presence of NonDedger units
				ship.nonDredgerUnits = nonDredgerUnits;
				ship.report = ""; // Initialise the report
			}
		}
	},
	processPlanetBusting: function(debug,events,baseMap,s2p,hex,ship) {
		// Planet Busting (if ordered to so)
		if (ship.order.bust) {
			if (!ship.canPlanetBust) {
				ship.report += `Cannot planet bust ${hex.id}. No operational Planet Busters! `;
			} else if (!hex.planet) {
				ship.report += `Cannot planet bust ${hex.id}. There is no planet here to bust! `;
			} else {
				const busted = `The planet at ${hex.id} has been destroyed! `;
				ship.report += busted;
				if (hex.owner) {
					const player = s2p[hex.owner];
					if (debug) console.info(busted);
					player.addTurnUpdate(busted);
					player.addTurnNote("planetBusted",hex.id);
				}
				events.push(busted);
				baseMap.destroyPlanet(hex.id);
				// Update the working map too
				delete hex.planet; delete hex.marines; delete hex.level;
			}
		}
	},
	processShipOrder: function(debug,events,baseMap,s2p,hex,ship,owner,troopsByStyle,thereWasAPlanet) {
		let keepShip = true;

		// Scrap (if so ordered)
		if (ship.order instanceof ScrapShipOrder) {
			keepShip = false;
			const actualRefund = ship.getScrapValue();
			ship.report = `Scrapped ${ship.name} for ${+actualRefund} PPs (which has been added to the Brought Forward amount). `;
			owner.carriedProduction += actualRefund;
		}

		// Repair orders
		if (ship.retreatingTo == null && ship.order instanceof RepairShipOrder) {
			if (ship.getRepairCost() == ship.order.value) {
				ship.repairAll();
				ship.report = `Fully repaired at a cost of ${ship.order.value} PPs. `;
				owner.carriedProduction -= ship.order.value;
			} else if (ship.retreatingTo !== null){
				ship.report = "Repair aborted as the ship was further damaged. ";
			}
		}

		// Colonisation (if ordered to do so)
		if (ship.order.colonise) {
			const hexowner = s2p[hex.owner];
			if (hex.planet && hex.owner !== owner.stylename && hex.level > 0) {
				ship.report += `Unable to colonise ${hex.id} as it has already been colonised by ${hexowner.fullname}. `;
			} else if (hex.planet && hex.owner !== owner.stylename && hex.level == 0) {
				ship.report += `There is a star base belonging to ${hexowner.fullname} here. Unable to colonise. `;
			} else if (hex.planet && hex.owner == owner.stylename && hex.level > 4) {
				ship.report += `Colonisation of ${hex.id} cannot be made as the colony is already at level ${hex.level}. `;
			} else if (hex.planet && hex.owner == owner.stylename && ship.getColonyImprovement() == 0) {
				ship.report += `Colonisation of ${hex.id} cannot be made as the colony ship does not have at least six undamaged colony blocks. `;
			} else if (hex.planet && hex.owner == owner.stylename && hex.level < 5) {
				ship.report += `Colonisation of ${hex.id} has been achieved adding ${ship.getColonyImprovement()} to the existing colony. `;
				baseMap.colonise(ship.location,owner.stylename,ship.getColonyImprovement());
				baseMap.addMarines(ship.location,ship.marines);
				hex.level += ship.getColonyImprovement();
				if (debug) console.info(`Colonisation of ${hex.id} by ${owner.fullname} raising it to ${hex.level}`);
				owner.addTurnNote("colonised",hex.id);
				keepShip = false;
			} else if (hex.planet) {
				ship.report += `Colonisation of ${hex.id} has been achieved. `;
				events.push(`Colonisation of ${hex.id} by ${owner.fullname}`);
				if (debug) console.info(`Colonisation of ${hex.id} by ${owner.fullname}`);
				baseMap.colonise(hex.id,owner.stylename,ship.getColonyImprovement());
				baseMap.addMarines(ship.location,ship.marines);
				hex.level = ship.getColonyImprovement(); // So that the events/updates are correct for the next ship
				hex.owner = owner.stylename;
				owner.addTurnNote("colonised",hex.id);
				keepShip = false;
			} else if (thereWasAPlanet) {
				ship.report += `Unable to colonise ${hex.id}, the planet here was destroyed! `;
			} else {
				ship.report += `Unable to colonise ${hex.id}, there is no planet here! `;
			}
		}

		// Collect troops (if so ordered)
		if (ship.order.marinesEmbarking) {
			if (hex.planet) {
				if (hex.owner == owner.stylename) {
					if (hex.marines > 0) {
						if (hex.marines >= ship.getMaxMarines()-ship.marines) {
							troopsByStyle[hex.owner] -= ship.getMaxMarines()-ship.marines;
							hex.marines -= ship.getMaxMarines()-ship.marines; // Our local copy of the hex details
							baseMap.addMarines(ship.location,ship.marines-ship.getMaxMarines()); // This removes the marines from the colony
							ship.marines = ship.getMaxMarines();
						} else {
							troopsByStyle[hex.owner] -= hex.marines;
							ship.report += `Only ${hex.marines} marine compan${hex.marines == 1 ? "y" : "ies"} loaded as that was all that was available. `;
							baseMap.setMarines(ship.location,0);
							ship.marines += hex.marines;
							hex.marines = 0;
						}
					} else {
						ship.report += "No marines loaded as none were available. ";
					}
				} else {
					ship.report += "No marines loaded as we don't own this colony. '";
				}
			} else {
				ship.report += "No marines loaded as there is no planet here. ";
			}
		}

		// Disembark troops (if so ordered) {
		if (ship.order.marinesDisembarking && ship.marines > 0) {
			if (hex.planet && hex.level) {
				troopsByStyle[owner.stylename] = troopsByStyle[owner.stylename] || 0;
				troopsByStyle[owner.stylename] += ship.marines;
				ship.marines = 0;
			} else if (hex.planet) {
				ship.report += "Marines have not disembarked as there is no colony at the planet here! ";
			} else {
				ship.report += "Marines have not disembarked as there is no planet here! ";
			}
		}

		// And return whether the ship is to be kept
		return keepShip;
	},
	processInvasion: function(debug,events,baseMap,s2p,hex,troopsByStyle) {
		let planetBusted = false;
		s2p[hex.owner].addTurnNote("invaded",hex.id);
		const involved = Object.keys(troopsByStyle);
		const [owner,marines] = TurnProcessor.conductGroundCombat(s2p,hex.id,troopsByStyle,debug,events);
		if (owner !== hex.owner) {
			const newowner = s2p[owner];
			const oldowner = s2p[hex.owner];
			baseMap.setInterdicted(hex.id,false); // The hex is no longer interdicted
			if (debug) console.info(`${hex.id} conquered by ${newowner.fullname}`);
			const newreport = (hex.level > 1 ? `We have conquered ${hex.id} with  ${marines} marine compan${marines == 1 ? "y" : "ies"} remaining after the combat.` : `We have destroyed the level 1 colony at ${hex.id}.`);
			const oldreport = (hex.level > 1 ? `We have lost ${hex.id} to ${newowner.fullname}` : `${newowner.fullname} has destroyed the colony at ${hex.id}`);
			newowner.addTurnUpdate(newreport);
			oldowner.addTurnUpdate(oldreport);
			newowner.map.setColonyReport(hex.id,newreport);
			oldowner.map.setColonyReport(hex.id,oldreport);
			for(let other of involved) {
				if (other !== owner && other !== hex.owner) {
					const otherowner = s2p[other];
					const otherreport = `We did not conquer ${hex.id} but ${newowner.fullnamer} conquered it instead.`;
					otherowner.addTurnUpdate(otherreport);
				}
			}

			if (oldowner.researchLevel(TECHS.planetaryDefences) == 8) {
				planetBusted = true;
					if (debug) console.info(`However ${oldowner} will be suicide planet busting`);
			} else {
				if (hex.level > 1) {
					events.push(`${hex.id} conquered by ${newowner.fullname}`);
					baseMap.colonise(hex.id,owner,-1); // Changes owner and reduces the level by one
					hex.owner = owner; hex.level--; // And update the copy we're using in this turn processor
					baseMap.setMarines(hex.id,marines);
					hex.marines = marines;
				} else {
					if (debug) console.info(`${hex.id} destroyed by ${newowner.fullname}`);
					events.push(`The colony at ${hex.id} destroyed by ${newowner.fullname}`);
					delete hex.owner; delete hex.level;
					baseMap.removeColony(hex.id);
					delete hex.owner; delete hex.level; delete hex.marines; // And update the copy we're using in this turn processor
				}
			}
		} else {
			baseMap.setMarines(hex.id,marines);
			hex.marines = marines;
			for(let other of involved) {
				if (other !== hex.owner) {
					const otherowner = s2p[other];
					const otherreport = `We did not conquer ${hex.id}.`;
					otherowner.addTurnUpdate(otherreport);
				}
			}
		}
		return planetBusted ? s2p[owner].fullname : null;
	},
	suicidePlanetBust: function(debug,events,baseMap,s2p,hex,winner,shipsToBeDestroyed) {
		if (debug) console.info(`${hex.id} suicide planet busted as a result of being conquered by ${winner}`);
		const text = `${hex.id} suicide planet busted as a result of being conquered by ${winner}`;
		const oldowner = s2p[hex.owner];
		events.push(text);
		baseMap.destroyPlanet(hex.id);
		oldowner.addTurnUpdate(text);
		oldowner.addTurnNote("planetBusted",hex.id);
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const ship = object.ship;
				ship.report += `The colony at ${hex.id} has suicide planet busted and ${ship.name} was destroyed. `;
				shipsToBeDestroyed.push(object);
			}
		}
		delete hex.owner; delete hex.level; delete hex.marines; delete hex.planet; // And update the copy we're using in this turn processor
	},
	updateReports: function(turn,s2p,hex,ship,owner,thereWasAPlanet) {
		// Scanning of the hex
		if (hex.planet && hex.owner) {
			const hexowner = s2p[hex.owner];
			const colonyString = `${ship.location} is a class ${hex.level} colony belonging to ${hexowner.fullname}.`;
			const marineString = `There ${hex.marines == 1 ? "is" : "are"} ${hex.marines} marine compan${hex.marines == 1? "y" : "ies"} here. `;
			ship.report += colonyString + (hex.marines ? " "+marineString : "");
		} else if (hex.planet) {
			ship.report += `${ship.location} is an uncolonised world. `;
		} else if (thereWasAPlanet) {
			ship.report += `The planet at ${ship.location} has been destroyed! `;
		}

		// Presence of other ships
		ship.report += TurnProcessor.getReportOnOtherShips(s2p,hex,owner.stylename);

		// Update the hex report (even if it's on the the toBeDestroyed list)
		owner.map.setLastScanned(hex.id,turn,ship.report); // If a player has multiple ships we will set this multiple times (to the same value)
	},
	getReportOnOtherShips: function(s2p,hex,owner) {
		const bystyle = TurnProcessor.getShipsByStyle(hex);
		delete bystyle[owner];
		let report = "";
		for(let otherowner of Object.keys(bystyle)) {
			for(let size=1; size<bystyle[otherowner].length; size++) {
				if (bystyle[otherowner][size]) {
					const other = s2p[otherowner];
					report += "There " +
						(bystyle[otherowner][size]==1 ? "is a" : "are "+bystyle[otherowner][size]) +
						" size "+size+" ship" +
						(bystyle[otherowner][size]==1 ? "" : "s") +
						" belonging to " +
						other.fullname +
						" here. ";
				}
			}
		}
		return report;
	},
	anyHostileShips: function(hex,owner) {
		for(let object of hex.objects) {
			if (object.ship && object.owner.stylename != owner && object.ship.hasWeapons(true,false)) {
				return true;
			}
		}
		return false;
	},
	resolvePossibleCombat: function(s2p,hex,bystyle,debug,events,shipsToBeDestroyed) {
		const from = TurnProcessor.listPlayers(s2p,Object.keys(bystyle));

		// Is this ship an intruder to the hex? - if so, notify the hex owner of an intrusion
		for(let object of hex.objects) {
			if (object.ship && object.owner && hex.owner && object.owner !== hex.owner) {
				s2p[hex.owner].addTurnNote("intruder",hex.id);
			}
		}

		let list = "";
		for(let i=0; i<hex.objects.length; i++) {
			if (i>0 && i<hex.objects.length-1) {
				list += ", ";
			} else if (i == hex.objects.length-1) {
				list += " and ";
			}
			list += hex.objects[i].ship.name;
		}

		// Does any ship have any weapons?
		if (hex.objects.some(object => object.ship.hasWeapons(true,false))) {
			const text = `Combat!!!: In ${hex.id} there are ships from ${from}: ${list}`;
			for(let involved of Object.keys(bystyle)) {
				s2p[involved].addTurnUpdate(text);
			}
			if (debug) console.info(text);
			events.push(`Ship Combat in ${hex.id} involving ships from ${from}`);
			TurnProcessor.resolveCombat(hex,debug,shipsToBeDestroyed);
		}
	},
	listPlayers: function(s2p,styles) {
		let from = "";
		for(let i=0; i<styles.length; i++) {
			if (i>0 && i<styles.length-1) {
				from += ", ";
			} else if (i == styles.length-1) {
				from += " and ";
			}
			from += s2p[styles[i]].fullname;
		}
		return from;
	},
	whosPresent: function(hex) {
			const bystyle = {}; // each element (when present) has a sparse array showing who has ships in the hex
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					const name = object.owner.stylename;
					bystyle[name] = bystyle[name] ? bystyle[name] : true;
				}
			}
		return Object.keys(bystyle);
	},
	resolveCombat(hex,debug,shipsToBeDestroyed) {
		let notified = {};
		let involved = [];
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const name = object.owner.stylename;
				if (!notified[name]) {
					object.owner.addTurnNote("shipCombat",hex.id);
					involved.push(object.owner);
					notified[name] = true;
				}
			}
		}
		let maxPhaseNumber = 1;
		for(let object of hex.objects) {
			object.owner.addTurnNote("shipCombat",hex.id);
			maxPhaseNumber = Math.max(maxPhaseNumber, object.ship.getCombatSpeed());
			// Also do the start of combat preparations (raising shield, launching fighters)
			if (object.ship) {
				object.ship.raiseShieldsAtStartOfCombat();
				const fighters = object.ship.launchFighters();
				for(let i=0; i<fighters.count; i++) {
					const fighter = new Fighter(object.ship,fighters.spec,i+1);
					hex.objects.push({ship: fighter, owner:object.owner, fighter: true});
					if (fighter.getCombatSpeed() > maxPhaseNumber) maxPhaseNumber = fighter.getCombatSpeed();
				}
			}
		}
		if (debug) console.info(`This combat will have ${maxPhaseNumber==0?"a single":maxPhaseNumber+1} phase${maxPhaseNumber==0?"":"s"} per combat round`);
		for(let inv of involved) {
			inv.addTurnUpdate(`This combat will have ${maxPhaseNumber==0?"a single":maxPhaseNumber+1} phase${maxPhaseNumber==0?"":"s"} per combat round`);
		}

		let round = 0;
		while((TurnProcessor.whosPresent(hex).length > 1) && (hex.objects.some(object => object.ship.hasWeapons(false,false)))) {
			round++;
			for(let phase=maxPhaseNumber; phase>=0; phase--) {
				let hitTargets = new Map();
				for(let object of hex.objects) {
					if (!object.ship) continue; // It's not a ship
					const owner = object.owner;
					const ship = object.ship;
					if (!ship.hasABridge()) continue; // It's lost all of its bridges
					ship.nextCombatPhase(); // Let the ship use its reflecting shields
					let targets = hex.objects.filter(other => other.ship && other.owner !== owner).map(obj => obj.ship);
					if (targets.length > 0) {
						const target = ROT.RNG.getItem(targets);
						const hits = ship.getHits(ship.getCombatSpeed() >= phase);
						if (hits > 0) {
							if (debug) console.info(`Round ${round} Phase ${phase+1}: ${ship.name} scores ${hits==1?"a hit":hits+" hits"} on ${target.name}`);
							for(let inv of involved) {
								inv.addTurnUpdate(`Round ${round} Phase ${phase+1}: ${ship.name} scores ${hits==1?"a hit":hits+" hits"} on ${target.name}`);
							}
							if (hitTargets.has(target)) {
								hitTargets.set(target,hitTargets.get(target)+hits);
							} else {
								hitTargets.set(target,hits);
							}
						}
					}
				}
				for(const [target,hits] of hitTargets) {
					let resolution = target.resolveHits(hits);
					if (debug && resolution) console.info(`Round ${round} Phase ${phase+1}: ${target.name} ${resolution}`);
					for(let inv of involved) {
						inv.addTurnUpdate(`Round ${round} Phase ${phase+1}: ${target.name} ${resolution}`);
					}
				}
				// Resolve destroyed ships
				let newlist = [];
				for(let object of hex.objects) {
					if (object.ship && !object.ship.hasABridge()) {
						shipsToBeDestroyed.push(object);
						if (debug) console.info(`Round ${round} Phase ${phase+1}: ${object.ship.name} has been destroyed`);
						for(let inv of involved) {
							inv.addTurnUpdate(`Round ${round} Phase ${phase+1}: ${object.ship.name} has been destroyed`);
						}
					} else{
						// The ship is still in the fight
						newlist.push(object);
					}
				}
				hex.objects = newlist;
			}

			// Resolve retreats if there's still more than one side
			if ((TurnProcessor.whosPresent(hex).length > 1) && (hex.objects.some(object => object.ship.hasWeapons(true,true)))) {
				let newlist = [];
				for(let object of hex.objects) {
					if (!object.ship || object.fighter) {
						newlist.push(object); // It's still going to be there but
						continue; // It's not a ship
					}
					const owner = object.owner;
					const ship = object.ship;
					if (ship && ship.getJumpRange() > 0) {
						const retreat = ship.getRetreatDestination(hex.objects.filter(other => other.ship && other.owner == owner).map(obj => obj.ship));
						if (retreat) {
							// Retreat the ship
							ship.retreatingTo = retreat;
							if (debug) console.info(`Round ${round}: ${ship.name} is retreating to ${ship.retreatingTo}`);
							for(let inv of involved) {
								inv.addTurnUpdate(`Round ${round}: ${ship.name} is retreating${inv === ship.owner?" to ${ship.owner}":""}`);
							}
						} else {
							// The ship is still in the fight, so we'll need its surrounding object
							newlist.push(object);
						}
					} else {
						// The ship is still in the fight, so we'll need its surrounding object
						newlist.push(object);
					}
				}
				hex.objects = newlist;
			}
		}

		// And now return any remaining fighters back to their carrier
		let newlist = [];
		for(let object of hex.objects) {
			if (object.ship && object.fighter) {
				// This might return the fighter to a destroyed carrier - but that's OK as the fighter won't survive anyway
				object.ship.carrier.returnAFighter();
			} else {
				newlist.push(object);
			}
		}
		hex.objects = newlist;
	},
	conductGroundCombat(s2p,location,troopsByStyle,debug,events) {
		const involved = Object.keys(troopsByStyle);
		if (debug) console.info(`Ground Combat at ${location}: ${TurnProcessor.listTroops(s2p,troopsByStyle)}`);
		for(let inv of involved) {
			s2p[inv].addTurnUpdate(`Ground Combat at ${location}: ${TurnProcessor.listTroops(s2p,troopsByStyle)}`);
		}
		const from = TurnProcessor.listPlayers(s2p,Object.keys(troopsByStyle));
		events.push(`Ground Combat at ${location} involving ${from}`);
		// Multiply up the companies by the planetary defence tech
		for(let owner of Object.keys(troopsByStyle)) {
			troopsByStyle[owner] *= s2p[owner].troopValue;
		}

		let report = function(round,troopsByStyle) {
			let report = `Round ${round} Effective Troops:`;
			let comma = "";
			for(let owner of Object.keys(troopsByStyle)) {
				report += `${comma} ${owner}:${troopsByStyle[owner]}`;
				comma = ",";
			}
			return report;
		};

		let round = 0;
		while(Object.keys(troopsByStyle).length > 1) {
			round++;
			const rep = report(round,troopsByStyle);
			for(let inv of involved) {
				s2p[inv].addTurnUpdate(rep);
			}

			if (debug) console.info(rep);
			let newTroopsByStyle = Object.assign({},troopsByStyle);
			const keys = Object.keys(newTroopsByStyle);
			for(let owner of keys) {
				for(let opponent of keys) {
					if (owner !== opponent) {
						newTroopsByStyle[opponent] = Math.max(0, newTroopsByStyle[opponent] - Math.floor(troopsByStyle[owner] * (50+ROT.RNG.getPercentage()/2)/100));
					}
				}
			}
			for(let owner of keys) {
				if (newTroopsByStyle[owner] == 0) {
					delete newTroopsByStyle[owner];
				}
			}
			troopsByStyle = newTroopsByStyle;
		}
		const rep = report(round+1,troopsByStyle);
		for(let inv of involved) {
			s2p[inv].addTurnUpdate(rep);
		}
		if (debug) console.info(rep);
		const winner = Object.keys(troopsByStyle)[0];
		return [winner, Math.floor(troopsByStyle[winner]/s2p[winner].troopValue)];
	},
	listTroops(s2p,troopsByStyle) {
		const keys = Object.keys(troopsByStyle);
		let list = "There ";
		for(let i=0; i<keys.length; i++) {
			const owner = keys[i];
			if (i == keys.length-1) {
				list +=" and ";
			} else if (i >0) {
				list += ", ";
			} else if (troopsByStyle[owner] == 1) {
				list += "is ";
			} else {
				list += "are ";
			}
			if (troopsByStyle[owner] == 1) {
				list += "a marine company from ";
			} else {
				list += troopsByStyle[owner]+" marine companies from ";
			}
			list += s2p[owner].fullname;
		}
		return list;
	}
};
