/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals ROT, Ship, module, exports */
/* exported Block, ShipDesign ShipDesignType */

class ShipDesignType {
	constructor(name) {
		this._name = name;
	}
	toString() {
		return this._name;
	}
	toJSON() {
		return this._name;
	}
}
ShipDesignType.Ship = new ShipDesignType("ship");
ShipDesignType.SpaceBase = new ShipDesignType("base");
ShipDesignType.StarBase = new ShipDesignType("starbase");
ShipDesignType.fromJSON = function(json) {
	if (json == "starbase") return ShipDesignType.StarBase;
	if (json == "base") return ShipDesignType.SpaceBase;
	return ShipDesignType.Ship;
};

class Block {
	constructor(key,size,cost,eps,epsj,fighterSpec) {
		if (key instanceof Block) {
			this._key = key._key; this._size = key._size; this._cost = key._cost;
			this._eps = key._eps; this._epsj = key._epsj; this._fighterSpec = key._fighterSpec;
		} else {
			this._key = key; this._size = size; this._cost = cost;
			this._eps = eps; this._epsj = epsj; this._fighterSpec = (key == "_df" ? fighterSpec : "");
		}
		this._destroyed = 0;
		this._powered = false;
	}
	get key() { return this._key; }
	get name() { return ShipDesign.SHIPBLOCKNAMES[this._key]; }
	get description() {
		if (this._key !== "_df") {
				return ShipDesign.SHIPBLOCKDESCRIPTIONS[this._key];
			}
		return `Fighter bay whose fighter has a combat speed of ${this._fighterSpec.speed} and is armed with a ${this._fighterSpec.weaponDesc}`;
	}
	getFullName(onShip) {
		let desc = this.name;
		if (onShip) {
			if (this.destroyed) {
				desc += " (destroyed)";
			} else if (this._destroyed > 0) {
				desc += ` (${this._destroyed} of ${this._size} blocks destroyed)`;
			} else if (!this._powered) {
				desc += " (unpowered)";
			}
		}
		return desc;
	}
	get size() { return this._size; }
	get cost() { return this._cost; }
	get eps()  { return this._eps; }
	get epsj() { return this._epsj; }
	get destroyed() {
		return (this._destroyed >= this._size);
	}
	get operational() {
		return ((this._destroyed < this._size) && (this._powered));
	}
	setPowered(powered) {
		this._powered = powered;
	}
	get damage() {
		return this._destroyed;
	}
	set damage(damage) {
		this._destroyed = damage;
	}
	addDamage() {
		this._destroyed++;
	}
	hit() {
		if (this.operational) {
			this._destroyed++;
		}
	}
	repairCost() {
		return (this._destroyed)*this._cost/this._size;
	}
	repair() {
		this._destroyed = 0;
	}
}

class ShipDesign {
	constructor(name,type,id) {
		// NOTE: Due to the way the mutate and cross methods work (for genetic algorithms) any keys that are of the form _ab (i.e. an underscore
		//		 followed by exactly two letters are reserved for the counts of the number of each kind of block
		this._name = name;
		this._identifier = id;
		ShipDesign.REGISTRY[this._identifier] = this;
		this._type = type;
		// Counts are of the number of this kind of block in the design
		// bridge (b0 = without computer, b1 and up with that level of computer)
		this._b0 = 0; this._b1 = 0; this._b2 = 0; this._b3 = 0; this._b4 = 0; this._b5 = 0; this._b6 = 0; this._b7 = 0; this._b8 = 0; this._b9 = 0;
		// p1 = chemical power plant, p2 = atomic, p3 = fusion, p4 = large fusion, p5 = anti-matter, p6 = large anti-matter, p7 = ultimate
		this._p1 = 0; this._p2 = 0; this._p3 = 0; this._p4 = 0; this._p5 = 0; this._p6 = 0; this._p7 = 0;
		// jx = Jump-x drive
		this._ja = 0; this._jb = 0; this._jc = 0; this._jd = 0; this._je = 0; this._jf = 0; this._jg = 0; this._jh = 0;
		// mx = Maneuver-x drive
		this._ma = 0; this._mb = 0; this._mc = 0; this._md = 0; this._me = 0; this._mf = 0; this._mg = 0; this._mh = 0;
		// wl = laser, wp = phasers, wa = plasma, wf = fusion, wr = par.acc. spinal mount, wc = par. acc.], ws = meson spinal, wm = meson, wt = anti-matter spinal, wx = anti-matter, wb = planet buster
		this._wl = 0; this._wp = 0; this._wa = 0; this._wf = 0; this._wr = 0; this._wc = 0; this._ws = 0; this._wm = 0; this._wt = 0; this._wx = 0; this._wb = 0;
		// da = armour, dc = damage control, ds = shield, df = fighter, di = improved shield, dg = regenerating shield, dr = reflecting shield, dv = invulnerability shield
		this._da = 0; this._dc = 0; this._ds = 0; this._df = 0; this._di = 0; this._dg = 0; this._dr = 0; this._dv = 0;
		// bb = marine block
		this._bb = 0;
		// cb = colony block
		this._cb = 0;
		// an = neural net, ad= dredger, at = tractor beam, as = statis beam
		this._an = 0; this._ad = 0; this._at = 0; this._as = 0;

		// No cached prototype (used for calculations) as yet
		this._protoShip = null;

		// How many of this class of ship has been built by each player (player not in object = no ships of class built)
		this._built = {};

		// Prereq tech levels
		this._computePrereqTechs();

		// Fighter techs if a fighter block is added
		this._fighterTechs = undefined;
	}
	clone(other) {
		if (Object.keys(this._built).length !== 0) throw new Error("Cannot clone from another design, this design has been built");
		this._forEachBlockType(bt => this[bt] = other[bt]);
	}
	mutate() {
		if (Object.keys(this._built).length !== 0) throw new Error("Cannot mutate designs that have been built");
		let keys = [];
		this._forEachBlockType(bt => { if (this._type === ShipDesignType.Ship || !(ShipDesign.SHIPBLOCKSHIPONLY[bt])) { keys.push(bt); }});
		const key = ROT.RNG.getItem(keys);

		if (this[key] == 0 || ROT.RNG.getPercentage() < 51) {
			this[key]++;
		} else {
			this[key]--;
		}

		this._computePrereqTechs();
		this._protoShip = null; // Remove the prototype ship)
	}
	cross(other) {
		if (Object.keys(this._built).length !== 0) throw new Error("Cannot cross this design as it has been built");
		let keys = [];
		this._forEachBlockType(bt => keys.push(bt));
		keys = ROT.RNG.shuffle(keys);
		let crossCount = Math.floor(ROT.RNG.getUniform()*keys.length);
		for(let i=0; i<crossCount; i++) {
			const key = keys[i];
			this[key] = other[key];
		}
	}
	delete() {
		// Remove this design from the registry
		delete ShipDesign.REGISTRY[this._identifier];
	}
	get id() { return this._identifier; }
	get name() { return this._name; }
	get type() { return this._type; }
	set type(t) { this._type  = t; }
	set name(name) { this._name = name; }
	getPrereqTechs() {
		return Object.assign({},this._prereqs);
	}

	addBlock(blocktype) {
		if (ShipDesign.SHIPBLOCKNAMES[blocktype]) {
			this[blocktype]++;
		}
		this._computePrereqTechs();
		this._protoShip = null; // Remove the prototype ship
	}
	_computePrereqTechs() {
		this._prereqs = { power: 1, jump: 1, man: 1, size: 1, weapons: 1, defences: 1, computer: 0, arch: 1, planet: 1};
		if (this._type === ShipDesignType.SpaceBase) this._prereqs.planet = 3;
		if (this._type === ShipDesignType.StarBase) this._prereqs.planet = 4;

		this._forEachBlockType(bt => {
			if (this[bt] && ShipDesign.SHIPBLOCKTECHS[bt] && ShipDesign.SHIPBLOCKTLS[bt] > this._prereqs[ShipDesign.SHIPBLOCKTECHS[bt]]) {
				this._prereqs[ShipDesign.SHIPBLOCKTECHS[bt]] = ShipDesign.SHIPBLOCKTLS[bt];
			}
		});

		const size = this.totalSize();
		     if (size>150) this._prereqs.size = 9;
		else if (size>100) this._prereqs.size = 8;
		else if (size>80)  this._prereqs.size = 7;
		else if (size>60)  this._prereqs.size = 6;
		else if (size>40)  this._prereqs.size = 5;
		else if (size>30)  this._prereqs.size = 4;
		else if (size>20)  this._prereqs.size = 3;
		else if (size>10)  this._prereqs.size = 2;
		else this._prereqs.size = 1;
	}
	deleteBlock(blocktype) {
		if (ShipDesign.SHIPBLOCKNAMES[blocktype] && this[blocktype] > 0) {
			this[blocktype]--;
			this._computePrereqTechs();
			this._protoShip = null; // Remove the prototype ship
		}
	}

	_forEachBlockType(f) {
		for(let blocktype in ShipDesign.SHIPBLOCKNAMES) {
			f(blocktype);
		}
	}

	totalCost() {
		if (this.totalSize() === 0) { return 40; /* Just to make the display look right (since we claim that the hull costs 40 PPs after the discount for the (non-existant) bridge)  */ }
		let total = -ShipDesign.SHIPBLOCKCOSTS._b0; // First bridge is free
		this._forEachBlockType(blocktype => {
			if (this[blocktype]) {
				if (ShipDesign.SHIPBLOCKCOSTS[blocktype] == "*") {
					total += this.fighterSpec().cost * this[blocktype];
				} else {
					total += ShipDesign.SHIPBLOCKCOSTS[blocktype] * this[blocktype];
				}
			}
		});

		// Add the hull
		total += this.hullCost();

		if (this._type === ShipDesignType.SpaceBase) {
			total = Math.ceil(total/2);
		} else if (this._type === ShipDesignType.StarBase) {
			total = Math.ceil(total*2/3);
		}
		return total;
	}
	hullCost() {
		const hullcost = [undefined,50,90,125,155,210,250,285,360,420];
		return hullcost[this._prereqs.size];
	}
	totalSize() {
		let size = 0;
		this._forEachBlockType(blocktype => {
			size += ShipDesign.SHIPBLOCKSIZES[blocktype] * this[blocktype];
		});
		return size;
	}
	_getPrototypeShip() {
		if (!this._protoShip) {
			this._protoShip = new Ship(undefined,this,undefined);
		}
	}
	totalEnergyBalance() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		return this._protoShip.getEnergyBalance();
	}
	totalEnergyBalanceJump() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		return this._protoShip.getEnergyBalanceJump();
	}
	getManeuverSpeed() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		return this._protoShip.getManeuverSpeed();
	}
	getCombatSpeed() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		return this._protoShip.getCombatSpeed();
	}
	getAverageHits() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		let hits = this._protoShip.getAverageHitsIgnoringFighters()*(this._protoShip.getCombatSpeed()+1);
		if (this._df > 0) hits += this.fighterSpec().averageHits * this._df;
		return Math.round(hits*1000)/1000;
	}
	getEstimatedDurability() {
		const armour = this._da;
		const bridges = this._b0 + this._b1 + this._b2 + this._b3 + this._b4 + this._b5 + this._b6 + this._b7 + this._b8 + this._b9;
		const dummy = new Ship(undefined,this,undefined);
		const shields = dummy.getTotalShields();
		const invulnerable = this._dv * this.totalSize(); // We'll assume that the ship will take its size as damage while invulnerable
		const nonarmour = this.totalSize() - this._da;

		const minHitsSurvived = bridges+armour+shields+invulnerable;
		const maxHitsSurvived = this.totalSize()+shields+invulnerable;
		const avgHitsSurvived = (nonarmour+1)*(bridges)/(bridges+1) + armour+shields+invulnerable;

		return [minHitsSurvived,Math.floor(avgHitsSurvived*1000)/1000,maxHitsSurvived];
	}
	getMarinesOnBoard() {
		if (this.totalSize() === 0) return 0;
		return this._bb;
	}
	getDredgersOnBoard() {
		if (this.totalSize() === 0) return 0;
		return this._ad;
	}
	getFightersOnBoard() {
		if (this.totalSize() === 0) return 0;
		return this._df;
	}
	getJumpRange() {
		if (this.totalSize() === 0) return 0;
		this._getPrototypeShip();
		return this._protoShip.getJumpRange();
	}
	getBlocks() {
		if (this.totalSize() === 0) return [];
		this._getPrototypeShip();
		return this._protoShip.getBlocks();
	}
	getDesignBlocks() {
		if (this.totalSize() === 0) return [];
		let blocks = [];
		this._forEachBlockType(blocktype => {
			const size = ShipDesign.SHIPBLOCKSIZES[blocktype];
			const fighterSpec = this.fighterSpec();
			const cost = (ShipDesign.SHIPBLOCKCOSTS[blocktype] == "*" ? (fighterSpec ? fighterSpec.cost : 0) : ShipDesign.SHIPBLOCKCOSTS[blocktype]);
			const eps  = ShipDesign.SHIPBLOCKEPS[blocktype];
			const epsj = ShipDesign.SHIPBLOCKEPS_J[blocktype];
			for(let i=0; i<this[blocktype]; i++) {
				blocks.push(new Block(blocktype,size,cost,eps,epsj,fighterSpec));
			}
		});
		return blocks;
	}
	isValid() {
		return this.hasBridge();
	}
	isTechSufficent(techs) {
		for(let tech in this._prereqs) {
			if (this._prereqs[tech] > 1) {
				if (!techs[tech]) return false; // If the prereq tech isn't even listed in the argument techs - that's a fail :)
				if (this._prereqs[tech] > techs[tech]) return false;
			}
		}
		return true;
	}
	numberOfClassBuilt(player) {
		return this._built[player.stylename] ? this._built[player.stylename] : 0;
	}
	hasBridge() {
		return this._b0+this._prereqs.computer > 0;
	}
	setFighterTechs(techs) {
		this._fighterTechs = techs;
	}
	fighterSpec() {
		if (this.totalSize() === 0) return null;
		if (this._df > 0) {
			return ShipDesign.fighterDetailedSpec(this._fighterTechs);
		} else {
			return null;
		}
	}
	getColonyImprovement() {
		if (this.totalSize() === 0) return 0;
		return this._cb;
	}
	hasWeapons() {
		if (this.totalSize() === 0) return false;
		return this._wl || this._wp || this._wa || this._wf || this._wr || this._wc || this._ws || this._wm || this._wt || this._wx || this._wb || this._df || this._dr;
	}
	getRatingColony() {
		if (this.totalSize() === 0) return 0;
		return Math.floor(this.getColonyImprovement() * this.getJumpRange() / this.totalCost() * 10000) / 100;
	}
	getRatingTroops() {
		if (this.totalSize() === 0) return 0;
		return Math.floor(this.getMarinesOnBoard() * this.getJumpRange() / this.totalCost() * 10000) / 100;
	}
	getRatingAttack() {
		if (this.totalSize() === 0) return 0;
		return Math.floor(this.getRatingDefence() * this.getJumpRange() * 100)/100;
	}
	getRatingDefence() {
		if (this.totalSize() === 0) return 0;
		const avg = this.getEstimatedDurability()[1];
		return Math.floor(this.getAverageHits() * avg / this.totalCost() * 10000)/100;
	}
	getRatingProduction() {
		if (this.totalSize() === 0 || this._ad == 0 || (this._type != ShipDesignType.StarBase && this.getJumpRange() == 0)) return 0;
		return Math.floor(this._ad / this.totalCost() * 10000);
	}
	build(player,shipname,startlocation) {
		if (this._built[player.stylename]) {
			this._built[player.stylename]++;
		} else {
			this._built[player.stylename] = 1;
		}
		return new Ship(shipname,this,startlocation);
	}
	toJSON() {
		let data = { _id: this._identifier, _name: this._name, _type: this._type, _built: this._built };
		if (this._df > 0) {
			data._fighterTechs = this._fighterTechs;
		}
		const self = this;
		this._forEachBlockType(function(blocktype){
			if (self[blocktype] > 0) data[blocktype] = self[blocktype];
		});
		if (data._id == data._name) {
			delete data._name; // Avoid duplicate data
		}
		return data;
	}
}

ShipDesign.REGISTRY = {};
ShipDesign.fromJSON = function(json) {
	const name = (json._name ? json._name : json._id); // If the name is missing it was deleted as being the same as the id, so restore it now
	const result = new ShipDesign(name,ShipDesignType.fromJSON(json._type),json._id);
	// Remove the fields resolved by the constructor (only _id absolutely needs to be deleted (as it is (a) a two-character name and (b) different from that used in the actual ShipDesign instance)
	delete json._name; delete json._type; delete json._id;
	// And copy across the remaining fields
	Object.assign(result,json);
	return result;
};
ShipDesign.fighterDetailedSpec = function(tech) {
	let spec = {};
	let damage = 0;
	switch(tech.weapons) {
		case 1: spec.weapon = "_wl"; damage = 1/6; break;
		case 2: spec.weapon = "_wp"; damage = 1/3; break;
		case 3: spec.weapon = "_wa"; damage = 1/2; break;
		case 4:
		case 5: spec.weapon = "_wf"; damage = 1;break;
		case 6:
		case 7: spec.weapon = "_wc"; damage = 2 ;break;
		case 8:
		case 9: spec.weapon = "_wm"; damage = 3; break;
		case 10:
		case 11: spec.weapon = "_wx"; damage = 10; break;
	}
	spec.weaponDesc = ShipDesign.SHIPBLOCKNAMES[spec.weapon];
	const manblock = "_m"+String.fromCharCode(96+tech.man);
	const energyneed = ShipDesign.SHIPBLOCKEPS["_b"+tech.computer]+ShipDesign.SHIPBLOCKEPS[spec.weapon]+ShipDesign.SHIPBLOCKEPS[manblock];
	const powerblocks = Math.ceil((-energyneed)/ShipDesign.SHIPBLOCKEPS["_p"+tech.power]);
	const size = 3+powerblocks;
	const points = 10*tech.man + (tech.man>5 ? 15*(tech.man-6) : 0) + (tech.man == 8 ? 75 : 0);
	spec.speed = Math.floor(points/size)+tech.computer;
	spec.cost = ShipDesign.SHIPBLOCKCOSTS["_b"+tech.computer] - 10;
	spec.cost += ShipDesign.SHIPBLOCKCOSTS[spec.weapon];
	spec.cost += ShipDesign.SHIPBLOCKCOSTS[manblock];
	spec.cost += powerblocks*ShipDesign.SHIPBLOCKCOSTS["_p"+tech.power];
	spec.cost = Math.ceil(spec.cost/2);
	spec.averageHits = (spec.speed+1) * damage;
	return spec;
};

ShipDesign.SHIPBLOCKNAMES = { // Note we use this ordering to order the blocks in the design/ships (invul last as only one needs to be powered per turn)
		_b0: "Bridge", _b1: "Bridge (TL1 Computer)", _b2: "Bridge (TL2 Computer)", _b3: "Bridge (TL3 Computer)", _b4: "Bridge (TL4 Computer)",
		_b5: "Bridge (TL5 Computer)", _b6: "Bridge (TL6 Computer)", _b7: "Bridge (TL7 Computer)", _b8: "Bridge (TL8 Computer)", _b9: "Bridge (TL9 computer)",
		_p1: "Chemical Power Plant", _p2: "Atomic Power Plant", _p3: "Fusion Power Plant", _p4: "Large Fusion Power Plant",
		_p5: "Antimatter Power Plant", _p6: "Large Antimatter Power Plant", _p7: "Ultimate Power Plant",
		_ja: "Jump Drive-A", _jb: "Jump Drive-B", _jc: "Jump Drive-C", _jd: "Jump Drive-D",
		_je: "Jump Drive-E", _jf: "Jump Drive-F", _jg: "Jump Drive-G", _jh: "Jump Drive-H",
		_ma: "Maneuver Drive-A", _mb: "Maneuver Drive-B", _mc: "Maneuver Drive-C", _md: "Maneuver Drive-D",
		_me: "Maneuver Drive-E", _mf: "Maneuver Drive-F", _mg: "Maneuver Drive-G", _mh: "Maneuver Drive-H",
		_wl: "Laser Battery",  _wp: "Phaser Battery", _wa: "Plasma Gun", _wf: "Fusion Gun",
		_wr: "Particle Accelerator Spinal Mount", _wc: "Particle Accelerator", _ws:"Meson Gun Spinal Mount", _wm:"Meson Gun",
		_wt: "Anti-Matter Spinal Mount", _wx: "Anti-Matter Gun", _wb: "Planet Buster",
		_da: "Armour Block",  _dc: "Damage Control",  _ds: "Shield", _df: "Fighter Bay",_di: "Improved Shield Block",
		_dg: "Regenerating Shield Block", _dr: "Reflecting Shield Block",
		_bb: "Marine Barracks",
		_cb: "Colonisation System",
		_an: "Neural Net Block", _ad: "Jump Space Dredger", _at: "Tractor Beam", _as: "Statis Beam", _dv: "Invulnerability Block"
};
ShipDesign.SHIPBLOCKDESCRIPTIONS = { // Note we use this ordering to order the blocks in the design/ships (invul last as only one needs to be powered per turn)
		_b0: "", _b1: "", _b2: "", _b3: "", _b4: "",
		_b5: "", _b6: "", _b7: "", _b8: "", _b9: "",
		_p1: "", _p2: "", _p3: "", _p4: "",
		_p5: "", _p6: "", _p7: "",
		_ja: "10 Jump points per block", _jb: "20 Jump points per block", _jc: "30 jump points per block", _jd: "40 jump points per block",
		_je: "50 Jump points per block", _jf: "75 Jump points per block", _jg: "100 Jump points per block", _jh: "200 Jump points per block",
		_ma: "10 Maneuver Points per block", _mb: "20 Maneuver Points per block", _mc: "30 Maneuver Points per block", _md: "40 Maneuver Points per block",
		_me: "50 Maneuver Points per block", _mf: "75 Maneuver Points per block", _mg: "100 Maneuver Points per block", _mh: "200 Maneuver Points per block",
		_wl: "1 in 6 chance of hitting",  _wp: "1 in 3 chance of hitting", _wa: "1 in 2 chance of hitting", _wf: "1 hit per gun",
		_wr: "10 hits per spinal mount", _wc: "2 hits per particle accelerator", _ws:"20 hits per spinal mount", _wm:"3 hits per gun",
		_wt: "30 hits per spinal mount", _wx: "10 hits per gun", _wb: "Destroys target",
		_da: "First block hit",  _dc: "Repairs one block",  _ds: "Generates 10 shield blocks (which will be hit before the ship is)", _df: "Special",
		_di: "Generates 10 shield blocks (which will be hit before the ship is)",
		_dg: "Generates 10 shield blocks every combat round (which will be hit before the ship is)",
		_dr: "Generates 10 shield blocks (which are damaged before the ship is and reflects the damage as additional hits in the next phase of combat)",
		_bb: "",
		_cb: "Adds one to the level of the colony when used to colonise a level 5 or lower world",
		_an: "Doubles the effectiveness of any computer fitted", _ad:"",_at: "", _as: "", _dv: ""
};
ShipDesign.SHIPBLOCKSHIPONLY = {
		_b0: false, _b1: false, _b2: false, _b3: false, _b4: false, _b5: false, _b6: false, _b7: false, _b8: false, _b9: false,
		_p1: false, _p2: false, _p3: false, _p4: false, _p5: false, _p6: false, _p7: false,
		_ja: true,  _jb: true, _jc: true, _jd: true, _je: true, _jf: true, _jg: true, _jh: true,
		_ma: true,  _mb: true, _mc: true, _md: true, _me: true, _mf: true, _mg: true, _mh: true,
		_wl: false,  _wp: false, _wa: false, _wf: false, _wr:false, _wc: false, _ws:false, _wm: false, _wt: false, _wx: false, _wb:false,
		_da: false,  _dc: false,  _ds: false, _df: false,_di: false, _dg: false, _dr: false, _dv:false,
		_bb: true,
		_cb: true,
		_an: false, _ad: false, _at: false, _as: false
};
ShipDesign.SHIPBLOCKCOSTS = {
		_b0: 10, _b1: 20, _b2: 27, _b3: 35, _b4: 40, _b5: 45, _b6: 50, _b7: 55, _b8: 60, _b9: 65,
		_p1: 10, _p2: 12, _p3: 15, _p4: 67, _p5: 20, _p6: 80, _p7: 25,
		_ja: 5,  _jb: 10, _jc: 15, _jd: 20, _je: 25, _jf: 36, _jg: 45, _jh: 80,
		_ma: 5,  _mb: 10, _mc: 15, _md: 20, _me: 25, _mf: 36, _mg: 45, _mh: 80,
		_wl: 5,  _wp: 10, _wa: 15, _wf: 20, _wr:100, _wc: 25, _ws:150, _wm: 30, _wt: 250, _wx: 40, _wb:2000,
		_da: 5,  _dc: 1,  _ds: 10, _df: "*",_di: 15, _dg: 20, _dr: 25, _dv:100,
		_bb: 11,
		_cb: 30,
		_an: 10, _ad: 250, _at: 10, _as: 50
};
ShipDesign.SHIPBLOCKSIZES = {
		_b0: 1, _b1: 1, _b2: 1, _b3: 1, _b4: 1, _b5: 1, _b6: 1, _b7: 1, _b8: 1, _b9: 1,
		_p1: 1, _p2: 1, _p3: 1, _p4: 5, _p5: 1, _p6: 5, _p7: 1,
		_ja: 1, _jb: 1, _jc: 1, _jd: 1, _je: 1, _jf: 1, _jg: 1, _jh:1,
		_ma: 1, _mb: 1, _mc: 1, _md: 1, _me: 1, _mf: 1, _mg: 1, _mh:1,
		_wl: 1, _wp: 1, _wa: 1, _wf: 1, _wr:8, _wc: 1, _ws:8, _wm:1, _wt: 8, _wx: 1, _wb:50,
		_da: 1, _dc: 1, _ds: 1, _df: 2, _di: 1, _dg: 1, _dr: 1, _dv:1,
		_bb: 1,
		_cb: 6,
		_an: 1, _ad: 5, _at: 1, _as: 1
};
ShipDesign.SHIPBLOCKEPS = {
		_b0: 0, _b1: -1, _b2: -2, _b3: -3, _b4: -5, _b5: -8, _b6: -12, _b7: -16, _b8: -20, _b9: -24,
		_p1: 12, _p2: 15, _p3: 20, _p4: 150, _p5: 40, _p6: 300, _p7: 100,
		_ja: 0,  _jb: 0, _jc: 0, _jd: 0, _je: 0, _jf: 0, _jg: 0, _jh:0,
		_ma: -10, _mb: -18, _mc: -25, _md: -30, _me: -35, _mf: -48, _mg: -60, _mh:-100,
		_wl: -1,  _wp: -2, _wa: -3, _wf: -5, _wr:-40, _wc: -8, _ws: -60, _wm: -10, _wt: -80, _wx: -20, _wb:-1000,
		_da: 0,  _dc: 0,  _ds: 0, _df: 0,_di: 0, _dg: 0, _dr: 0, _dv:"*50",
		_bb: 0,
		_cb: 0,
		_an: 0, _ad: -10, _at: -10, _as: "*50"
};
ShipDesign.SHIPBLOCKEPS_J = {
		_b0: 0, _b1: 0, _b2: 0, _b3: 0, _b4: 0, _b5: 0, _b6: 0, _b7: 0, _b8: 0, _b9: 0,
		_p1: 12, _p2: 15, _p3: 20, _p4: 150, _p5: 40, _p6: 300, _p7: 100,
		_ja: -10, _jb: -18, _jc: -25, _jd: -30, _je: -35, _jf: -48, _jg: -60, _jh:-100,
		_ma: 0, _mb: 0, _mc: 0, _md: 0, _me: 0, _mf: 0, _mg: 0, _mh:0,
		_wl: 0, _wp: 0, _wa: 0, _wf: 0, _wr: 0, _wc: 0, _ws: 0, _wm: 0, _wt: 0, _wx: 0, _wb: 0,
		_da: 0, _dc: 0, _ds: -10, _df: 0,_di: -25, _dg: -10, _dr: -10, _dv:0,
		_bb: 0,
		_cb: 0,
		_an: 0, _ad: 0, _at: 0, _as: 0
};
ShipDesign.SHIPBLOCKTECHS = {
		_b0: null, _b1: "computer", _b2: "computer", _b3: "computer", _b4: "computer", _b5: "computer", _b6: "computer", _b7: "computer", _b8: "computer", _b9: "computer",
		_p1: "power", _p2: "power", _p3: "power", _p4: "power", _p5: "power", _p6: "power", _p7: "power",
		_ja: "jump", _jb: "jump", _jc: "jump", _jd: "jump", _je: "jump", _jf: "jump", _jg: "jump", _jh:"jump",
		_ma: "man", _mb: "man", _mc: "man", _md: "man", _me: "man", _mf: "man", _mg: "man", _mh:"man",
		_wl: "weapons", _wp: "weapons", _wa: "weapons", _wf: "weapons", _wr: "weapons", _wc: "weapons", _ws: "weapons", _wm: "weapons", _wt: "weapons", _wx: "weapons", _wb: "weapons",
		_da: "defences", _dc: "defences", _ds: "defences", _df: "defences",_di: "defences", _dg: "defences", _dr: "defences", _dv:"defences",
		_bb: null,
		_cb: null,
		_an: "arch", _ad: "arch",_at: "arch", _as: "arch"
};
ShipDesign.SHIPBLOCKTLS = {
		_b0: 0, _b1: 1, _b2: 2, _b3: 3, _b4: 4, _b5: 5, _b6: 6, _b7: 7, _b8: 8, _b9: 9,
		_p1: 1, _p2: 2, _p3: 3, _p4: 4, _p5: 5, _p6: 6, _p7: 7,
		_ja: 1, _jb: 2, _jc: 3, _jd: 4, _je: 5, _jf: 6, _jg: 7, _jh:8,
		_ma: 1, _mb: 2, _mc: 3, _md: 4, _me: 5, _mf: 6, _mg: 7, _mh:8,
		_wl: 1, _wp: 2, _wa: 3, _wf: 4, _wr: 5, _wc: 6, _ws: 7, _wm: 8, _wt: 8, _wx: 9, _wb: 10,
		_da: 1, _dc: 2, _ds: 3, _df: 4,_di: 5, _dg: 6, _dr: 7, _dv:8,
		_bb: 0,
		_cb: 0,
		_an: 2, _ad: 3, _at: 4, _as: 5
};
// Needs to happen after we've initialised all those globals above, since we're about to use 'em
ShipDesign.init = function() {
	// Set up the standard ship types (the order will be reversed when the expenditure screen shows them (so colony ship will initially be at top of the list)
	const artifact = new ShipDesign("Artifact",ShipDesignType.Ship,0);
	artifact.addBlock("_b9"); artifact.addBlock("_p7"); artifact.addBlock("_p7"); artifact.addBlock("_jh"); artifact.addBlock("_mh");
	artifact.addBlock("_wx"); artifact.addBlock("_an"); artifact.addBlock("_dv"); artifact.addBlock("_dv"); artifact.addBlock("_dv");

	const exampleSDB = new ShipDesign("TL1 System Defense Boat",ShipDesignType.Ship,1);
	exampleSDB.addBlock("_b1"); exampleSDB.addBlock("_p1"); exampleSDB.addBlock("_p1"); exampleSDB.addBlock("_p1");
	exampleSDB.addBlock("_ma"); exampleSDB.addBlock("_ma"); exampleSDB.addBlock("_ma");
	exampleSDB.addBlock("_wl"); exampleSDB.addBlock("_wl"); exampleSDB.addBlock("_wl");

	const exampleTroopTransport = new ShipDesign("TL1 Troop Transport",ShipDesignType.Ship,2);
	exampleTroopTransport.addBlock("_b0"); exampleTroopTransport.addBlock("_p1"); exampleTroopTransport.addBlock("_ja"); exampleTroopTransport.addBlock("_ma");
	exampleTroopTransport.addBlock("_bb"); exampleTroopTransport.addBlock("_bb"); exampleTroopTransport.addBlock("_bb");
	exampleTroopTransport.addBlock("_bb"); exampleTroopTransport.addBlock("_bb"); exampleTroopTransport.addBlock("_bb");

	const exampleFighter = new ShipDesign("TL1 Picket Boat",ShipDesignType.Ship,3);
	exampleFighter.addBlock("_b1"); exampleFighter.addBlock("_p1"); exampleFighter.addBlock("_p1"); exampleFighter.addBlock("_ja");
	exampleFighter.addBlock("_ma"); exampleFighter.addBlock("_ma"); exampleFighter.addBlock("_wl");
	exampleFighter.addBlock("_wl"); exampleFighter.addBlock("_wl"); exampleFighter.addBlock("_da");

	const exampleScoutShip = new ShipDesign("TL1 Scout", ShipDesignType.Ship,4);
	exampleScoutShip.addBlock("_b0"); exampleScoutShip.addBlock("_p1");
	exampleScoutShip.addBlock("_ja"); exampleScoutShip.addBlock("_ma");

	const exampleColonyShip = new ShipDesign("TL1 Colony Ship",ShipDesignType.Ship,5);
	exampleColonyShip.addBlock("_b0"); exampleColonyShip.addBlock("_p1"); exampleColonyShip.addBlock("_ja"); exampleColonyShip.addBlock("_ma");
	exampleColonyShip.addBlock("_cb");

	const exampleSpaceBase = new ShipDesign("TL1 Space Base",ShipDesignType.SpaceBase,6);
	exampleSpaceBase.addBlock("_b1"); exampleSpaceBase.addBlock("_p1"); exampleSpaceBase.addBlock("_wl"); exampleSpaceBase.addBlock("_wl");
	exampleSpaceBase.addBlock("_wl"); exampleSpaceBase.addBlock("_wl"); exampleSpaceBase.addBlock("_wl");
	exampleSpaceBase.addBlock("_wl"); exampleSpaceBase.addBlock("_wl"); exampleSpaceBase.addBlock("_wl");

	const exampleStarBase = new ShipDesign("TL1 Star Base",ShipDesignType.StarBase,7);
	exampleStarBase.addBlock("_b1"); exampleStarBase.addBlock("_p1"); exampleStarBase.addBlock("_wl"); exampleStarBase.addBlock("_wl");
	exampleStarBase.addBlock("_wl"); exampleStarBase.addBlock("_wl"); exampleStarBase.addBlock("_wl");
	exampleStarBase.addBlock("_wl"); exampleStarBase.addBlock("_wl"); exampleStarBase.addBlock("_wl");
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
	exports.ShipDesignType = ShipDesignType;
	exports.Block = Block;
    exports.ShipDesign = ShipDesign;
}