/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals Game, ShipDesign, TECHS, ALLTECHS */
/* exported Order, ResearchOrder, CarryForwardOrder, ImproveColonyOrder, AddMarinesOrder */
/* exported BuyVehicleOrder, BuyStarBaseOrder, NullOrder, RegisterShipDesign */
class ResearchOrder {
	constructor(tech,amount) {
		if (amount && !isNaN(parseInt(amount))) {
			this._tech = tech;
			this._amount = amount;
		} else {
			throw new Error(`amount: ${amount} is not a number (or is zero) when constructing a ResearchOrder`);
		}
	}
	toString() {
		return "Research "+this._amount+" in "+this._tech.option;
	}
	toJSON() {
		return {type: "RO", tech: this._tech.index, amount: this._amount };
	}
	apply(player,_map,_turn,_s2p) {
		if (player.addResearch(this._tech,this._amount)) {
			const level = player.researchLevel(this._tech);
			// Special case: the artifact ship
			if (this._tech === TECHS.archaeology && level == 6) {
				player.addShip(Game.ARTIFACT_DESIGN.build(player,"Ancient Fighter found on "+player.capital,player.capital));
			}
			const item = document.getElementById(this._tech.domPrefix+level);
			const onlyOneDot = function(s) { return s.endsWith(".") ? s : s+"."; };
			return this._tech.option+" technology has now reached level "+level+". "+onlyOneDot(item.innerText)+(item.title ? " "+onlyOneDot(item.title) : "");
		} else {
			return null;
		}
	}
}

class CarryForwardOrder {
	constructor(amount) {
		this._amount = amount;
	}
	toString() {
		return "carry forward "+this._amount+" PPs";
	}
	toJSON() {
		return {type: "CFO", amount: this._amount };
	}
	apply(player,_map,_turn,_s2p) {
		player.carriedProduction = this._amount;
		return null;
	}
}

class ImproveColonyOrder {
	constructor(colony) {
		this._colony = colony;
	}
	get colony() {
		return this._colony;
	}
	toString() {
		return "Improve colony at "+this._colony;
	}
	toJSON() {
		return {type: "ICO", colony: this._colony };
	}
	apply(_player,map,_turn,_s2p) {
		map.improveColony(this._colony);
		return null;
	}
}

class AddMarinesOrder {
	constructor(colony,companies) {
		this._colony = colony;
		this._companies = companies;
	}
	get colony() {
		return this._colony;
	}
	get companies() {
		return this._companies;
	}
	toString() {
		return "Add "+this._companies+" marine companies to the colony at "+this._colony;
	}
	toJSON() {
		return {type: "AMO", colony: this._colony, companies: this._companies };
	}
	apply(_player,map,_turn,_s2p) {
		map.addMarines(this._colony,this._companies);
		return null;
	}
}

class BuyVehicleOrder {
	constructor(design,name,location) {
		this._type = "BVO";
		this._design = design;
		this._name = name;
		this._location = location;
		this._firstOfDesign = false; // Doesn't need to be saved in the JSON - it's used during the player's order entry
	}
	get design() {
		return this._design;
	}
	get startLocation() {
		return this._location;
	}
	get name() {
		return this._name;
	}
	get firstOfDesign() { return this._firstOfDesign; }
	setFirstOfDesign() { this._firstOfDesign = true; }
	toString() {
		return `Add a new ${this._design.name} called '${this._name}' at ${this._location}`;
	}
	toJSON() {
		return {type: this._type, design: this._design.id, name: this._name, location: this._location };
	}
	apply(player,_map,_turn,_s2p) {
		player.addShip(this._design.build(player,this._name,this._location));
		return null;
	}
}

class BuyStarBaseOrder extends BuyVehicleOrder {
	constructor(design,name,location) {
		super(design,name,location);
		this._type = "BSBO";
	}
	apply(player,map,turn,s2p) {
		// The star base can only be deployed to empty hexes - the star base is discarded if the hex has anything there
		const hex = map.hex(this._location);
		if (hex.owner && hex.owner !== player.stylename) {
			const hexowner = s2p(hex.owner);
			let report = null;
			if (hex.level > 0) {
				report = `Cannot deploy ${this._name} to ${this._location}. It is already occupied by a class ${hex.level} colony. Our Star Base has been destroyed to prevent its capture by ${hexowner.fullname}`;
			} else {
				report = `Cannot deploy ${this._name} to ${this._location}. It is already occupied by another Star Base. Our Star Base has been destroyed to prevent its capture by ${hexowner.fullname}`;
			}
			player.map.setLastScanned(this._location,turn,report); // (Note we update the player's map - not the baseMap that is supplied on the call to apply
			return report;
		} else {
			player.addShip(this._design.build(player,this._name,this._location));
			map.addStarBase(this._location,player.stylename);
			let report = null;
			if (hex.planet && hex.level == 0) report = `${hex.location} has an unoccupied planet.`;
			player.map.setLastScanned(this._location,turn,report); // (Note we update the player's map - not the baseMap that is supplied on the call to apply
			return report;
		}
	}
}

class RegisterShipDesign {
	constructor(design) {
		this._design = design;
	}
	toString() {
		return `Register a new ship design (id=${this._design.id})`;
	}
	toJSON() {
		return {type: "RSD", design: JSON.stringify(this._design)};
	}
	apply(player,_map,_turn,_s2p) {
		player.addShipDesign(this._design);
	}
}

class NullOrder {
	constructor() {
	}
	toString() {
		return "Do Nothing";
	}
	toJSON() {
		return {type: "NO" };
	}
	apply(_player,_map,_turn,_s2p) {
	}
}

var Order = {
	fromJSON: function(json) {
		switch(json.type) {
			case "RO":		return new ResearchOrder(ALLTECHS[json.tech],json.amount);
			case "CFO":		return new CarryForwardOrder(json.amount);
			case "ICO":		return new ImproveColonyOrder(json.colony);
			case "AMO":		return new AddMarinesOrder(json.colony, json.companies);
			case "BVO":
				const shipdesign = ShipDesign.REGISTRY[json.design];
				if (shipdesign)
					return new BuyVehicleOrder(shipdesign,json.name,json.location);
				else
					throw new Error(`Couldn't find ${json.design} in the design registry`);
				break;
			case "BSBO":
				const basedesign = ShipDesign.REGISTRY[json.design];
				if (basedesign)
					return new BuyStarBaseOrder(basedesign,json.name,json.location);
				else
					throw new Error(`Couldn't find ${json.design} in the design registry`);
				break;
			case "RSD": 	return new RegisterShipDesign(ShipDesign.fromJSON(JSON.parse(json.design)));
			case "NO":
			default:		return new NullOrder();
		}
	}
};