//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals CommutatorState, SegmentDisplay, Capture */

// Do a render when the canvas has resized
function RomseyBoxView(window,id,model) {
    this._model = model;
    this._id = id;
    const setResizeHandler = function(callback, timeout) {
        var timer_id ;
        window.addEventListener("resize", function() {
            if(timer_id) {
                clearTimeout(timer_id);
                timer_id = null;
            }
            timer_id = setTimeout(function() {
                timer_id = null;
                callback();
            }, timeout);
        });
    };
    
    // resize handler
    const onResize = function(view) {
        return function() {
           Capture.exception(view._id,view.renderDiagram,view);
           Capture.exception(view._id,view.renderClock,view);
        };
    };
    setResizeHandler(onResize(this), 100);
}
RomseyBoxView.prototype = {
    render: function() {
        for(var i=1; i<=23; i++) {
            const signal = document.getElementById("djvRomseySignalBox-"+this._id+"-lever-"+i);
            const lever = this._model.getLever(i);

            signal.classList.remove("leveron","leveroff");
            signal.classList.add(lever.isPulled() ? "leveron" : "leveroff");
            
            signal.title = lever.getLockCause() ? lever.getLockCause() : "";
        }
        Capture.exception(this._id,this.renderDiagram,this);
        Capture.exception(this._id,this.renderCommutators,this);
    },
    renderDiagram: function() {
        const diagramwidth = 560;
        const diagramheight = 160;
        const rads = function(x) { return Math.PI*x/180; };
        const x = function(z) { return z; };
        const y = function(z) { return diagramheight-z; };
        const map = document.getElementById("djvRomseySignalBox-"+this._id+"-map");
        const canvas = document.getElementById("djvRomseySignalBox-"+this._id+"-diagram");
        canvas.style.width = '100%';
        canvas.style.height = '100%';
        canvas.height = map.offsetHeight; // Note: as a side effect this actually clears the canvas too!
        canvas.width = map.offsetWidth;
        const c = canvas.getContext("2d");
        const scaleX = canvas.width / diagramwidth;
        const scaleY = canvas.height / diagramheight;
        
        c.scale(scaleX,scaleY); // Note: we avoid a negative scale to avoid reflecting text as well
        c.font = "4pt sans-serif";

        // Key dimensions
        const siline = 25;
        const baline = 32;
        const upline = 47;
        const dnline = 62;
        const brleng = diagramheight - 12 - dnline;
        const kimbri = diagramwidth - 8;

        const eastrt = 8;
        const pntsea = 44;
        const pnts11 = 230;
        const pnts08 = 320;
        const pts10u = 332;
        const pts12u = 347;
        const ptsb4u = 447;
        const ptsb2u = 480;
        const pnts07 = 326;
        const pts10d = pts10u + (dnline - upline);
        const ptsb2d = ptsb2u + (dnline - upline);
        const pts12s = pts12u + (upline - baline);
        const ptsb4s = ptsb4u + (upline - baline);
        
        const dnbrsx = pnts07 - brleng;
        const dnbrsy = dnline + brleng;
        const upbrsx = pnts08-brleng-(dnline-upline)/2;
        const upbrsy = upline+brleng+(dnline-upline)/2;
        
        const drawPosition = function(xpos,ypos,position,branch,suppressLabel) {
            c.fillStyle = (position.getOccupier() !== null ? "red" : "darkred");
            c.beginPath();
            if (branch) {
                c.moveTo(x(xpos-3),y(ypos));
                c.arc(x(xpos+1.5),y(ypos-1.5),2.12,rads(135),rads(315),true);
                c.arc(x(xpos-1.5),y(ypos+1.5),2.12,rads(315),rads(135),true);
            } else {
                c.moveTo(x(xpos+2),y(ypos+2));
                c.arc(x(xpos-2),y(ypos),2,rads(270),rads(90),true);
                c.arc(x(xpos+2),y(ypos),2,rads(90),rads(270),true);
            }
            c.stroke();
            c.fill();
            if (!!!suppressLabel) {
                c.fillStyle = "black";
                c.textBaseline = "bottom"; c.textAlign = "center";
                c.fillText(position.getName(),x(xpos),y(ypos+3));
            }
        };

        // Eastleigh lines
        c.moveTo(x(eastrt),y(baline));
        c.lineTo(x(pntsea-(upline-baline)),y(baline));
        c.lineTo(x(pntsea-2),y(upline-2));
        c.moveTo(x(eastrt),y(upline));
        c.lineTo(x(pntsea-3),y(upline));

        // Eastleigh point
        if (!!!this._model.getEastleigh().getPointPulled()) {
            c.lineTo(x(pntsea-1),y(upline+2));
            c.moveTo(x(pntsea-2),y(upline-2));
        }
        c.lineTo(x(pntsea),y(upline));

        // Chandlers Ford line
        c.lineTo(x(pnts11-2),y(upline));

        // Point 11
        if (!!!this._model.getLever(11).isPulled()) {
            c.lineTo(x(pnts11),y(upline+2));
            c.moveTo(x(pnts11),y(upline-2));
            c.lineTo(x(pnts11+2),y(upline));
        }

        // Up main line to points 8
        c.lineTo(x(pnts08-3),y(upline));
        
        // Point 8 - up main line
        if (this._model.getLever(8).isPulled()) {
            c.moveTo(x(pnts08),y(upline));
        }
        
        // Up main line to points 10
        c.lineTo(x(pts10u),y(upline));
        
        // Points 10 - up main line
        if (this._model.getLever(10).isPulled()) {
            c.moveTo(x(pts10u+3),y(upline));
        }
        
        // Up main line to point 12
        c.lineTo(x(pts12u),y(upline));
        
        // Points 12 up main line
        if (this._model.getLever(12).isPulled()) {
            c.moveTo(x(pts12u+3),y(upline));
        }
        
        // Up main line to ④
        c.lineTo(x(ptsb4u),y(upline));
        
        // Points ④
        if (this._model.getLever(14).isPulled()) {
            c.moveTo(x(ptsb4u+3),y(upline));
        }
        
        // Up main line to ②
        c.lineTo(x(ptsb2u),y(upline));
        
        // Points ②
        if (this._model.getLever(15).isPulled()) {
            c.moveTo(x(ptsb2u+3),y(upline));
        }
        
        // Up main line to map edge
        c.lineTo(x(kimbri),y(upline));
        
        // Down main line from points 11 to points 7
        c.moveTo(x(pnts11),y(upline+2));
        c.lineTo(x(pnts11+(dnline-upline)-2),y(dnline));
        c.lineTo(x(pnts07-3),y(dnline));

        // Point 7
        if (!!!this._model.getLever(7).isPulled()) {
            c.lineTo(x(pnts07-1),y(dnline-2));
            c.moveTo(x(pnts07),y(dnline));
        }
        
        // Down line to points 10
        c.lineTo(x(pts10d-3),y(dnline));
        
        // Points 10
        if (this._model.getLever(10).isPulled()) {
            c.moveTo(x(pts10d),y(dnline));
        }
        
        // Down line to ②
        c.lineTo(x(ptsb2d-3),y(dnline));
        
        // Points ②
        if (this._model.getLever(15).isPulled()) {
            c.moveTo(x(ptsb2d),y(dnline));
        }
        
        // Down main line to map edge
        c.lineTo(x(kimbri),y(dnline));

        // Up Branch line to point 8
        c.moveTo(x(upbrsx),y(upbrsy));
        c.lineTo(x(pnts08-3),y(upline+3));
        
        // point 8
        if (this._model.getLever(8).isPulled()) {
            c.lineTo(x(pnts08),y(upline));
        }
        
        // Down Branch line to point 7
        c.moveTo(x(dnbrsx),y(dnbrsy));
        c.lineTo(x(pnts07-3),y(dnline+3));

        // point 7
        if (!!!this._model.getLever(7).isPulled()) {
            c.lineTo(x(pnts07),y(dnline));
        }
        
        // Cross over points 10
        if (this._model.getLever(10).isPulled()) {
            c.moveTo(x(pts10u),y(upline));
            c.lineTo(x(pts10d),y(dnline));
        } else {
            c.moveTo(x(pts10u+2),y(upline+2));
            c.lineTo(x(pts10d-2),y(dnline-2));
        }
        
        // Cross over points ②
        if (this._model.getLever(15).isPulled()) {
            c.moveTo(x(ptsb2u),y(upline));
            c.lineTo(x(ptsb2d),y(dnline));
        } else {
            c.moveTo(x(ptsb2u+2),y(upline+2));
            c.lineTo(x(ptsb2d-2),y(dnline-2));
        }
        
        // Up Siding No.1
        if (this._model.getLever(12).isPulled()) {
            c.moveTo(x(pts12u),y(upline));
        } else {
            c.moveTo(x(pts12u+3),y(upline-3));
        }
        c.lineTo(x(pts12s-2),y(baline+2));
        if (!!!this._model.getLever(12).isPulled()) {
            c.lineTo(x(pts12s),y(baline+2));
            c.moveTo(x(pts12s-2),y(baline));
        }
        c.lineTo(x(pts12s),y(baline));
        c.lineTo(x(pts12s+(baline-siline)),y(siline));
        c.lineTo(x(pts12s+(baline-siline)+16),y(siline));

        // Up Siding No.2
        if (this._model.getLever(14).isPulled()) {
            c.moveTo(x(ptsb4u),y(upline));
        } else {
            c.moveTo(x(ptsb4u+2),y(upline-2));
        }
        c.lineTo(x(ptsb4s+2),y(baline-2));
        if (!!!this._model.getLever(14).isPulled()) {
            c.moveTo(x(ptsb4s),y(baline-4));
        }
        c.lineTo(x(ptsb4s+4),y(baline-4));
        c.lineTo(x(ptsb4s+36),y(baline-4));
        // Draw the lines
        c.stroke();
        
        // Positions not attached to signals (which will be drawn with the signals
        drawPosition(pnts08-10,upline,this._model.getPosition("C"),false);
        drawPosition((ptsb4u+ptsb2u)/2,upline,this._model.getPosition("B"),false);
        drawPosition(pnts08-(dnline-upline),dnline,this._model.getPosition("J"),false);
        drawPosition(pnts07+10,dnline,this._model.getPosition("K"),false);
        
        // Point lock
        c.lineWidth = 0.5;
        if (this._model.getLever(9).isPulled()) {
            c.beginPath();
            c.moveTo(x(pnts08+3),y(upline+1.5));
            c.lineTo(x(pnts08+3),y(upline-2.5));
            c.moveTo(x(pnts08+2),y(upline-1.5));
            c.lineTo(x(pnts08+4),y(upline-1.5));
            c.stroke();
        }
        
        // Point names
        c.fillStyle = "#000";
        c.textBaseline = "top"; c.textAlign = "start";
        c.fillText("11",x(pnts11-2),y(upline-3));
        c.fillText("9",x(pnts08),y(upline-0.5));
        c.fillText("10",x(pts10u-4),y(upline-1));
        c.fillText("②",x(ptsb2d-1),y(dnline-1));
        c.textAlign = "end";
        c.fillText("12",x(pts12u),y(upline-1));
        c.fillText("④",x(ptsb4u-1),y(upline-1));
        c.textBaseline = "bottom"; c.textAlign = "start";
        c.fillText("8",x(pnts08),y(upline+1));
        c.fillText("7",x(pnts07),y(dnline+1));
        c.fillText("10",x(pts10d),y(dnline+1));
        c.fillText("④",x(ptsb4s),y(baline-3));
        c.textAlign = "end";
        c.fillText("②",x(ptsb2u-1),y(upline+1));
        c.textBaseline = "middle";
        c.fillText("12",x(pts12s-3),y(baline));
        
        // Siding Names
        c.textBaseline = "top"; c.textAlign = "start";
        c.fillText("Up Siding No. 1",x(pts12s),y(siline-1));
        c.fillText("Up Siding No. 2",x(ptsb4s),y(baline-5));
        
        // Platforms
        c.fillStyle = "#888";
        c.beginPath();
        c.rect(x(pntsea+52),y(upline-4),26,7);
        c.rect(x(pts12u+44),y(upline-4),36,7);
        c.rect(x(pts10d+57),y(dnline+11),36,7);
        c.stroke();
        c.fill();
        drawPosition(pts10d+20,dnline,this._model.getPosition("L"),false);
        drawPosition(pts10d+80,dnline,this._model.getPosition("P2"),false,true);
        drawPosition(pts12u+60,upline,this._model.getPosition("P1"),false,true);
        c.textBaseline = "top"; c.textAlign = "center";
        c.fillStyle = "#000";
        c.fillText("Chandlers Ford",x(pntsea+65),y(upline-12));
        c.textBaseline = "bottom";
        c.fillText("Romsey Stn",x(pts10d+75),y(dnline+12));
        
        // Level Crossings
        c.beginPath();
        c.moveTo(x(pnts11-101),y(upline+12));
        c.lineTo(x(pnts11-101),y(upline-12));
        c.moveTo(x(pnts11-93),y(upline+12));
        c.lineTo(x(pnts11-93),y(upline-12));
        c.moveTo(x(pnts11-101),y(upline+2));
        c.arc(x(pnts11-101),y(upline+2),8,rads(0),rads(270),true);
        c.moveTo(x(pnts11-93),y(upline-2));
        c.arc(x(pnts11-93),y(upline-2),8,rads(180),rads(90),true);
        c.stroke();
        c.fillStyle = "#000";
        c.textAlign = "center";
        c.textBaseline = "top";
        c.fillText("crossing",x(pnts11-97),y(upline-26));
        c.textBaseline = "bottom";
        c.fillText("Crampmoor",x(pnts11-97),y(upline-25));
        c.beginPath();
        c.moveTo(x(pnts11-54),y(upline+12));
        c.lineTo(x(pnts11-54),y(upline-12));
        c.moveTo(x(pnts11-46),y(upline+12));
        c.lineTo(x(pnts11-46),y(upline-12));
        c.moveTo(x(pnts11-53),y(upline+7));
        c.lineTo(x(pnts11-49),y(upline+5));
        c.lineTo(x(pnts11-53),y(upline+3));
        c.moveTo(x(pnts11-47),y(upline-7));
        c.lineTo(x(pnts11-51),y(upline-5));
        c.lineTo(x(pnts11-47),y(upline-3));
        c.stroke();
        c.beginPath();
        c.rect(x(pnts11-55),y(upline+7),2,4);
        c.rect(x(pnts11-47),y(upline-3),2,4);
        c.fill();
        c.textBaseline = "top";
        c.fillText("crossing (AHB)",x(pnts11-50),y(upline-26));
        c.textBaseline = "bottom";
        c.fillText("Halterworth",x(pnts11-50),y(upline-25));

        // Signal Box
        c.lineWidth = 0.8;
        c.beginPath();
        c.rect(x(pnts08-5),y(upline-7),16,9);
        c.moveTo(x(pnts08-2),y(upline-9));
        c.lineTo(x(pnts08+6),y(upline-9));
        c.stroke();
        c.fillStyle = "#000";
        c.beginPath();
        c.arc(x(pnts08+3),y(upline-13),1.5,0,rads(360),false);
        c.fill();
        
        // Ground frame
        c.lineWidth = 0.5;
        c.beginPath();
        c.rect(x(ptsb4s-16),y(baline-5),10,6);
        c.moveTo(x(ptsb4s-14),y(baline-6)); c.lineTo(x(ptsb4s-14),y(baline-10));
        c.moveTo(x(ptsb4s-12),y(baline-6)); c.lineTo(x(ptsb4s-12),y(baline-10));
        c.moveTo(x(ptsb4s-10),y(baline-6)); c.lineTo(x(ptsb4s-10),y(baline-10));
        c.moveTo(x(ptsb4s- 8),y(baline-6)); c.lineTo(x(ptsb4s- 8),y(baline-10));
        c.stroke();
        
        // Annotations
        c.beginPath();
        c.moveTo(x(eastrt),y(baline+4));
        c.lineTo(x(eastrt+2),y(baline+2));
        c.moveTo(x(eastrt),y(baline+4));
        c.lineTo(x(eastrt+2),y(baline+6));
        c.moveTo(x(eastrt),y(baline+4));
        c.lineTo(x(eastrt+20),y(baline+4));
        c.stroke();
        c.textBaseline = "bottom"; c.textAlign = "start";
        c.fillText("EASTLEIGH",x(eastrt),y(baline+8));
        c.beginPath();
        c.moveTo(x(pntsea+52),y(upline+4));
        c.lineTo(x(pntsea+54),y(upline+6));
        c.moveTo(x(pntsea+52),y(upline+4));
        c.lineTo(x(pntsea+54),y(upline+2));
        c.moveTo(x(pntsea+52),y(upline+4));
        c.lineTo(x(pntsea+62),y(upline+4));
        c.stroke();
        c.textBaseline = "bottom"; c.textAlign = "center";
        c.fillText("Up",x(pntsea+57),y(upline+7));
        c.beginPath();
        c.moveTo(x(pntsea+78),y(upline+4));
        c.lineTo(x(pntsea+76),y(upline+6));
        c.moveTo(x(pntsea+78),y(upline+4));
        c.lineTo(x(pntsea+76),y(upline+2));
        c.moveTo(x(pntsea+78),y(upline+4));
        c.lineTo(x(pntsea+68),y(upline+4));
        c.stroke();
        c.textBaseline = "bottom"; c.textAlign = "center";
        c.fillText("Down",x(pntsea+73),y(upline+7));
        c.beginPath();
        c.moveTo(x(pntsea+90),y(upline+20));
        c.lineTo(x(pntsea+88),y(upline+22));
        c.moveTo(x(pntsea+90),y(upline+20));
        c.lineTo(x(pntsea+88),y(upline+18));
        c.moveTo(x(pntsea+90),y(upline+20));
        c.lineTo(x(pntsea+60),y(upline+20));
        c.stroke();
        c.textBaseline = "bottom"; c.textAlign = "end";
        c.fillText("6 - Direction Lever",x(pntsea+90),y(upline+23));
        c.beginPath();
        c.moveTo(x(pts10d+75),y(dnline-5));
        c.lineTo(x(pts10d+73),y(dnline-3));
        c.moveTo(x(pts10d+75),y(dnline-5));
        c.lineTo(x(pts10d+73),y(dnline-7));
        c.moveTo(x(pts10d+75),y(dnline-5));
        c.lineTo(x(pts10d+65),y(dnline-5));
        c.stroke();
        c.textBaseline = "middle"; c.textAlign = "end";
        c.fillText("Down main",x(pts10d+63),y(dnline-5));
        c.beginPath();
        c.moveTo(x(pts12u+40),y(upline+4));
        c.lineTo(x(pts12u+42),y(upline+6));
        c.moveTo(x(pts12u+40),y(upline+4));
        c.lineTo(x(pts12u+42),y(upline+2));
        c.moveTo(x(pts12u+40),y(upline+4));
        c.lineTo(x(pts12u+50),y(upline+4));
        c.stroke();
        c.textBaseline = "middle"; c.textAlign = "start";
        c.fillText("Up main",x(pts12u+52),y(upline+4));
        // Annotations
        c.beginPath();
        c.moveTo(x(kimbri),y(dnline-4));
        c.lineTo(x(kimbri-2),y(dnline-2));
        c.moveTo(x(kimbri),y(dnline-4));
        c.lineTo(x(kimbri-2),y(dnline-6));
        c.moveTo(x(kimbri),y(dnline-4));
        c.lineTo(x(kimbri-20),y(dnline-4));
        c.stroke();
        c.textBaseline = "top"; c.textAlign = "end";
        c.fillText("KIMBRIDGE",x(kimbri),y(dnline-8));
        c.beginPath();
        c.moveTo(x(pnts07-37),y(dnline+31));
        c.lineTo(x(pnts07-37),y(dnline+32.5));
        c.moveTo(x(pnts07-37),y(dnline+31));
        c.lineTo(x(pnts07-38.5),y(dnline+31));
        c.moveTo(x(pnts07-37),y(dnline+31));
        c.lineTo(x(pnts07-44),y(dnline+38));
        c.stroke();
        c.save(); // So we can restore the transform
        c.translate(x(pnts07-45),y(dnline+39));
        c.rotate(rads(45)); // Rotate 45 degress clockwise around the origin
        c.textBaseline = "middle"; c.textAlign = "end";
        c.fillText("Down branch",0,0); // Positioning applied by the translate call
        c.restore();
        c.beginPath();
        c.moveTo(x(upbrsx+19),y(upbrsy-13));
        c.lineTo(x(upbrsx+19),y(upbrsy-14.5));
        c.moveTo(x(upbrsx+19),y(upbrsy-13));
        c.lineTo(x(upbrsx+20.5),y(upbrsy-13));
        c.moveTo(x(upbrsx+19),y(upbrsy-13));
        c.lineTo(x(upbrsx+26),y(upbrsy-20));
        c.stroke();
        c.save(); // So we can restore the transform
        c.translate(x(upbrsx+27),y(upbrsy-21));
        c.rotate(rads(45)); // Rotate 45 degress clockwise around the origin
        c.textBaseline = "middle"; c.textAlign = "start";
        c.fillText("Up branch",0,0); // Positioning applied by the translate call
        c.restore();
        c.save(); // So we can restore the transform
        c.translate(x(upbrsx+24),y(upbrsy-16));
        c.rotate(rads(45)); // Rotate 45 degress clockwise around the origin
        c.textBaseline = "bottom"; c.textAlign = "end";
        c.fillText("REDBRIDGE",0,0); // Positioning applied by the translate call
        c.restore();

        const aspectSignal = function(up,name,aspects,remote,xpos,ypos,state,next,element) {
            const xmult = (up ? -1 : 1);
            const ymult = (up ? -1 : 1);
            const red = "#f00";
            const darkred = "#8b0000";
            const yellow = "#ff0";
            const darkyellow = "#b8860b";
            const green = "#0c0";
            const darkgreen = "#006400";
            
            // decide on the aspects
            var lights = [];
            if (remote) {
                if (state) {
                    if (aspects === 2) {
                        lights.push(darkyellow); lights.push(green);
                        if (element) { element.className = "twolightdistant green"; }
                    } else {
                        if (next) {
                            lights.push(darkyellow); lights.push(green); lights.push(darkyellow);
                            if (element) { element.className = "threelightdistant green"; }
                        } else {
                            lights.push(yellow); lights.push(darkgreen); lights.push(yellow);
                            if (element) { element.className = "threelightdistant twoyellow"; }
                        }
                    }
                } else {
                    // remote and not state
                    lights.push(yellow);
                    if (aspects === 3) {
                        lights.push(darkgreen); lights.push(darkyellow);
                        if (element) { element.className = "threelightdistant yellow"; }
                    } else {
                        lights.push(darkgreen);
                        if (element) { element.className = "twolightdistant yellow"; }
                    }
                }
            } else {
                if (state) {
                    // not remote and state
                    lights.push(darkred);
                    if (aspects === 3) {
                        if (next) {
                            lights.push(darkgreen); lights.push(green);
                            if (element) { element.className = "threelight green"; }
                        } else {
                            lights.push(yellow); lights.push(darkgreen);
                            if (element) { element.className = "threelight yellow"; }
                        }
                     } else {
                         lights.push(green);
                         if (element) { element.className = "twolight green"; }
                     }
                } else {
                    // not remote and not state
                    lights.push(red);
                    if (aspects === 3) {
                        lights.push(darkyellow);
                        if (element) { element.className = "threelight red"; }
                    } else {
                        if (element) { element.className = "twolight red"; }
                    }
                    lights.push(darkgreen);
                }
            }
            // The arrow
            c.lineWidth = 0.5;
            c.beginPath();
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos-2),y(ypos+2*ymult));
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos+2),y(ypos+2*ymult));
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos),y(ypos+9*ymult));
            c.moveTo(x(xpos),y(ypos+6*ymult));
            c.lineTo(x(xpos+8*xmult),y(ypos+6*ymult));
            c.stroke();
            
            // Now draw the aspects 
            var cx = xpos+8*xmult+1.5, cy = ypos+6*ymult;
            lights.forEach(function(color){
                c.beginPath();
                c.fillStyle = color;
                c.arc(x(cx),y(cy),1.5,0,rads(360),false);
                cx += 3*xmult;
                c.fill();
            });
            c.fillStyle = "#000";
            c.textBaseline = (up ? "top" : "bottom"); c.textAlign = "start";
            c.fillText(name,x(xpos+(8+(3*lights.length)/2)*xmult),y(ypos+9*ymult));
        };
        
        aspectSignal(true,"ZW30",3,false,pntsea+16,upline,this._model.getEastleigh().getZw30(),this._model.getEastleigh().getZw30Next());
        drawPosition(pntsea+26,upline,this._model.getPosition("PC"),false);
        aspectSignal(true,"ZW30R",2,true,pntsea+36,upline,this._model.getEastleigh().getZw30(),this._model.getEastleigh().getZw30Next());
        drawPosition(pnts11-66,upline,this._model.getPosition("PD"),false);
        aspectSignal(false,"1R",3,true,pnts11-89,upline,this._model.getLever(1).isPulled(),this._model.getLever(2).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-1R"));
        drawPosition(pnts11-37,upline,this._model.getPosition("PE"),false);
        drawPosition(pnts11-24,upline,this._model.getPosition("PF"),false);
        aspectSignal(false,"1",3,false,pnts11-16,upline,this._model.getLever(1).isPulled(),this._model.getLever(2).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-1"));
        aspectSignal(true,"17",2,false,pnts11+30,upline,this._model.getLever(17).isPulled(),undefined,
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-17"));
        drawPosition(pnts11+14,upline,this._model.getPosition("PG"),false);
        drawPosition(pnts11+50,upline,this._model.getPosition("UZ"),false);
        aspectSignal(false,"2",3,false,pnts07-(dnline-upline)-44,dnline,this._model.getLever(2).isPulled(),this._model.getLever(3).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-2"));
        drawPosition(pnts07-(dnline-upline)-54,dnline,this._model.getPosition("PH"),false);
        aspectSignal(true,"20R",2,true,kimbri-20,upline,this._model.getLever(20).isPulled(),undefined);
        
        const banner = function(up,name,xpos,ypos,state) {
            const xmult = (up ? -1 : 1);
            const ymult = (up ? -1 : 1);

            // The arrow
            c.lineWidth = 0.5;
            c.beginPath();
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos-2),y(ypos+2*ymult));
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos+2),y(ypos+2*ymult));
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos),y(ypos+9*ymult));
            c.moveTo(x(xpos),y(ypos+6*ymult));
            c.lineTo(x(xpos+8*xmult),y(ypos+6*ymult));
            // The roundel (and base)
            c.arc(x(xpos+10.5*xmult),y(ypos+6*ymult),2.5,rads(0),rads(360),false);
            c.moveTo(x(xpos+8.75*xmult),y(ypos+4.25*ymult));
            c.lineTo(x(xpos+8*xmult),y(ypos+3.5*ymult));
            c.lineTo(x(xpos+8*xmult),y(ypos+8.5*ymult));
            c.lineTo(x(xpos+8.75*xmult),y(ypos+7.75*ymult));
            c.stroke();

            c.fillStyle = "black";
            c.beginPath();
            if (state) {
                c.moveTo(x(xpos+(461/52)*xmult),y(ypos+(347/52)*ymult));
                c.lineTo(x(xpos+(511/52)*xmult),y(ypos+(397/52)*ymult));
                c.lineTo(x(xpos+(631/52)*xmult),y(ypos+(277/52)*ymult));
                c.lineTo(x(xpos+(581/52)*xmult),y(ypos+(227/52)*ymult));
                c.lineTo(x(xpos+(461/52)*xmult),y(ypos+(347/52)*ymult));
            } else {
                const horz = 25/26;
                const vert = 30/13;
                c.moveTo(x(xpos+(10.5-horz)*xmult),y(ypos+(6+vert)*ymult));
                c.lineTo(x(xpos+(10.5-horz)*xmult),y(ypos+(6-vert)*ymult));
                c.lineTo(x(xpos+(10.5+horz)*xmult),y(ypos+(6-vert)*ymult));
                c.lineTo(x(xpos+(10.5+horz)*xmult),y(ypos+(6+vert)*ymult));
                c.lineTo(x(xpos+(10.5-horz)*xmult),y(ypos+(6+vert)*ymult));
            }
            c.fill();

            c.fillStyle = "#000";
            c.textBaseline = (up ? "top" : "bottom"); c.textAlign = "start";
            c.fillText(name,x(xpos+12*xmult),y(ypos+9*ymult));
        }; 
        
        banner(true,"17BR",pnts11+70,upline,this._model.getLever(17).isPulled());
        
        const semaphore = function(view,up,branch,name,remote,extra,xpos,ypos,state,element) {
            var cx, cy;
            if (branch) {
                if (up) {
                    cx = function(x,y) { return -0.7*x - 0.7*y; };
                    cy = function(x,y) { return  0.7*x - 0.7*y; };
                } else {
                    cx = function(x,y) { return  0.7*x + 0.7*y; };
                    cy = function(x,y) { return -0.7*x + 0.7*y; };
                }
            } else {
                if (up) {
                    cx = function(x,_y) { return -x; };
                    cy = function(_x,y) { return -y; };
                } else {
                    cx = function(x,_y) { return x; };
                    cy = function(_x,y) { return y; };
                }
            }
            
            const length = (extra ? 18 : 12);
            const width = (extra ? (up ? 11 : 6) : 6);

            // Start with the post
            c.beginPath();
            c.moveTo(x(xpos)                  ,y(ypos));
            c.lineTo(x(xpos+cx(-2,2))         ,y(ypos+cy(-2,2)));
            c.moveTo(x(xpos)                  ,y(ypos));
            c.lineTo(x(xpos+cx(2,2))          ,y(ypos+cy(2,2)));
            c.moveTo(x(xpos)                  ,y(ypos));
            c.lineTo(x(xpos+cx(0,width+3))    ,y(ypos+cy(0,width+3)));
            c.moveTo(x(xpos+cx(0,width))      ,y(ypos+cy(0,width)));
            c.lineTo(x(xpos+cx(length,width)) ,y(ypos+cy(length,width)));
            c.stroke();
            
            // then the arm itself
            if (remote) {
                c.fillStyle = "yellow";
            } else {
                c.fillStyle = "red";
            }
            c.beginPath();
            c.moveTo(x(xpos+cx(length-3,width)),y(ypos+cy(length-3,width)));
            if (state) {
                c.lineTo(x(xpos+cx(length-0.75,width+2.25)),y(ypos+cy(length-0.75,width+2.25)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-0.75,width+0.75)),y(ypos+cy(length-0.75,width+0.75)));
                }
                c.lineTo(x(xpos+cx(length+0.75,width+0.75)),y(ypos+cy(length+0.75,width+0.75)));
                c.lineTo(x(xpos+cx(length-1.5,width-1.5)),y(ypos+cy(length-1.5,width-1.5)));
                c.lineTo(x(xpos+cx(length-3,width)),y(ypos+cy(length-3,width)));
                c.moveTo(x(xpos+cx(length,width+3)),y(ypos+cy(length,width+3)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length,width+1.5)),y(ypos+cy(length,width+1.5)));
                }
                c.lineTo(x(xpos+cx(length+1.5,width+1.5)),y(ypos+cy(length+1.5,width+1.5)));
                c.lineTo(x(xpos+cx(length+2.25,width+2.25)),y(ypos+cy(length+2.25,width+2.25)));
                c.lineTo(x(xpos+cx(length+0.75,width+3.75)),y(ypos+cy(length+0.75,width+3.75)));
                c.lineTo(x(xpos+cx(length,width+3)),y(ypos+cy(length,width+3)));
            } else {
                c.lineTo(x(xpos+cx(length-3,width+3)),y(ypos+cy(length-3,width+3)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-2,width+2)),y(ypos+cy(length-2,width+2)));
                }
                c.lineTo(x(xpos+cx(length-1,width+3)),y(ypos+cy(length-1,width+3)));
                c.lineTo(x(xpos+cx(length-1,width)),y(ypos+cy(length-1,width)));
                c.lineTo(x(xpos+cx(length-3,width)),y(ypos+cy(length-3,width)));
                c.moveTo(x(xpos+cx(length-3,width+4)),y(ypos+cy(length-3,width+4)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-2,width+3)),y(ypos+cy(length-2,width+3)));
                }
                c.lineTo(x(xpos+cx(length-1,width+4)),y(ypos+cy(length-1,width+4)));
                c.lineTo(x(xpos+cx(length-1,width+5)),y(ypos+cy(length-1,width+5)));
                c.lineTo(x(xpos+cx(length-3,width+5)),y(ypos+cy(length-3,width+5)));
                c.lineTo(x(xpos+cx(length-3,width+4)),y(ypos+cy(length-3,width+4)));
            }
            c.fill();
            if (remote) {
                c.fillStyle = "black";
            } else {
                c.fillStyle = "white";
            }
            c.beginPath();
            if (state) {
                c.moveTo(x(xpos+cx(length-0.75,width+2.25)),y(ypos+cy(length-0.75,width+2.25)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-0.75,width+0.75)),y(ypos+cy(length-0.75,width+0.75)));
                }
                c.lineTo(x(xpos+cx(length+0.75,width+0.75)),y(ypos+cy(length+0.75,width+0.75)));
                c.lineTo(x(xpos+cx(length+1.5,width+1.5)),y(ypos+cy(length+1.5,width+1.5)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length,width+1.5)),y(ypos+cy(length,width+1.5)));
                }
                c.lineTo(x(xpos+cx(length,width+3)),y(ypos+cy(length,width+3)));
                c.lineTo(x(xpos+cx(length-0.75,width+2.25)),y(ypos+cy(length-0.75,width+2.25)));
            } else {
                c.moveTo(x(xpos+cx(length-3,width+3)),y(ypos+cy(length-3,width+3)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-2,width+2)),y(ypos+cy(length-2,width+2)));
                }
                c.lineTo(x(xpos+cx(length-1,width+3)),y(ypos+cy(length-1,width+3)));
                c.lineTo(x(xpos+cx(length-1,width+4)),y(ypos+cy(length-1,width+4)));
                if (remote) {
                    c.lineTo(x(xpos+cx(length-2,width+3)),y(ypos+cy(length-2,width+3)));
                }
                c.lineTo(x(xpos+cx(length-3,width+4)),y(ypos+cy(length-3,width+4)));
                c.lineTo(x(xpos+cx(length-3,width+3)),y(ypos+cy(length-3,width+3)));
            }
            c.fill();
            
            // Add the name of the signal
            c.fillStyle = "#000";
            c.textBaseline = (up ? "middle" : "bottom"); c.textAlign = (up ? "start" : "end");
            if (up && branch) {
                c.fillText(name,x(xpos+cx(length-5,width+6)),y(ypos+cy(length-5,width+6)));
            } else {
                c.fillText(name,x(xpos+cx(length-4,width+3)),y(ypos+cy(length-4,width+3)));
            }
            
            // Update the repeater
            if (element) {
                var newClassname = "semaphore ";
                if (remote) {
                    newClassname += "remote ";
                }
                newClassname += (state ? "clear" : "danger");
                element.className = newClassname;
            }
            
            // And add the extra arm (if any)
            if (extra) {
                extra(view,xpos+cx(4,width),ypos+cy(4,width));
            }
        };
        // Note: Ideally (for reuse) these should be function returning functions - but I only need one of each in this simulation
        // so I'm going to keep it simple(ish) for the moment by not doing so.
        const shuntSemaphore = function(view,xpos,ypos) {
            const name = "19";
            const state = view._model.getLever(19).isPulled();
            const element = document.getElementById("djvRomseySignalBox-"+view._id+"-signal-19");

            c.beginPath();
            c.moveTo(x(xpos),y(ypos));
            c.lineTo(x(xpos),y(ypos+3));
            c.lineTo(x(xpos+2),y(ypos+3));
            c.lineTo(x(xpos+2),y(ypos+1));
            c.arc(x(xpos+2),y(ypos+5),4,rads(90),rads(0),true);
            c.lineTo(x(xpos+2),y(ypos+5));
            c.lineTo(x(xpos+2),y(ypos+3));
            c.stroke();
            
            if (state) {
                c.fillStyle="#FFF";
            } else {
                c.fillStyle = "#000";
            }
            c.beginPath();
            c.arc(x(xpos+3),y(ypos+2.5),0.5,rads(0),rads(360),false);
            c.arc(x(xpos+4.5),y(ypos+4),0.5,rads(0),rads(360),false);
            c.fill();
            
            c.fillStyle = "#000";
            c.textBaseline = "bottom"; c.textAlign = "start";
            c.fillText(name,x(xpos),y(ypos+6));
            
            // Update the repeater
            if (state) {
                element.className = "shunt clear";
            } else {
                element.className = "shunt danger";
            }
        };
        const branchHomeSemaphore = function(view,xpos,ypos) {
            const name = "16";
            const state = view._model.getLever(16).isPulled();
            const element = document.getElementById("djvRomseySignalBox-"+view._id+"-signal-16");
            // Start with the post
            c.beginPath();
            c.moveTo(x(xpos)   , y(ypos));
            c.lineTo(x(xpos)   , y(ypos+8));
            c.lineTo(x(xpos-12), y(ypos+8));
            c.stroke();
            c.fillStyle = "red";
            c.beginPath();
            c.moveTo(x(xpos-9),y(ypos+8));
            if (state) {
                c.lineTo(x(xpos-11.25),y(ypos+5.75));
                c.lineTo(x(xpos-12.75),y(ypos+7.25));
                c.lineTo(x(xpos-10.5 ),y(ypos+9.5));
                c.lineTo(x(xpos-9    ),y(ypos+8));
                c.moveTo(x(xpos-12   ),y(ypos+5));
                c.lineTo(x(xpos-12.75),y(ypos+4.25));
                c.lineTo(x(xpos-14.25),y(ypos+5.75));
                c.lineTo(x(xpos-13.5 ),y(ypos+5.5));
                c.lineTo(x(xpos-12   ),y(ypos+5));
            } else {
                c.lineTo(x(xpos-9 ),y(ypos+5));
                c.lineTo(x(xpos-11),y(ypos+5));
                c.lineTo(x(xpos-11),y(ypos+8));
                c.lineTo(x(xpos-9 ),y(ypos+8));
                c.moveTo(x(xpos-9 ),y(ypos+4));
                c.lineTo(x(xpos-9 ),y(ypos+3));
                c.lineTo(x(xpos-11),y(ypos+3));
                c.lineTo(x(xpos-11),y(ypos+4));
                c.lineTo(x(xpos-9 ),y(ypos+4));
            }
            c.fill();
            c.fillStyle = "white";
            c.beginPath();
            if (state) {
                c.moveTo(x(xpos-11.25),y(ypos+5.75));
                c.lineTo(x(xpos-12   ),y(ypos+5));
                c.lineTo(x(xpos-13.5 ),y(ypos+6.5));
                c.lineTo(x(xpos-12.75),y(ypos+7.25));
                c.lineTo(x(xpos-11.25),y(ypos+5.75));
            } else {
                c.moveTo(x(xpos-9 ),y(ypos+5));
                c.lineTo(x(xpos-9 ),y(ypos+4));
                c.lineTo(x(xpos-11),y(ypos+4));
                c.lineTo(x(xpos-11),y(ypos+5));
                c.lineTo(x(xpos-9 ),y(ypos+5));
            }
            c.fill();
            
            // Add the name of the signal
            c.fillStyle = "#000";
            c.textBaseline = "middle"; c.textAlign = "end";
            c.fillText(name,x(xpos-14),y(ypos+6));
            
            // Update the repeater
            if (state) {
                element.className = "semaphore clear";
            } else {
                element.className = "semaphore danger";
            }
        };
        semaphore(this,false,false,"3",false,shuntSemaphore,ptsb2d-16,dnline,this._model.getLever(3).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-3"));
        semaphore(this,false,true,"4",true,undefined,pnts07-80,dnline+80,this._model.getLever(4).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-4"));
        semaphore(this,false,true,"5",false,undefined,pnts07-52,dnline+52,this._model.getLever(5).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-5"));
        drawPosition(pnts07-38,dnline+38,this._model.getPosition("H"),true);
        semaphore(this,true,true,"13",false,undefined,upbrsx+50,upbrsy-50,this._model.getLever(13).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-13"));
        drawPosition(upbrsx+60,upbrsy-60,this._model.getPosition("D"),true);
        semaphore(this,true,false,"18",false,branchHomeSemaphore,pts12u+40,upline,this._model.getLever(18).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-18"));
        semaphore(this,true,false,"20",false,undefined,ptsb2u+20,upline,this._model.getLever(20).isPulled(),
                document.getElementById("djvRomseySignalBox-"+this._id+"-signal-20"));
        drawPosition(ptsb2u+30,upline,this._model.getPosition("A",false));
        
        const shunt = function(up,branch,name,xpos,ypos,state) {
            // Note: All shunt signals aligned for wrong-way working
            var cx, cy;
            if (branch) {
                if (up) {
                    cx = function(x,y) { return  0.7*x + 0.7*y; };
                    cy = function(x,y) { return -0.7*x + 0.7*y; };
                } else {
                    cx = function(x,y) { return -0.7*x - 0.7*y; };
                    cy = function(x,y) { return  0.7*x - 0.7*y; };
                }
            } else {
                if (up) {
                    cx = function(x,_y) { return x; };
                    cy = function(_x,y) { return y; };
                } else {
                    cx = function(x,_y) { return -x; };
                    cy = function(_x,y) { return -y; };
                }
            }
            c.beginPath();
            c.moveTo(x(xpos)         , y(ypos));
            c.lineTo(x(xpos+cx(-2,2)), y(ypos+cy(-2,2)));
            c.moveTo(x(xpos)         , y(ypos));
            c.lineTo(x(xpos+cx(2,2)) , y(ypos+cy(2,2)));
            c.moveTo(x(xpos)         , y(ypos));
            c.lineTo(x(xpos+cx(0,6)) , y(ypos+cy(0,6)));
            c.stroke();

            c.beginPath();
            c.arc(x(xpos+cx(2,5)),y(ypos+cy(2,5)),2,rads(0),rads(360),false);
            c.stroke();
            c.fillStyle = "red";
            c.beginPath();
            if (state) {
               const horz = 7/13;
               const vert = 17/13;
               c.moveTo(x(xpos+cx(2-horz,5+vert)),y(ypos+cy(2-horz,5+vert)));
               c.lineTo(x(xpos+cx(2+vert,5-horz)),y(ypos+cy(2+vert,5-horz)));
               c.lineTo(x(xpos+cx(2+horz,5-vert)),y(ypos+cy(2+horz,5-vert)));
               c.lineTo(x(xpos+cx(2-vert,5+horz)),y(ypos+cy(2-vert,5+horz)));
               c.lineTo(x(xpos+cx(2-horz,5+vert)),y(ypos+cy(2-horz,5+vert)));

            } else {
                const horz = 10/13;
                const vert = 24/13;
                c.moveTo(x(xpos+cx(2+horz,5+vert)),y(ypos+cy(2+horz,5+vert)));
                c.lineTo(x(xpos+cx(2+horz,5-vert)),y(ypos+cy(2+horz,5-vert)));
                c.lineTo(x(xpos+cx(2-horz,5-vert)),y(ypos+cy(2-horz,5-vert)));
                c.lineTo(x(xpos+cx(2-horz,5+vert)),y(ypos+cy(2-horz,5+vert)));
                c.lineTo(x(xpos+cx(2+horz,5+vert)),y(ypos+cy(2+horz,5+vert)));
            }
            c.fill();
            
            // Add the name of the signal
            c.fillStyle = "#000";
            if (branch) {
                c.textBaseline = "top"; c.textAlign = "end";
                c.fillText(name,x(xpos+cx(0,6)),y(ypos+cy(0,6)));
            } else {
                c.textBaseline = (up ? "bottom" : "top"); c.textAlign = "center";
                c.fillText(name,x(xpos+cx(2,7)),y(ypos+cy(2,7)));
            }
        };
        shunt(true ,false,"21",pts10u-3,upline  ,this._model.getLever(21).isPulled());
        shunt(false,false,"22",pts10d+3,dnline  ,this._model.getLever(22).isPulled());
        shunt(false,true ,"23",pts12s+2,baline-2,this._model.getLever(23).isPulled());

        // 17 Free light
        c.beginPath();
        c.arc(x(pnts11+20),y(upline-20),2,rads(0),rads(360),true);
        c.stroke();
        if (this._model.getLight17().isFree()) {
            c.fillStyle = "red";
        } else {
            c.fillStyle = "darkred";
        }
        c.fill();
        c.fillStyle = "black";
        c.textBaseline = "top"; c.textAlign = "center";
        c.fillText("17 FREE",x(pnts11+20),y(upline-24));
        
        // Main Title
        c.font = "10pt sans-serif";
        c.textBaseline = "top"; c.textAlign = "center";
        c.fillText("ROMSEY",x(pts12u+52),y(diagramheight-12));
        c.font = "6pt sans-serif";
        c.fillText('"W B W"',x(pts12u+52),y(diagramheight-24));
    },
    renderCommutators: function() {
        const comms = [this._model.getRedbridgeUpCommutator().getState(),
                       this._model.getRedbridgeDownCommutator().getState(),
                       this._model.getKimbridgeUpCommutator().getState(),
                       this._model.getKimbridgeDownCommutator().getState()];
        const cdivs = [document.getElementById("djvRomseySignalBox-"+this._id+"-redbridge-theirs-commutator"),
                       document.getElementById("djvRomseySignalBox-"+this._id+"-redbridge-our-commutator"),
                       document.getElementById("djvRomseySignalBox-"+this._id+"-kimbridge-our-commutator"),
                       document.getElementById("djvRomseySignalBox-"+this._id+"-kimbridge-theirs-commutator")];
        const sdivs = [null,
                       document.getElementById("djvRomseySignalBox-"+this._id+"-redbridge-switch-state"),
                       document.getElementById("djvRomseySignalBox-"+this._id+"-kimbridge-switch-state"),
                       null];
        
        var cn;
        
        for(var i=0; i<comms.length; i++) 
        {
            cn = (comms[i] === CommutatorState.TRAIN_ON_LINE ? "trainonline" : (comms[i] === CommutatorState.LINE_CLEAR ? "lineclear" : "normal"));
            cdivs[i].className = "commutator "+cn;
            if (sdivs[i]) {
                sdivs[i].className = cn;
            }
        }
    },
    renderClock : function() {
        const container = document.getElementById("djvRomseySignalBox-"+this._id+"-clock");
        const canvas = document.getElementById("djvRomseySignalBox-"+this._id+"-clock-canvas");
        canvas.style.width = '100%';
        canvas.style.height = '100%';
        canvas.height = container.offsetHeight; // Note: as a side effect this actually clears the canvas too!
        canvas.width = container.offsetWidth;

        this._clock = new SegmentDisplay("djvRomseySignalBox-"+this._id+"-clock-canvas");
        this._clock.pattern         = "##:##:##";
        this._clock.displayAngle    = 0;
        this._clock.digitHeight     = 20;
        this._clock.digitWidth      = 14;
        this._clock.digitDistance   = 2.5;
        this._clock.segmentWidth    = 2;
        this._clock.segmentDistance = 0.3;
        this._clock.segmentCount    = 7;
        this._clock.cornerType      = 2;
        this._clock.colorOn         = "#ffd700";
        this._clock.colorOff        = "#4b1e05";
        this._clock.draw();
    },
    updateClock: function(t) {
        this._clock.setValue(t);
    },
    addBellPush :function(t,boxname,code) {
        const tablediv = document.getElementById("djvRomseySignalBox-"+this._id+"-bellcodelogdiv");
        const table = document.getElementById("djvRomseySignalBox-"+this._id+"-bellcodelog");
        var tr = document.createElement("tr");
        var td = document.createElement("td");
        td.appendChild(document.createTextNode(t));
        tr.appendChild(td);
        td = document.createElement("td");
        td.appendChild(document.createTextNode(boxname));
        tr.appendChild(td);
        td = document.createElement("td");
        td.appendChild(document.createTextNode(code.toString()));
        td.title = code.getDescription();
        tr.appendChild(td);
        table.appendChild(tr);
        tablediv.scrollTop = tablediv.scrollHeight;
    },
    showTimetable: function() {
        document.getElementById("djvRomseySignalBox-"+this._id+"-content").classList.remove("show");
        document.getElementById("djvRomseySignalBox-"+this._id+"-content").classList.add("hide");
        document.getElementById("djvRomseySignalBox-"+this._id+"-timetable").classList.remove("hide");
        document.getElementById("djvRomseySignalBox-"+this._id+"-timetable").classList.add("show");
    },
    _addTrainRow: function(code,origin,destination,description) {
        const table = document.getElementById("djvRomseySignalBox-"+this._id+"-timetablelog");
        var tr = table.insertRow();
        var td = tr.insertCell();
        td.appendChild(document.createTextNode(code));
        td = tr.insertCell();
        td.appendChild(document.createTextNode(origin));
        td = tr.insertCell();
        td.appendChild(document.createTextNode(destination));
        td = tr.insertCell();
        td.appendChild(document.createTextNode(""));
        td = tr.insertCell();
        td.appendChild(document.createTextNode(description));
        tr.className = "inactiveTrain";
        return tr;
    },
    addTrainToTimetable: function(code,origin,destination,description) {
        return this._addTrainRow(code,origin,destination,description);
    },
    showTrain: function(tr) {
        tr.classList.remove("inactiveTrain");
    },
    completeTrain: function(tr,reason) {
        this.updateStatus(tr,reason);
        tr.classList.add("completedTrain");
    },
    updateLocation: function(tr,location) {
        tr.childNodes[3].firstChild.data = location;
    },
    updateStatus: function(tr,status) {
        tr.childNodes[4].firstChild.data = status;
    }
};