//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals Capture: true, Subclass: true */
/* globals module, exports */
/* jshint debug: true */

// Try a function call - if it throws an exception add the text (and stack trace) to the exception div for the simulation
// with the given id
Capture = {
    exception: function(id, f, them, ...args) {
        try {
            f.apply(them,args);
        } catch (ex) {
            debugger;
            if (id) {
                var exdiv = document.getElementById("djvRomseySignalBox-"+id+"-exception");
                exdiv.classList.remove("hide");
                exdiv.classList.add("show");
                var element = document.getElementById("djvRomseySignalBox-"+id+"-exception.text");
                var text;
                if (ex instanceof Error) {
                    text = ex.name + " : " + ex.message;
                } else {
                    text = ex.toString();
                }
                text = text + '\n' + (ex.stack || ex.stacktrace || '');
    
                // Remove the chunk in the stack trace between @ and /js/
                text = text.replace(/@.*\/js\//g, " in ");
                // And version field too
                text = text.replace(/\?ver=[0-9a-fA-F]*:/g, ":");
                if (element.hasChildNodes()) {
                    element.firstChild.appendData("\n"+text);
                } else {
                    element.appendChild(document.createTextNode(text));
                }
            } else {
                // Invoked by someone who didn't provide an id in which to post the exception. We assume it's a test!
                throw ex;
            }
        }
    }
};

// An object that represents an invocation of a function on an object. The function is NOT actually
// invocation when the Invocation is created, but when the invoke method is called.
function Invocation(what, who, ...why) {
    if (!!!Invocation.isFunction(what)) {
        debugger;
        throw new Error("No function to call provided when creating an invocation");
    }
    this._what = what;
    this._who = who;
    this._why = why;
}
Invocation.isFunction = function(object) {
    return !!(object && object.constructor && object.call && object.apply);
};
Invocation.isNumber = function(n) {
    return !(isNaN(parseFloat(n)) && !isNaN(n - 0)); 
};
Invocation.prototype.invoke = function(id) {
    Capture.exception(id, this._what, this._who, ...this._why);
};
Invocation.prototype.toString = function() {
    var args = (this._why ? (Array.isArray(this._why) ? this._why.join(",") : this._why.toString()) : "");
    var who = (this._who ? this._who.toString()+"." : "");
    return who+this._what.name+"("+args+")";
};

// Make an object a subclass of a parent object by setting its prototype to a copy of that
// of the parent and then resetting the constructor of that copy back to the child
Subclass = {
    make: function(child,parent) {
        child.prototype = Object.create(parent.prototype);
        Object.defineProperty(child.prototype,'constructor', {
            value: child, enumerable: false, writable: true
        });
    }
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.Capture     = Capture;
    exports.Invocation  = Invocation;
    exports.Subclass    = Subclass;
}