//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

/* globals StartTrainAction, PauseTrainAction, UpdateLocationTrainAction, WaitAtSignalTrainAction, StopTrainAction, ArrivedTrainAction */
/* globals SetEastleighSignalsTrainAction, SetEastleighPointTrainAction, ChangeTrainAction, ExpectedTrainAction, CheckPointsTrainAction */
/* globals AtPlatformTrainAction, LeftViewOfTrainAction, TraverseLocationTrainAction */
/* globals Invocation */
function Train(controller,traincodes,name,actions,maxSpeed) {
    this._controller = controller;
    this._name = name;
    this._actions = actions;
    this._location = null;
    this._nextStep = 0;
    this._listeners = [];
    this._traincodes = traincodes;
    this._traincode = this._traincodes[0];
    this._maxSpeed = maxSpeed;
    this._speed = maxSpeed;
    this._doneStepInvocation = new Invocation(this.doneStep,this);
}
Train.prototype = {
    addListener: function(listener) {
        this._listeners.push(listener);
    },
    _start: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i].start();
        }
    },
    doneUpdateLocation: function(location) {
        const self = this;
        const f = function() {
            for(var i = 0; i<self._listeners.length; i++) {
                self._listeners[i].updateLocation(location);
            }
            self.doneStep();
        };
        return new Invocation(f);
    },
    doneChangeTrain: function() {
        const self = this;
        const f = function() {
            var i;
            for(i = 0; i<self._listeners.length; i++) {
                self._listeners[i].updateLocation("");
            }
            for(i = 0; i<self._listeners.length; i++) {
                self._listeners[i].change();
            }
            self.doneStep();
        };
        return new Invocation(f);
    },
    _updateStatus: function() {
        const status = this._actions[this._nextStep].getStatus();
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i].updateStatus(status);
        }
    },
    _completed: function() {
        for(var i = 0; i<this._listeners.length; i++) {
            this._listeners[i].completed();
        }
    },
    getName: function() { return this._name; },
    setLocation: function(location) { this._location = location; },
    getLocation: function() { return this._location; },
    getTrainCode: function(i) { i = (i ? i : 0); return this._traincodes[i]; },
    getCurrentTrainCode: function() { return this._traincode; },
    setCurrentTrainCode: function(code) { this._traincode = code; },
    getSpeed: function() { return this._speed; },
    getMaxSpeed: function() { return this._maxSpeed; },
    setSpeed: function(speed) { this._speed = speed; },
    isActive: function() { return (this._nextStep < this._actions.length); },
    doNextStep: function() {
        if (this.isActive()) {
            const action = this._actions[this._nextStep];
            if (action instanceof StartTrainAction) {
                this._start();
            }
            var doneInvocation = this._doneStepInvocation;
            if (action instanceof UpdateLocationTrainAction) {
                doneInvocation = this.doneUpdateLocation(action.getLocationName());
            } else if (action instanceof ChangeTrainAction) {
                doneInvocation = this.doneChangeTrain();
            }
            this._actions[this._nextStep].doAction(this._controller,this,doneInvocation);
            this._updateStatus();
        } else {
            this._completed();
        }
            
    },
    doneStep: function() {
        this._nextStep++;
        this.doNextStep();
    }
};

function TrainFactory(controller,model,name,origin,destination) {
    this._controller = controller;
    this._model = model;
    this._name = name;
    this._origin = origin;
    this._destinations = [destination];
    this._location = null;
    this._actions = [];
}
TrainFactory.prototype = {
    getName: function() { return this._name; },
    getOrigin: function() { return this._origin; },
    getDestination: function(i) { i = (i ? i : 0); return this._destinations[i]; },
    getDestinationsCount: function() { return this._destinations.length; },
    make: function(code) {
        const codes = [].concat(code);
        if (codes.length !== this._destinations.length) {
            throw new Error("Only "+codes.length+" train codes supplied for train '"+this._name+"', need "+this._destinations.length);
        }
        return new Train(this._controller,codes,this._name,this._actions,this._maxSpeed);
    },
    setActions: function(actions) { this._actions = actions; },
    startTrain: function(box,bellcode) {
        this._maxSpeed = bellcode.getSpeed();
        return new StartTrainAction(box,bellcode);
    },
    pause: function(duration) {
        return new PauseTrainAction(duration);
    },
    atPlatform: function(duration) {
        return new AtPlatformTrainAction(duration);
    },
    updateLocation: function(posName) {
        const newPosition = this._model.getPosition(posName);
        if (!!!newPosition) {
            throw new Error("Unknown Location passed for updateLocation");
        }
        const result = new UpdateLocationTrainAction(this._location,newPosition);
        this._location = newPosition;
        return result;
    },
    endTrain: function() {
        const oldLocation = this._location;
        this._location = null;
        return new UpdateLocationTrainAction(oldLocation,null);
    },
    waitAtSignal: function(lever) {
        const signalLever = this._model.getLever(lever);
        return new WaitAtSignalTrainAction(signalLever);
    },
    setEastleighSignals: function(first,next) {
        return new SetEastleighSignalsTrainAction(this._model,first,next);
    },
    setEastleighPoint: function(state) {
        return new SetEastleighPointTrainAction(this._model,state);
    },
    stopTrain: function(box) {
        return new StopTrainAction(box);
    },
    change: function(destination) {
        this._destinations.push(destination);
        return new ChangeTrainAction();
    },
    traverse: function(fraction,remaining) {
        return new TraverseLocationTrainAction(this._location,fraction,remaining);
    },
    expected: function(fraction) {
        return new ExpectedTrainAction(this._location,fraction);
    },
    arrived: function(box) {
        return new ArrivedTrainAction(box);
    },
    leftViewOf: function(box) {
        return new LeftViewOfTrainAction(box);
    },
    checkPoints: function(leverNumber,state) {
        return new CheckPointsTrainAction(this._model.getLever(leverNumber),state);
    }
};