//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals BoxState : true, BellCode, Subclass, BoxState, SignalBox, Invocation */
/* globals module, exports */
function SingleLineBox(controller, bell, light17, lever6) {
    SignalBox.call(this,controller, bell);
    this._light17 = light17;
    this._lever6 = lever6;
    this._trainOnTrack = false;
    
    this._releaseLightInvocation = new Invocation(this._light17.release,this._light17);
    this._lockLightInvocation    = new Invocation(this._light17.lock   ,this._light17);
    this._checkDirectionLeverPulledInvocation = new Invocation(this.checkDirectionLever,this,true);
    this._checkDirectionLeverNormalInvocation = new Invocation(this.checkDirectionLever,this,false);
    
    // Define actions we can take and assign based on current state and the bell push
    const self = this;
    
    this._action[BoxState.RECEIVING][BellCode.OFFERCODES[0]] = function() {
        if (self._receiveTimeout) {
            self._receiveTimeout.cancel();
            self._receiveTimeout = null;
        }
        // We can only accept if
        //   (a) no offered trains are remaining (i.e. we haven't sent one that still (notionally at least) in section)
        //   (b) nothing is on the line
        if (self._offeredTrains.length === 0 && !!!self._trainOnTrack) {
            const bellLength = self._bell.echoBellCode();
            self._scheduler.addRealTimeEvent(bellLength+1   ,self._releaseLightInvocation);
            self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_TRAIN_ENTERING);
        }
    };
    for(var i=1; i<BellCode.OFFERCODES.length; i++) {
        this._action[BoxState.RECEIVING][BellCode.OFFERCODES[i]] = this._action[BoxState.RECEIVING][BellCode.OFFERCODES[0]];
    }
    this._action[BoxState.RECEIVING][BellCode.Shunt] = function() {
        const bellLength = self._bell.echoBellCode();
        self._scheduler.addRealTimeEvent(bellLength+1   ,self._releaseLightInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_SHUNT_ATTN);
    };
}
Subclass.make(SingleLineBox,SignalBox);

SingleLineBox.prototype.checkDirectionLever = function(need) {
    const clockTicks = this._scheduler.getClock().getTicks();
    if (need && this._state === BoxState.AWAITING_THEIR_LINE_CLEAR) {
        if (this._lever6.isPulled()) {
            this.changeState(BoxState.PAUSE_TRAIN_START);
            this._scheduler.addGameTimeEvent(SignalBox.pauseTrainDelay,this._sendOutOfSectionInvocation);
        } else {
            if (clockTicks > this._startLineClearCheck+60) {
                this._startLineClearCheck = clockTicks;
                this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. I received an acknowledgement "+
                        "of my offer of a train from you, but the direction lever has not been pulled. Can you pull the direction lever " +
                        "now so that we can continue operation.");
            }
            this._scheduler.rescheduleCurrentEvent(0.5);
        }
    } else if (!!!need && this._state === BoxState.AWAITING_THEIR_TRAIN_ON_LINE && !!!this._lever6.isPulled()) {
        this._trainOnTrack = true;
        this.makeTrainOnLineCall();
        this._scheduler.addRealTimeEvent(1,new Invocation(this.changeState,this,this._preOfferState));
    } else {
        this._scheduler.addRealTimeEvent(0.1,new Invocation(this.checkDirectionLever,this,need));
    }
};
SingleLineBox.prototype.canTrainExit = function() {
    return this._state === BoxState.EXPECTING_TRAIN && !!!this._light17.isFree();
};
SingleLineBox.prototype.canTrainBeOffered = function(_bellcode) {
    return !!!this._trainOnTrack && this._state === BoxState.NONE;
};
SingleLineBox.prototype.doEnterSectionReturningNewState = function(bellLength) {
    this._scheduler.addRealTimeEvent(bellLength+1,this._lockLightInvocation);
    this._trainOnTrack = true;
    return BoxState.EXPECTING_TRAIN;
};
SingleLineBox.prototype.cancelEnterSection = function(bellLength) {
    this._scheduler.addRealTimeEvent(bellLength+1,this._lockLightInvocation);
};
SingleLineBox.prototype.scheduleCheckLineEmpty = function() {
    this._scheduler.addRealTimeEvent(0.1,this._checkDirectionLeverPulledInvocation);
};
SingleLineBox.prototype.scheduleCheckLineOccupied = function() {
    this._scheduler.addRealTimeEvent(0.1,this._checkDirectionLeverNormalInvocation);
};
SingleLineBox.prototype.doTrainOutOfSection = function() {
    this._trainOnTrack = false;
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.SingleLineBox = SingleLineBox;
}