//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals BoxState : true, BellCode, CommutatorState, Subclass, BoxState, SignalBox, Invocation */
/* globals module, exports */
function DoubleLineBox(controller, bell, ourLine, theirLine) {
    SignalBox.call(this, controller, bell);
    this._ourLine = ourLine;
    this._theirLine = theirLine;
    
    this._setOurLineToClearInvocation  = new Invocation(this._ourLine.setState,this._ourLine,CommutatorState.LINE_CLEAR);
    this._setOurLineToOnlineInvocation = new Invocation(this._ourLine.setState,this._ourLine,CommutatorState.TRAIN_ON_LINE);
    this._setOurLineToNormalInvocation = new Invocation(this._ourLine.setState,this._ourLine,CommutatorState.NORMAL);
    this._changePrevStateToNONE      = new Invocation(this.changePrevState,this,BoxState.NONE);
    this._changePrevStateToRECEIVING = new Invocation(this.changePrevState,this,BoxState.RECEIVING);
    this._checkTheirCommutatorClear  = new Invocation(this.checkTheirCommutator,this,CommutatorState.LINE_CLEAR);
    this._checkTheirCommutatorOnline = new Invocation(this.checkTheirCommutator,this,CommutatorState.TRAIN_ON_LINE);
   
    // Define actions we can take and assign based on current state and the bell push
    var i;
    const self = this;
    
    this._action[BoxState.RECEIVING][BellCode.OFFERCODES[0]] = function() {
        if (self._receiveTimeout) {
            self._receiveTimeout.cancel();
            self._receiveTimeout = null;
        }
        // We believe it's a train offer - if our commutator is normal we can accept the offer (it'll be TRAIN_ON_LINE if we were expecting)
        // attn for out of section
        if (self._ourLine.getState() === CommutatorState.NORMAL) {
            const bellLength = self._bell.echoBellCode();
            self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToClearInvocation);
            self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_TRAIN_ENTERING);
        } else {
            // We're ignoring the offer and hence a new attn will be needed to offer it again
            self.changeState(BoxState.NONE);
        }
    };
    for(i=1; i<BellCode.OFFERCODES.length; i++) {
        this._action[BoxState.RECEIVING][BellCode.OFFERCODES[i]] = this._action[BoxState.RECEIVING][BellCode.OFFERCODES[0]];
    }
    this._action[BoxState.RECEIVING][BellCode.Shunt] = function() {
        if (self._receiveTimeout) {
            self._receiveTimeout.cancel();
            self._receiveTimeout = null;
        }
        const bellLength = self._bell.echoBellCode();
        self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToOnlineInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_SHUNT_ATTN);
    };
    // We can get ATTN while we waiting for TRAIN_ENTERING, so we need to handle case with a state change (and it's associated bell handling)
    this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.Attn] = this._echoBellAndChangeStateFunction(BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT);
    this._action[BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT][BellCode.EnteringSection]  = this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.EnteringSection];
    this._action[BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT][BellCode.WronglyDescribed] = this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.WronglyDescribed];
    this._action[BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT][BellCode.Cancel]           = this._action[BoxState.AWAITING_TRAIN_ENTERING][BellCode.Cancel];
    this._action[BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT][BellCode.Attn] = function() {
        // Repeat of attn, so just echo the attn
        self._bell.echoBellCode();
    };
    this._action[BoxState.AWAITING_TRAIN_ENTERING_POSSIBLE_EXIT][BellCode.OutOfSection] = function() {
        if (self._offeredTrains.length > 0) {
            self._bell.echoBellCode();
            self._offeredTrains.shift();
        }
        self.changeState(BoxState.AWAITING_TRAIN_ENTERING);
    };
    
    // And the worst case is that we were just pausing before sending another train and they get in first :)
    const echoBellAndChangePreOfferState = function(state) {
        return function() {
            self._preOfferState = state;
            return self._bell.echoBellCode();
        };
    };
    this._preOfferAction = [];
    this._preOfferAction[BoxState.NONE] = [];
    this._preOfferAction[BoxState.NONE][BellCode.Attn] = echoBellAndChangePreOfferState(BoxState.RECEIVING);
   
    this._preOfferAction[BoxState.RECEIVING] = [];
    this._preOfferAction[BoxState.RECEIVING][BellCode.OutOfSection] = function() {
        if (self._offeredTrains.length > 0) {
            self._bell.echoBellCode();
            self._offeredTrains.shift();
        }
        self.changePrevState(BoxState.NONE);
    };
    this._preOfferAction[BoxState.RECEIVING][BellCode.Cancel] = echoBellAndChangePreOfferState(BoxState.NONE);
    this._preOfferAction[BoxState.RECEIVING][BellCode.Shunt] = function() {
        const bellLength = self._bell.echoBellCode();
        self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToOnlineInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_SHUNT_ATTN);
    };
    this._preOfferAction[BoxState.RECEIVING][BellCode.OFFERCODES[0]] = function() {
        // We believe it's a train offer - if our commutator is normal we can accept the offer (it'll be TRAIN_ON_LINE if we were expecting
        // attn for out of section
        if (self._ourLine.getState() === CommutatorState.NORMAL) {
            const bellLength = self._bell.echoBellCode(); 
            self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToClearInvocation);
            self._scheduler.addRealTimeEvent(bellLength+1.01,self._changeStateToAWAITING_TRAIN_ENTERING);
        } else {
            // We're ignoring the offer and hence a new attn will be needed to offer it again
            self.changePrevState(BoxState.NONE);
        }
    };
    for(i=1; i<BellCode.OFFERCODES.length; i++) {
        this._preOfferAction[BoxState.RECEIVING][BellCode.OFFERCODES[i]] = this._preOfferAction[BoxState.RECEIVING][BellCode.OFFERCODES[0]];
    }
    this._preOfferAction[BoxState.AWAITING_TRAIN_ENTERING] = [];
    this._preOfferAction[BoxState.AWAITING_TRAIN_ENTERING][BellCode.EnteringSection] = function() {
        const bellLength = self._bell.echoBellCode();
        self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToOnlineInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._trainIntoSectionInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.02,self._changePrevStateToNONE);
    };
    this._preOfferAction[BoxState.AWAITING_TRAIN_ENTERING][BellCode.WronglyDescribed] = function() {
        const bellLength = self._bell.echoBellCode();
        self.cancelEnterSection(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changePrevStateToRECEIVING);
    };
    this._preOfferAction[BoxState.AWAITING_TRAIN_ENTERING][BellCode.Cancel] = function() {
        const bellLength = self._bell.echoBellCode();
        self.cancelEnterSection(bellLength);
        self._scheduler.addRealTimeEvent(bellLength+1.01,self._changePrevStateToNONE);
    };
    this._preOfferAction[BoxState.AWAITING_SHUNT_ATTN] = [];
    this._preOfferAction[BoxState.AWAITING_SHUNT_ATTN][BellCode.Attn.equals(this._bell.getBellCode())] = echoBellAndChangePreOfferState(BoxState.AWAITING_END_SHUNT);
    this._preOfferAction[BoxState.AWAITING_END_SHUNT] = [];
    this._preOfferAction[BoxState.AWAITING_END_SHUNT][BellCode.EndShunt] = function() {
        const bellLength = self._bell.echoBellCode();
        self._scheduler.addRealTimeEvent(bellLength+1   ,self._setOurLineToNormalInvocation);
        self._scheduler.addRealTimeEvent(bellLength+1.01,this._changePrevStateToNONE);
    };
    
    const doPauseAction = function() {
        if (self._preOfferAction[self._preOfferState] && self._preOfferAction[self._preOfferState][self._bell.getBellCode()]) {
            self._preOfferAction[self._preOfferState][self._bell.getBellCode()]();
        }
    };
    this._action[BoxState.PAUSE_TRAIN_START] = [];
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.Attn] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.OutOfSection] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.Cancel] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.Shunt] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.Shunt] = doPauseAction;
    for(i=0; i<BellCode.OFFERCODES.length; i++) {
        this._action[BoxState.PAUSE_TRAIN_START][BellCode.OFFERCODES[i]] = doPauseAction;
    }
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.EnteringSection] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.WronglyDescribed] = doPauseAction;
    this._action[BoxState.PAUSE_TRAIN_START][BellCode.EndShunt] = doPauseAction;
}

Subclass.make(DoubleLineBox,SignalBox);

DoubleLineBox.prototype.checkTheirCommutator = function(need) {
    const clockTicks = this._scheduler.getClock().getTicks();
    if (need === CommutatorState.LINE_CLEAR && this._state === BoxState.AWAITING_THEIR_LINE_CLEAR) {
        if (this._theirLine.getState() === need) {
            this.changeState(BoxState.PAUSE_TRAIN_START);
            this._scheduler.addGameTimeEvent(SignalBox.pauseTrainDelay,this._sendOutOfSectionInvocation);
        } else {
            if (clockTicks > this._startLineClearCheck+60) {
                this._startLineClearCheck = clockTicks;
                this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. I received an acknowledgement "+
                        "of my offer of a train from you, but your commutator is not showing LINE CLEAR. Can you move your commutator to LINE CLEAR so "+
                        "that I can signal my train to proceed to your box.");
            }
            this._scheduler.addRealTimeEvent(1,new Invocation(this.checkTheirCommutator,this,need));
        }
    } else if (need === CommutatorState.TRAIN_ON_LINE && this._state === BoxState.AWAITING_THEIR_TRAIN_ON_LINE && this._theirLine.getState() === need) {
        this.makeTrainOnLineCall();
        this._scheduler.addRealTimeEvent(1,new Invocation(this.changeState,this,this._preOfferState));
    } else if (need === CommutatorState.NORMAL) {
        if (this._theirLine.getState() === CommutatorState.TRAIN_ON_LINE) { // Note: LINE_CLEAR for this case is treated as if it moved through NORMAL first
            if (this._idleStarted && clockTicks > this._idleStarted+60) {
                this._messagebox.showMessage(this.getName(),"This is "+this.getName()+" Signaller. I received an 'Train Out of Section' "+
                        "bell code from you, but your commutator is still showing TRAIN ON LINE. Can you move your commutator to NORMAL so "+
                        "that we can get back to standard operation.");
                this._idleStarted = clockTicks;
            }
            this._scheduler.addGameTimeEvent(3,new Invocation(this.checkTheirCommutator,this,need));
        } else {
            this._idleStarted = null;
            this.changeState(BoxState.PAUSE_AFTER_EXIT_ACK);
            this._scheduler.addRealTimeEvent(5,this._changeAfterPauseInvocation);
        }
    } else {
        this._scheduler.addRealTimeEvent(1,new Invocation(this.checkTheirCommutator,this,need));
    }
};
DoubleLineBox.prototype.canTrainExit = function() {
    return ( (this._state === BoxState.NONE || this._state === BoxState.AWAITING_TRAIN_ENTERING) && 
             (this._ourLine.getState() === CommutatorState.TRAIN_ON_LINE) );
};
DoubleLineBox.prototype.canTrainBeOffered = function(_bellcode) {
    if (this._theirLine.getState() !== CommutatorState.NORMAL) {
        return false;
    }
    return this._state === BoxState.NONE || this._state === BoxState.AWAITING_TRAIN_ENTERING || this._state === BoxState.AWAITING_SHUNT_ATTN;
};
DoubleLineBox.prototype.doEnterSectionReturningNewState = function(bellLength) {
    this._scheduler.addRealTimeEvent(bellLength+1,this._setOurLineToOnlineInvocation);
    return BoxState.NONE;
};
DoubleLineBox.prototype.cancelEnterSection = function(bellLength) {
    this._scheduler.addRealTimeEvent(bellLength+1.01,this._setOurLineToNormalInvocation);
};
DoubleLineBox.prototype.scheduleCheckLineEmpty = function() {
    this._scheduler.addRealTimeEvent(1,this._checkTheirCommutatorClear);
};
DoubleLineBox.prototype.scheduleCheckLineOccupied = function() {
    this._scheduler.addRealTimeEvent(1,this._checkTheirCommutatorOnline);
};
DoubleLineBox.prototype.doTrainOutOfSection = function() {
    this._scheduler.addRealTimeEvent(2.0,this._setOurLineToNormalInvocation);
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.DoubleLineBox = DoubleLineBox;
}