//
// © Copyright 2020 David Vines
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived 
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
/* globals module, exports */
function BellCode(s,description) {
    if (s && s.length > 0) {
        this._code = s;
    } else if (s === ""){
        this._code = "";
    } else {
        this._code = "1";
    }
    this._description = description;
    this._gaps = this._analyze();
    this._longGaps = [];
    for(var i=0; i<this._gaps.length; i++) {
        if (this._gaps[i] === "-") {
            this._longGaps.push(i);
        }
    }
    BellCode._KNOWNCODES.push(this);
}

BellCode._KNOWNCODES = [];

BellCode.recognise = function(gapLengths) {
    if (gapLengths.length === 0) {
        // No gaps -> must be ATTN
        return BellCode.Attn;
    } else if (gapLengths.length === 1) {
        // Only one gap - it must be short and hence must be train entering section
        return BellCode.EnteringSection;
    } else {
        // Build the list of possible - the length must match for starters
        var possibles = BellCode._KNOWNCODES.filter(function(p) { return p.getGaps().length === gapLengths.length; });
        
        // Easy cases - no or only one match
        if (possibles.length === 0) {
            return BellCode.Unrecognised;
        } else if (possibles.length === 1) {
            return possibles[0];
        }
        
        // OK, we have two or more choices to pick from.
        // Build a score for each possible
        var scores = [];
        var shorts,j,averageShort;
        var filterTrue = function() { return true; };
        var average = function(result,x,_i,a) { return result+x/a.length; };
        for(var i=0; i<possibles.length; i++) {
            // Start with one point, but lose a point for each long gap
            scores[i] = 1 - possibles[i]._longGaps.length;

            if (possibles[i]._longGaps.length !== 0) {
                // Calcuate averages for the (putative) short gaps
                shorts = gapLengths.slice(0);
                for(j=0; j<possibles[i]._longGaps.length; j++) {
                    delete shorts[possibles[i]._longGaps[j]];
                }
                shorts = shorts.filter(filterTrue);
                averageShort = shorts.reduce(average,0);
    
                // Score three points if the long gap is at least 0.25s more that the average length of the short gaps
                for(j=0; j<possibles[i]._longGaps.length; j++) {
                    if (gapLengths[possibles[i]._longGaps[j]] > averageShort+0.25) {
                        scores[i] += 3;
                    }
                }
            }
        }
        
        // Pick the possible with highest score
        var bestScore = scores[0];
        var bestIndex = 0;
        for(i=1; i<scores.length; i++) {
            if (scores[i] > bestScore) {
                bestIndex = i;
                bestScore = scores[bestIndex];
            }
        }
        return possibles[bestIndex];
    }
};

BellCode.prototype = {
    _analyze: function() {
        var gaps = [];
        var one = '1'.charCodeAt(0);
        
        for(var j=0; j< this._code.length; j++) {
            if (this._code.charAt(j) === "-") {
                gaps.push("-");
            } else {
                for(var k=one; k<this._code.charAt(j).charCodeAt(0); k++) {
                    gaps.push(".");
                }
            }
        }
        
        return gaps;
    },
    toLongString: function() {
        return this._code + "("+this.getDescription()+(this._frozen ? "" : "[warm]") +")";
    },
    toString: function() {
        return this._code;
    },
    getDescription: function() {
        if (this._description) {
            return this._description;
        } else {
            for(var i=0; i<BellCode._KNOWNCODES.length; i++) {
                if (this._code === BellCode._KNOWNCODES[i]._code) {
                    return BellCode._KNOWNCODES[i]._description;
                }
            }
            return this._code;
        }
    },
    getGaps: function() {
        if (this._gaps) {
            return this._gaps;
        } else {
            return this._analyze();
        }
    },
    equals: function(other) {
        if (other instanceof BellCode) {
            return this._code === other._code;
        } else {
            return false;
        }
    },
    getSpeed: function() {
        return undefined; // Unless overriden by the object
    }
};

BellCode.Empty               = new BellCode("","[empty bell code]");           /* Gaps [] */
BellCode.Attn                = new BellCode("1","Attention");                  /* Gaps [] */
BellCode.EnteringSection     = new BellCode("2","Train Entering Section");     /* Gaps [.] */
BellCode.OutOfSection        = new BellCode("2-1","Train Leaving Section");    /* Gaps [.-] */
BellCode.OfferExpress        = new BellCode("4","Express Train");              /* Gaps [...] */
BellCode.OfferExpress.getSpeed = function() { return 80; };
BellCode.OfferPassenger      = new BellCode("3-1","Passenger Train");          /* Gaps [..-] */
BellCode.OfferPassenger.getSpeed = function() { return 75; };
BellCode.OfferEmptyTrain     = new BellCode("2-2-1","Empty Coaching Stock");   /* Gaps [.-.-] */
BellCode.OfferEmptyTrain.getSpeed = function() { return 75; };
BellCode.OfferExpressFreight = new BellCode("5","Express Freight Train");      /* Gaps [....] */
BellCode.OfferExpressFreight.getSpeed = function() { return 60; };
BellCode.OfferFreight        = new BellCode("4-1","Freight Train");            /* Gaps [...-] */
BellCode.OfferFreight.getSpeed = function() { return 40; };
BellCode.OfferSlowFreight    = new BellCode("3-2","Slow Freight");             /* Gaps [..-.] */
BellCode.OfferSlowFreight.getSpeed = function() { return 30; };
BellCode.OfferLightLoco      = new BellCode("2-3","Light Loco");               /* Gaps [.-..] */
BellCode.OfferLightLoco.getSpeed = function() { return 45; };
BellCode.Shunt               = new BellCode("3-3-2","Train Entering Shunt");   /* Gaps [..-..-.] */
BellCode.EndShunt            = new BellCode("8","Train Leaving Shunt");        /* Gaps [.......] */
BellCode.Cancel              = new BellCode("3-5","Cancel");                   /* Gaps [..-....] */
BellCode.WronglyDescribed    = new BellCode("5-3","Train Wrongly Described");  /* Gaps [....-..] */
BellCode.Unrecognised        = new BellCode("","Unrecognised bell code");

BellCode.OFFERCODES = [BellCode.OfferExpress,     BellCode.OfferPassenger,   BellCode.OfferEmptyTrain, BellCode.OfferExpressFreight,
                       BellCode.OfferFreight,     BellCode.OfferSlowFreight, BellCode.OfferLightLoco
                      ];

//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.BellCode = BellCode;
}