/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals console, Game, GameMap, ROT, TECHS, RepairShipOrder, ScrapShipOrder , Fighter, ShipDesignType */
/* exported TurnProcessor */
var TurnProcessor = {
	resetPlayerReports: function() {
		for(let player of Game._players) {
			player.resetReports();
		}
	},
	resetPlayerCommsGrids: function() {
		for(let player of Game._players) {
			player.resetCommsGrid();
		}
	},
	moveShips: function(workingMap) {
		for(let player of Game._players) {
			if (player.active) {
				for(let ship of player.ships) {
					ship.lastHex = ship.location;
					ship.retreatingTo = null;
					ship.makeJump(ship.order.jump);
					workingMap.addObject(ship.location, {ship: ship, owner:player});
				}
			}
		}
	},

	processTurn: function() {
		const debug = location.hostname == 'virtual.internal';
		const updates = []; // Updates for the human
		const events = [];  // Major events for the game history
		const shipsToBeDestroyed = [];

		// Step 0. Prep the players for the results of the turn
		TurnProcessor.resetPlayerReports();

		// Ship orders
		// Step 1. create a new map and place all ships in their new locations
		const workingMap = new GameMap(Game.State.baseMap);
		TurnProcessor.moveShips(workingMap);

		// Step 2. Resolve what happening in each hex
		workingMap.forEachHex(hex => {
			// Step 2a. Work out what is in the hex
			const bystyle = TurnProcessor.getShipsByStyle(hex);	// Each element of bystyle (when present) is itself a sparse array showing how many ships of that size are present
			const troopsByStyle = TurnProcessor.getTroopsByStyle(hex);

			// Step 2b. Look for non-dredger units
			TurnProcessor.updateShipStatus(hex);

			// Step 2c. Resolve combat
			if (Object.keys(bystyle).length > 1) {
				TurnProcessor.resolvePossibleCombat(hex,
					bystyle,
					debug,
					(Object.keys(bystyle).some(style => style == Game.STYLENAMES[0])) ? updates : null,
					events,
					shipsToBeDestroyed); // Will also remove destroyed ships from hex.objects
			}

			// Step 2d. Resolve planet busting (if any)
			const thereWasAPlanet = Boolean(hex.planet);
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					TurnProcessor.processPlanetBusting(debug,events,updates,hex,object.ship);
				}
			}

			// Step 2e. Give the colony (if any) a report - also mark the hex as interdicted or not
			if (hex.owner && hex.level) {
				const report = TurnProcessor.getReportOnOtherShips(hex,hex.owner);
				Game._stylenameToPlayerMapping[hex.owner].map.setColonyReport(hex.id,report);
				Game._stylenameToPlayerMapping[hex.owner].map.setLastScanned(hex.id,Game.State.turn,report);
				const interdict = TurnProcessor.anyHostileShips(hex,hex.owner);
				Game.State.baseMap.setInterdicted(hex.id,interdict);
				if (interdict) {
					Game._stylenameToPlayerMapping[hex.owner].addTurnNote("interdicted",hex.id);
				}
			} else {
				Game.State.baseMap.setInterdicted(hex.id,false);
			}

			// Step 2f. And then for each ship determine what it does
			ROT.RNG.shuffle(hex.objects); // Shuffle the ships so that each colony ship has the same chance to be the first to colonise
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					// processShipOrder returns true is the ship is to be kept, false if it is to be destroyed
					if (!TurnProcessor.processShipOrder(debug,events,hex,object.ship,object.owner,troopsByStyle,thereWasAPlanet)) {
						shipsToBeDestroyed.push(object);
					}
				}
			}

			// Step 2g. Resolve ground combat (if any)
			if (Object.keys(troopsByStyle).length > 1) {
				// processInvasion returns the 'winner' if the planet has suicide planet busted
				const updatePlayer = Object.keys(troopsByStyle).some(style => style == Game.STYLENAMES[0]);
				const winnerButBusted = TurnProcessor.processInvasion(debug,events,updatePlayer,updates,hex,troopsByStyle);
				if (winnerButBusted) {
					TurnProcessor.suicidePlanetBust(debug,events,updatePlayer,updates,hex,winnerButBusted,shipsToBeDestroyed);
				}
			} else if (Object.keys(troopsByStyle).length == 1) {
				Game.State.baseMap.setMarines(hex.id,troopsByStyle[hex.owner]); // Put the marines (plus any additions) back on the planet
				hex.marines = troopsByStyle[hex.owner];
			}

			// Step 2h. Finally generate ship reports (after colonisation and ground combat) and note presence of other ships
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					TurnProcessor.updateReports(hex,object.ship,object.owner,thereWasAPlanet);
				}
			}
		});

		// Step 3. Remove those ship no longer in existence
		for(let object of shipsToBeDestroyed) {
			if (!object.fighter) {
				object.owner.addTurnNote("shipDestroyed",object.ship.location, {name: object.ship.name});
				object.owner.destroyShip(object.ship);
				if (object.owner == (Game._players[0]) && object.ship.report) {
					updates.push(object.ship.report); // Since there's no other way the player will see the report!
				}
				if (object.ship.design.type === ShipDesignType.StarBase) {
					Game.State.baseMap.removeStarBase(object.ship.location);
				}
			}
		}

		// Step 4. Apply non-ship orders if the player still has their capital (if not - they're eliminated)
		for(let i=0; i<Game._players.length; i++) {
			const player = Game._players[i];
			const stylename = player.stylename;
			const capitalHex = Game.State.baseMap.hex(player.capital);
			if (capitalHex.planet && capitalHex.owner == stylename) {
				for(let order of Game._orders[stylename]) {
					let update = order.apply(Game._players[i],Game.State.baseMap,Game.State.turn);
					if (update && i===0) {
						// There's an update and since it's for the human we want to add it to the list of moves
						updates.push(update);
					}
					if (update && debug) {
						console.info(`${stylename}: ${update}`);
					}
				}
			} else if (player.active){
				if (i == 0 ) {
					updates.push("You've been eliminated from the game as you have lost your capital!");
					events.push("You've been eliminated from the game as you have lost your capital!");
				} else {
					updates.push(`A player, ${Game.FULLNAMES[stylename]}, has been eliminated from the game as they have lost their capital!`);
					events.push(`A player, ${Game.FULLNAMES[stylename]}, has been eliminated from the game as they have lost their capital!`);
				}
				player.deactivate();
				Game.State.baseMap.forEachHex(hex => {
					if (hex.owner == stylename) {
						Game.State.baseMap.removeColony(hex.id,true); // Also remove any starbases
					}
				});
			}
		}

		// Step 5. Get the players to recompute their comms grid following all the colonisation, invasions and destructions
		TurnProcessor.resetPlayerCommsGrids();

		return [updates,events];
	},
	getShipsByStyle: function(hex) {
		let bystyle = {}; // (re)build the mapping from stylename to a sparse array of ship counts (of a given ship size) present in the hex
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const name = object.owner.stylename;
				bystyle[name] = bystyle[name] ? bystyle[name] : [];
				bystyle[name][0] = bystyle[name][0] ? bystyle[name][0] : 1; // Use the 0 entry to count all ships
				bystyle[name][object.ship.size] = bystyle[name][object.ship.size] ? bystyle[name][object.ship.size]++ : 1;
			}
		}
		return bystyle;
	},
	getTroopsByStyle: function(hex) {
		let troopsByStyle = {};
		if (hex.planet && hex.marines) troopsByStyle[hex.owner] = hex.marines;
		return troopsByStyle;
	},
	updateShipStatus: function(hex) {
		let nonDredgerUnits = Boolean(hex.planet);
		for(let object of hex.objects) {
			if (!nonDredgerUnits) {
				nonDredgerUnits = (!Boolean(object.ship)) || object.ship.getDredgers() == 0;
			}
		}
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const ship = object.ship;
				// Presence of NonDedger units
				ship.nonDredgerUnits = nonDredgerUnits;
				ship.report = ""; // Initialise the report
			}
		}
	},
	processPlanetBusting: function(debug,events,updates,hex,ship) {
		// Planet Busting (if ordered to so)
		if (ship.order.bust) {
			if (!ship.canPlanetBust) {
				ship.report += `Cannot planet bust ${hex.id}. No operational Planet Busters! `;
			} else if (!hex.planet) {
				ship.report += `Cannot planet bust ${hex.id}. There is no planet here to bust! `;
			} else {
				const busted = `The planet at ${hex.id} has been destroyed! `;
				ship.report += busted;
				if (hex.owner && hex.owner == Game._player[0].stylename) {
					if (debug) console.info(busted);
					updates.push(busted);
					Game._stylenameToPlayerMapping[hex.owner].addTurnNote("planetBusted",hex.id);
				}
				events.push(busted);
				Game.State.baseMap.destroyPlanet(hex.id);
				// Update the working map too
				delete hex.planet; delete hex.marines; delete hex.level;
			}
		}
	},
	processShipOrder: function(debug,events,hex,ship,owner,troopsByStyle,thereWasAPlanet) {
		let keepShip = true;

		// Scrap (if so ordered)
		if (ship.order instanceof ScrapShipOrder) {
			keepShip = false;
			const actualRefund = ship.getScrapValue();
			ship.report = `Scrapped ${ship.name} for ${+actualRefund} PPs (which has been added to the Brought Forward amount). `;
			owner.carriedProduction += actualRefund;
		}

		// Repair orders
		if (ship.retreatingTo == null && ship.order instanceof RepairShipOrder) {
			if (ship.getRepairCost() == ship.order.value) {
				ship.repairAll();
				ship.report = `Fully repaired at a cost of ${ship.order.value} PPs. `;
				owner.carriedProduction -= ship.order.value;
			} else if (ship.retreatingTo !== null){
				ship.report = "Repair aborted as the ship was further damaged. ";
			}
		}

		// Colonisation (if ordered to do so)
		if (ship.order.colonise) {
			if (hex.planet && hex.owner !== owner.stylename && hex.level > 0) {
				ship.report += `Unable to colonise ${hex.id} as it has already been colonised by ${Game.FULLNAMES[hex.owner]}. `;
			} else if (hex.planet && hex.owner !== owner.stylename && hex.level == 0) {
				ship.report += `There is a star base belonging to ${Game.FULLNAMES[hex.owner]} here. Unable to colonise. `;
			} else if (hex.planet && hex.owner == owner.stylename && hex.level > 4) {
				ship.report += `Colonisation of ${hex.id} cannot be made as the colony is already at level ${hex.level}. `;
			} else if (hex.planet && hex.owner == owner.stylename && ship.getColonyImprovement() == 0) {
				ship.report += `Colonisation of ${hex.id} cannot be made as the colony ship does not have at least six undamaged colony blocks. `;
			} else if (hex.planet && hex.owner == owner.stylename && hex.level < 5) {
				ship.report += `Colonisation of ${hex.id} has been achieved adding ${ship.getColonyImprovement()} to the existing colony. `;
				Game.State.baseMap.colonise(ship.location,owner.stylename,ship.getColonyImprovement());
				Game.State.baseMap.addMarines(ship.location,ship.marines);
				hex.level += ship.getColonyImprovement();
				if (debug) console.info(`Colonisation of ${hex.id} by ${Game.FULLNAMES[owner.stylename]} raising it to ${hex.level}`);
				Game._stylenameToPlayerMapping[hex.owner].addTurnNote("colonised",hex.id);
				keepShip = false;
			} else if (hex.planet) {
				ship.report += `Colonisation of ${hex.id} has been achieved. `;
				events.push(`Colonisation of ${hex.id} by ${Game.FULLNAMES[owner.stylename]}`);
				if (debug) console.info(`Colonisation of ${hex.id} by ${Game.FULLNAMES[owner.stylename]}`);
				Game.State.baseMap.colonise(hex.id,owner.stylename,ship.getColonyImprovement());
				Game.State.baseMap.addMarines(ship.location,ship.marines);
				hex.level = ship.getColonyImprovement(); // So that the events/updates are correct for the next ship
				hex.owner = owner.stylename;
				Game._stylenameToPlayerMapping[hex.owner].addTurnNote("colonised",hex.id);
				keepShip = false;
			} else if (thereWasAPlanet) {
				ship.report += `Unable to colonise ${hex.id}, the planet here was destroyed! `;
			} else {
				ship.report += `Unable to colonise ${hex.id}, there is no planet here! `;
			}
		}

		// Collect troops (if so ordered)
		if (ship.order.marinesEmbarking) {
			if (hex.planet) {
				if (hex.owner == owner.stylename) {
					if (hex.marines > 0) {
						if (hex.marines >= ship.getMaxMarines()-ship.marines) {
							troopsByStyle[hex.owner] -= ship.getMaxMarines()-ship.marines;
							hex.marines -= ship.getMaxMarines()-ship.marines; // Our local copy of the hex details
							Game.State.baseMap.addMarines(ship.location,ship.marines-ship.getMaxMarines()); // This removes the marines from the colony
							ship.marines = ship.getMaxMarines();
						} else {
							troopsByStyle[hex.owner] -= hex.marines;
							ship.report += `Only ${hex.marines} marine compan${hex.marines == 1 ? "y" : "ies"} loaded as that was all that was available. `;
							Game.State.baseMap.setMarines(ship.location,0);
							ship.marines += hex.marines;
							hex.marines = 0;
						}
					} else {
						ship.report += "No marines loaded as none were available. ";
					}
				} else {
					ship.report += "No marines loaded as we don't own this colony. '";
				}
			} else {
				ship.report += "No marines loaded as there is no planet here. ";
			}
		}

		// Disembark troops (if so ordered) {
		if (ship.order.marinesDisembarking && ship.marines > 0) {
			if (hex.planet && hex.level) {
				troopsByStyle[owner.stylename] = troopsByStyle[owner.stylename] || 0;
				troopsByStyle[owner.stylename] += ship.marines;
				ship.marines = 0;
			} else if (hex.planet) {
				ship.report += "Marines have not disembarked as there is no colony at the planet here! ";
			} else {
				ship.report += "Marines have not disembarked as there is no planet here! ";
			}
		}

		// And return whether the ship is to be kept
		return keepShip;
	},
	processInvasion: function(debug,events,updatePlayer,updates,hex,troopsByStyle) {
		let planetBusted = false;
		Game._stylenameToPlayerMapping[hex.owner].addTurnNote("invaded",hex.id);
		const [owner,marines] = TurnProcessor.conductGroundCombat(hex.id,troopsByStyle,debug,(updatePlayer ? updates : null),events);
		const winner = Game.FULLNAMES[owner];
		if (owner !== hex.owner) {
			Game.State.baseMap.setInterdicted(hex.id,false); // The hex is no longer interdicted
			if (debug) console.info(`${hex.id} conquered by ${winner}`);
			if (owner == Game._players[0].stylename && hex.level > 1) {
				const report = `We have conquered ${hex.id} with  ${marines} marine compan${marines == 1 ? "y" : "ies"} remaining after the combat.`;
				updates.push(report);
				Game._stylenameToPlayerMapping[owner].map.setColonyReport(hex.id,report);
			} else if (owner == Game._players[0].stylename) {
				updates.push(`We have destroyed the level 1 colony at ${hex.id}.`);
			} else if (hex.owner == Game._players[0].stylename) {
				updates.push(`We have lost ${hex.id} to ${winner}`);
			} else if (updatePlayer) {
				updates.push(`We did not conquer ${hex.id} instead ${winner} conquered it instead.`);
			}

			const originalOwner = Game._stylenameToPlayerMapping[hex.owner];
			if (originalOwner.researchLevel(TECHS.planetaryDefences) == 8) {
				planetBusted = true;
			} else {
				if (hex.level > 1) {
					if (debug) console.info(`${hex.id} conquered by ${winner}`);
					events.push(`${hex.id} conquered by ${winner}`);
					Game.State.baseMap.colonise(hex.id,owner,-1); // Changes owner and reduces the level by one
					hex.owner = owner; hex.level--; // And update the copy we're using in this turn processor
					Game.State.baseMap.setMarines(hex.id,marines);
					hex.marines = marines;
				} else {
					if (debug) console.info(`${hex.id} destroyed by ${winner}`);
					events.push(`The colony at ${hex.id} destroyed by ${winner}`);
					delete hex.owner; delete hex.level;
					Game.State.baseMap.removeColony(hex.id);
					delete hex.owner; delete hex.level; delete hex.marines; // And update the copy we're using in this turn processor
				}
			}
		} else {
			Game.State.baseMap.setMarines(hex.id,marines);
			hex.marines = marines;
		}
		return planetBusted ? winner : null;
	},
	suicidePlanetBust: function(debug,events,updatePlayer,updates,hex,winner,shipsToBeDestroyed) {
		if (debug) console.info(`${hex.id} suicide planet busted as a result of being conquered by ${winner}`);
		events.push(`${hex.id} suicide planet busted as a result of being conquered by ${winner}`);
		if (updatePlayer) {
			updates.push(`${hex.id} suicide planet busted as a result of being conquered by ${winner}`);
		}
		Game.State.baseMap.destroyPlanet(hex.id);
		Game._stylenameToPlayerMapping[hex.owner].addTurnNote("planetBusted",hex.id);
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const ship = object.ship;
				ship.report += `The colony at ${hex.id} has suicide planet busted and ${ship.name} was destroyed. `;
				if (object.owner == Game._players[0]) {
					updates.push(`The colony at ${hex.id} has suicide planet busted and ${ship.name} was destroyed.`);
				}
				shipsToBeDestroyed.push(object);
			}
		}
		delete hex.owner; delete hex.level; delete hex.marines; delete hex.planet; // And update the copy we're using in this turn processor
	},
	updateReports: function(hex,ship,owner,thereWasAPlanet) {
		// Scanning of the hex
		if (hex.planet && hex.owner) {
			const colonyString = `${ship.location} is a class ${hex.level} colony belonging to ${Game.FULLNAMES[hex.owner]}.`;
			const marineString = `There ${hex.marines == 1 ? "is" : "are"} ${hex.marines} marine compan${hex.marines == 1? "y" : "ies"} here. `;
			ship.report += colonyString + (hex.marines ? " "+marineString : "");
		} else if (hex.planet) {
			ship.report += `${ship.location} is an uncolonised world. `;
		} else if (thereWasAPlanet) {
			ship.report += `The planet at ${ship.location} has been destroyed! `;
		}

		// Presence of other ships
		ship.report += TurnProcessor.getReportOnOtherShips(hex,owner.stylename);

		// Update the hex report before the ship is destroyed (if it is going to be destroyed)
		owner.map.setLastScanned(hex.id,Game.State.turn,ship.report); // If a player has multiple ships we will set this multiple times (to the same value)
	},
	getReportOnOtherShips: function(hex,owner) {
		const bystyle = TurnProcessor.getShipsByStyle(hex);
		delete bystyle[owner];
		let report = "";
		for(let otherowner of Object.keys(bystyle)) {
			for(let size=1; size<bystyle[otherowner].length; size++) {
				if (bystyle[otherowner][size]) {
					report += "There " +
						(bystyle[otherowner][size]==1 ? "is a" : "are "+bystyle[otherowner][size]) +
						" size "+size+" ship" +
						(bystyle[otherowner][size]==1 ? "" : "s") +
						" belonging to " +
						Game.FULLNAMES[otherowner] +
						" here. ";
				}
			}
		}
		return report;
	},
	anyHostileShips: function(hex,owner) {
		for(let object of hex.objects) {
			if (object.ship && object.owner.stylename != owner && object.ship.hasWeapons(true,false)) {
				return true;
			}
		}
		return false;
	},
	resolvePossibleCombat: function(hex,bystyle,debug,updates,events,shipsToBeDestroyed) {
		const from = TurnProcessor.listPlayers(Object.keys(bystyle));

		let list = "";
		for(let i=0; i<hex.objects.length; i++) {
			if (i>0 && i<hex.objects.length-1) {
				list += ", ";
			} else if (i == hex.objects.length-1) {
				list += " and ";
			}
			list += hex.objects[i].ship.name;
		}

		// Does any ship have any weapons?
		if (hex.objects.some(object => object.ship.hasWeapons(true,false))) {
			if (updates) updates.push(`Combat!!!: In ${hex.id} there are ships from ${from}: ${list}`);
			if (debug) console.info(`Combat!!!: In ${hex.id} there are ships from ${from}: ${list}`);
			events.push(`Ship Combat in ${hex.id} involving ships from ${from}`);
			TurnProcessor.resolveCombat(hex,debug,updates,shipsToBeDestroyed);
		}
	},
	listPlayers: function(styles) {
		let from = "";
		for(let i=0; i<styles.length; i++) {
			if (i>0 && i<styles.length-1) {
				from += ", ";
			} else if (i == styles.length-1) {
				from += " and ";
			}
			from += Game.FULLNAMES[styles[i]];
		}
		return from;
	},
	whosPresent: function(hex) {
			const bystyle = {}; // each element (when present) has a sparse array showing who has ships in the hex
			for(let object of hex.objects) {
				if (object.ship && object.owner) {
					const name = object.owner.stylename;
					bystyle[name] = bystyle[name] ? bystyle[name] : true;
				}
			}
		return Object.keys(bystyle);
	},
	resolveCombat(hex,debug,updates,shipsToBeDestroyed) {
		let notified = {};
		for(let object of hex.objects) {
			if (object.ship && object.owner) {
				const name = object.owner.stylename;
				if (!notified[name]) {
					object.owner.addTurnNote("shipCombat",hex.id);
					notified[name] = true;
				}
			}
		}
		let maxPhaseNumber = 1;
		for(let object of hex.objects) {
			object.owner.addTurnNote("shipCombat",hex.id);
			maxPhaseNumber = Math.max(maxPhaseNumber, object.ship.getCombatSpeed());
			// Also do the start of combat preparations (raising shield, launching fighters)
			if (object.ship) {
				object.ship.raiseShieldsAtStartOfCombat();
				const fighters = object.ship.launchFighters();
				for(let i=0; i<fighters.count; i++) {
					const fighter = new Fighter(object.ship,fighters.spec,i+1);
					hex.objects.push({ship: fighter, owner:object.owner, fighter: true});
					if (fighter.getCombatSpeed() > maxPhaseNumber) maxPhaseNumber = fighter.getCombatSpeed();
				}
			}
		}
		if (debug) console.info(`This combat will have ${maxPhaseNumber==0?"a single":maxPhaseNumber+1} phase${maxPhaseNumber==0?"":"s"} per combat round`);
		if (updates) updates.push(`This combat will have ${maxPhaseNumber==0?"a single":maxPhaseNumber+1} phase${maxPhaseNumber==0?"":"s"} per combat round`);

		let round = 0;
		while((TurnProcessor.whosPresent(hex).length > 1) && (hex.objects.some(object => object.ship.hasWeapons(false,false)))) {
			round++;
			for(let phase=maxPhaseNumber; phase>=0; phase--) {
				let hitTargets = new Map();
				for(let object of hex.objects) {
					if (!object.ship) continue; // It's not a ship
					const owner = object.owner;
					const ship = object.ship;
					if (!ship.hasABridge()) continue; // It's lost all of its bridges
					ship.nextCombatPhase(); // Let the ship use its reflecting shields
					let targets = hex.objects.filter(other => other.ship && other.owner !== owner).map(obj => obj.ship);
					if (targets.length > 0) {
						const target = ROT.RNG.getItem(targets);
						const hits = ship.getHits(ship.getCombatSpeed() >= phase);
						if (hits > 0) {
							if (debug) console.info(`Round ${round} Phase ${phase+1}: ${ship.name} scores ${hits==1?"a hit":hits+" hits"} on ${target.name}`);
							if (updates) updates.push(`Round ${round} Phase ${phase+1}: ${ship.name} scores ${hits==1?"a hit":hits+" hits"} on ${target.name}`);
							if (hitTargets.has(target)) {
								hitTargets.set(target,hitTargets.get(target)+hits);
							} else {
								hitTargets.set(target,hits);
							}
						}
					}
				}
				for(const [target,hits] of hitTargets) {
					let resolution = target.resolveHits(hits);
					if (debug && resolution) console.info(`Round ${round} Phase ${phase+1}: ${target.name} ${resolution}`);
					if (updates && resolution) updates.push(`Round ${round} Phase ${phase+1}: ${target.name} ${resolution}`);
				}
				// Resolve destroyed ships
				let newlist = [];
				for(let object of hex.objects) {
					if (object.ship && !object.ship.hasABridge()) {
						shipsToBeDestroyed.push(object);
						if (debug) console.info(`Round ${round} Phase ${phase+1}: ${object.ship.name} has been destroyed`);
						if (updates) updates.push(`Round ${round} Phase ${phase+1}: ${object.ship.name} has been destroyed`);
					} else{
						// The ship is still in the fight
						newlist.push(object);
					}
				}
				hex.objects = newlist;
			}

			// Resolve retreats if there's still more than one side
			if ((TurnProcessor.whosPresent(hex).length > 1) && (hex.objects.some(object => object.ship.hasWeapons(true,true)))) {
				let newlist = [];
				for(let object of hex.objects) {
					if (!object.ship || object.fighter) {
						newlist.push(object); // It's still going to be there but
						continue; // It's not a ship
					}
					const owner = object.owner;
					const ship = object.ship;
					if (ship && ship.getJumpRange() > 0 && ship.shouldRetreat(hex.objects.filter(other => other.ship && other.owner == owner).map(obj => obj.ship))) {
						// Retreat the ship
						ship.retreatingTo = ship.lastHex;
						if (debug) console.info(`Round ${round}: ${ship.name} is retreating to ${ship.lastHex}`);
						if (updates) {
							updates.push(`Round ${round}: ${ship.name} is retreating to ${ship.lastHex}`);
						}
					} else {
						// The ship is still in the fight, so we'll need its surrounding object
						newlist.push(object);
					}
				}
				hex.objects = newlist;
			}
		}

		// And now return any remaining fighters back to their carrier
		let newlist = [];
		for(let object of hex.objects) {
			if (object.ship && object.fighter) {
				// This might return the fighter to a destroyed carrier - but that's OK as the fighter won't survive anyway
				object.ship.carrier.returnAFighter();
			} else {
				newlist.push(object);
			}
		}
		hex.objects = newlist;
	},
	conductGroundCombat(location,troopsByStyle,debug,updates,events) {
		if (debug) console.info(`Ground Combat at ${location}: ${TurnProcessor.listTroops(troopsByStyle)}`);
		if (updates) updates.push(`Ground Combat at ${location}: ${TurnProcessor.listTroops(troopsByStyle)}`);
		const from = TurnProcessor.listPlayers(Object.keys(troopsByStyle));
		events.push(`Ground Combat at ${location} involving ${from}`);
		// Multiply up the companies by the planetary defence tech
		let humanInvolved = false;
		for(let owner of Object.keys(troopsByStyle)) {
			troopsByStyle[owner] *= Game._stylenameToPlayerMapping[owner].troopValue;
			if (owner == Game._players[0].stylename) {
				humanInvolved = true;
			}
		}

		let report = function(round,troopsByStyle) {
			let report = `Round ${round} Effective Troops:`;
			let comma = "";
			for(let owner of Object.keys(troopsByStyle)) {
				report += `${comma} ${owner}:${troopsByStyle[owner]}`;
				comma = ",";
			}
			return report;
		};

		let round = 0;
		while(Object.keys(troopsByStyle).length > 1) {
			round++;
			if (humanInvolved) {
				updates.push(report(round,troopsByStyle));
			}
			if (debug) console.info(report(round,troopsByStyle));
			let newTroopsByStyle = Object.assign({},troopsByStyle);
			const keys = Object.keys(newTroopsByStyle);
			for(let owner of keys) {
				for(let opponent of keys) {
					if (owner !== opponent) {
						newTroopsByStyle[opponent] = Math.max(0, newTroopsByStyle[opponent] - Math.floor(troopsByStyle[owner] * (50+ROT.RNG.getPercentage()/2)/100));
					}
				}
			}
			for(let owner of keys) {
				if (newTroopsByStyle[owner] == 0) {
					delete newTroopsByStyle[owner];
				}
			}
			troopsByStyle = newTroopsByStyle;
		}
		if (humanInvolved) {
			updates.push(report(round+1,troopsByStyle));
		}
		if (debug) console.info(report(round+1,troopsByStyle));
		const winner = Object.keys(troopsByStyle)[0];
		return [winner, Math.floor(troopsByStyle[winner]/Game._stylenameToPlayerMapping[winner].troopValue)];
	},
	listTroops(troopsByStyle) {
		const keys = Object.keys(troopsByStyle);
		let list = "There ";
		for(let i=0; i<keys.length; i++) {
			const owner = keys[i];
			if (i == keys.length-1) {
				list +=" and ";
			} else if (i >0) {
				list += ", ";
			} else if (troopsByStyle[owner] == 1) {
				list += "is ";
			} else {
				list += "are ";
			}
			if (troopsByStyle[owner] == 1) {
				list += "a marine company from ";
			} else {
				list += troopsByStyle[owner]+" marine companies from ";
			}
			list += Game.FULLNAMES[owner];
		}
		return list;
	}
};