/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals module, exports, ShipDesign, ShipDesignType, Block, ShipOrder, ROT */
/* exported Ship, Fighter */
class StandingOrders {
	constructor(name) {
		this._name = name;
	}
	toJSON() {
		return this._name;
	}
	retreat(_ship,_ships) {
		throw new Error("retreat not overriden by StandingOrder."+this._name);
	}
}
StandingOrders.Engage = new StandingOrders("Engage");
StandingOrders.Engage.retreat = function(ship,ships){
	if (ship.hasWeapons(false,true)) {
		return false; // ship still has weapons, so engage
	} else if (ship.hasFighters()) {
		const carrier = ship;
		// Stay if any of our fighters remain
		return !ships.some(ship => ship instanceof Fighter && ship.carrier == carrier);
	} else {
		return true; // No weapons or fighters - run away!
	}
};
StandingOrders.Remain = new StandingOrders("Remain");
StandingOrders.Remain.retreat = function(_ship,_ships) {
	return false;
};
StandingOrders.Retreat = new StandingOrders("Retreat");
StandingOrders.Retreat.retreat =function(_ship,_ships) {
	return true;
};
StandingOrders.Escorted = new StandingOrders("Escorted");
StandingOrders.Escorted.retreat = function(_ship,ships) {
	for(let s of ships) {
		if (s.hasWeapons(false,true)) return false;
	}
	return true;
};
StandingOrders.fromJSON = function(json) {
	if (json == "Engage") return StandingOrders.Engage;
	if (json == "Remain") return StandingOrders.Remain;
	if (json == "Retreat") return StandingOrders.Retreat;
	if (json == "Escorted") return StandingOrders.Escorted;
	throw new Error(`Unknown Standing Order '${json}'`);
};

class Fighter {
	constructor(carrier,fighterSpec,idNumber) {
		this._carrier = carrier;
		this._spec = fighterSpec;
		this._destroyed = false;
		this._id = idNumber;
	}
	get carrier() {
		return this._carrier;
	}
	hasWeapons(_includeFighters,_includeShields) {
		return true;
	}
	launchFighters() {
		return {count: 0, spec: null };
	}
	hasABridge() {
		return !this._destroyed;
	}
	getCombatSpeed() {
		return this._spec.speed;
	}
	markInvulnerable(_invulnerable) {
	}
	raiseShieldsAtStartOfCombat() {
	}
	nextCombatPhase() {
	}
	getHits(canFireThisPhase) {
		let hits = 0;
		if (canFireThisPhase) {
			if (this._spec.weapon == "_wl" && ROT.RNG.getUniform() < 0.166667) hits++;
			if (this._spec.weapon == "_wp" && ROT.RNG.getUniform() < 0.333334) hits++;
			if (this._spec.weapon == "_wa" && ROT.RNG.getUniform() < 0.5) hits++;
			if (this._spec.weapon == "_wf") hits++;
			if (this._spec.weapon == "_wc") hits += 2;
			if (this._spec.weapon == "_wm") hits += 3;
			if (this._spec.weapon == "_wx") hits += 10;
		}
		return hits;
	}
	get name() {
		return `Fighter ${this._id} from ${this._carrier.name}`;
	}
	resolveHits() {
		this._destroyed = true;
		return null; // So we don't get a double report on the fighter's destruction
	}
}

class Ship {
	constructor(name,design,startLocation) {
		this._name = name;
		this._design = design;
		this._location = startLocation;
		this._order = new ShipOrder();
		this._report = "Nothing to report";
		this._blocks = null; // No cached field yet
		this._powerNeedsCalculating = true;
		this._tractoredPoints = 0;
		this._shields = 0;
		this._reflectingShields = 0;
		this._reflectedDamageThisPhase = 0;
		this._reflectedDamageNextPhase = 0;
		this._retreatingTo = null;
		this._maxMarines = this._design.getMarinesOnBoard();
		this._marinesOnBoard = this._maxMarines;
		this._maxFighters = this._design.getFightersOnBoard();
		this._fightersOnBoard = this._maxFighters;
		this._invulnerable = false;
		this._unjumpedTurns = 0;
		this._nonDredgerUnits = false;
		this._energyBalance = undefined;
		this._energyBalanceJump = undefined;
		this._standingOrders = this._defaultStandingOrders();
	}
	get standingOrders() { return this._standingOrders; }
	set standingOrders(o) { this._standingOrders = o;}
	_defaultStandingOrders() {
		return this.hasWeapons(true,true) ? StandingOrders.Engage : StandingOrders.Escorted;
	}
	get name() { return this._name; }
	get design() { return this._design; }
	get classname() { return this._design.name; }
	get location() { return this._location; }
	get report() { return this._report;  }
	set report(report) { this._report = report; }
	resetReport() { this._report = ""; }
	hasAReport() { return this._report !== ""; }
	get order() { return this._order; }
	set order(order) { this._order = order; }
	get lastHex() { return this._lastHex; }
	set lastHex(hex) { this._lastHex = hex; }
	get retreatingTo() { return this._retreatingTo; }
	set retreatingTo(hex) { this._retreatingTo = hex; }
	get unjumpedTurns() { return this._unjumpedTurns; }
	resetUnjumpedTurns() { this._unjumpedTurns = 0; }
	get nonDredgerUnits() { return this._nonDredgerUnits; }
	set nonDredgerUnits(flag) { this._nonDredgerUnits = flag; if (flag) this.resetUnjumpedTurns(); }
	makeJump(jump) {
		if (jump) {
			this._location = jump;
			this._unjumpedTurns = 0;
		} else {
			this._unjumpedTurns++;
		}
	}
	getJumpRange() {
		this._powerSystems();
		let jumpPoints = 0;
		for(let block of this._blocks) {
			if (block.key == "_ja" && block.operational) jumpPoints += 10;
			if (block.key == "_jb" && block.operational) jumpPoints += 20;
			if (block.key == "_jc" && block.operational) jumpPoints += 30;
			if (block.key == "_jd" && block.operational) jumpPoints += 40;
			if (block.key == "_je" && block.operational) jumpPoints += 50;
			if (block.key == "_jf" && block.operational) jumpPoints += 75;
			if (block.key == "_jg" && block.operational) jumpPoints += 100;
			if (block.key == "_jh" && block.operational) jumpPoints += 200;
		}
		return Math.floor(jumpPoints/this._design.totalSize());
	}
	getManeuverSpeed() {
		this._powerSystems();
		let manPoints = -this._tractoredPoints;
		for(let block of this._blocks) {
			if (block.key == "_ma" && block.operational) manPoints += 10;
			if (block.key == "_mb" && block.operational) manPoints += 20;
			if (block.key == "_mc" && block.operational) manPoints += 30;
			if (block.key == "_md" && block.operational) manPoints += 40;
			if (block.key == "_me" && block.operational) manPoints += 50;
			if (block.key == "_mf" && block.operational) manPoints += 75;
			if (block.key == "_mg" && block.operational) manPoints += 100;
			if (block.key == "_mh" && block.operational) manPoints += 200;
		}
		if (manPoints < 0) { manPoints = 0; }
		return Math.floor(manPoints/this._design.totalSize());
	}
	getCombatSpeed() {
		return this.getManeuverSpeed() + this.getBestComputer();
	}
	getBestComputer() { // Note that a return value of -1 indicates that there are no operational bridges on the ship
						// And hence it should be destroyed
		this._powerSystems();
		let best = -1; let neuralNet = false;
		for(let block of this._blocks) {
			if (block.key == "_b0" && block.operational) best = 0;
			if (block.key == "_b1" && block.operational) best = 1;
			if (block.key == "_b2" && block.operational) best = 2;
			if (block.key == "_b3" && block.operational) best = 3;
			if (block.key == "_b4" && block.operational) best = 4;
			if (block.key == "_b5" && block.operational) best = 5;
			if (block.key == "_b6" && block.operational) best = 6;
			if (block.key == "_b7" && block.operational) best = 7;
			if (block.key == "_b8" && block.operational) best = 8;
			if (block.key == "_b9" && block.operational) best = 9;
			if (block.key == "_an" && block.operational) neuralNet = true;
		}
		if (best == -1) return 0;
		if (this._design.type === ShipDesignType.SpaceBase) {
			best++;
		}
		if (this._design.type === ShipDesignType.StarBase) {
			best = (best+1)*2;
		}
		if (neuralNet) {
			best *= 2;
		}
		return best;
	}
	hasABridge() {
		this._powerSystems();
		for(let block of this._blocks) {
			if (block.key == "_b0" && block.operational) return true;
			if (block.key == "_b1" && block.operational) return true;
			if (block.key == "_b2" && block.operational) return true;
			if (block.key == "_b3" && block.operational) return true;
			if (block.key == "_b4" && block.operational) return true;
			if (block.key == "_b5" && block.operational) return true;
			if (block.key == "_b6" && block.operational) return true;
			if (block.key == "_b7" && block.operational) return true;
			if (block.key == "_b8" && block.operational) return true;
			if (block.key == "_b9" && block.operational) return true;
		}
	}
	getColonyImprovement() {
		this._powerSystems();
		let colonies = 0;
		for(let block of this._blocks) {
			if (block.key == "_cb" && block.operational) colonies++;
		}
		return colonies;
	}
	canPlanetBust() {
		this._powerSystems();
		for(let block of this._blocks) {
			if (block.key == "_wb" && block.operational) return true;
		}
		return false;
	}
	getMaxDredgers() {
		return this._design.getDredgersOnBoard();
	}
	getDredgers() {
		this._powerSystems();
		let dredgers = 0;
		for(let block of this._blocks) {
			if (block.key == "_ad" && block.operational) dredgers++;
		}
		return dredgers;
	}
	hasWeapons(includeFighters,includeShields) {
		this._powerSystems();
		for(let block of this._blocks) {
			if (block.key == "_wl" && block.operational) return true;
			if (block.key == "_wp" && block.operational) return true;
			if (block.key == "_wa" && block.operational) return true;
			if (block.key == "_wf" && block.operational) return true;
			if (block.key == "_wr" && block.operational) return true;
			if (block.key == "_wc" && block.operational) return true;
			if (block.key == "_ws" && block.operational) return true;
			if (block.key == "_wm" && block.operational) return true;
			if (block.key == "_wt" && block.operational) return true;
			if (block.key == "_wx" && block.operational) return true;
			if (block.key == "_wb" && block.operational) return true;
			if (includeShields && block.key == "_dr" && block.operational && (this._reflectingShields > 0 || this._reflectedDamageNextPhase > 0)) return true;
			if (block.key == "_df" && block.operational && includeFighters) return true;
		}
		return false;
	}
	hasFighters() {
		this._powerSystems();
		for(let block of this._blocks) {
			if (block.key == "_df" && block.operational) return true;
		}
		return false;
	}
	getHits(canFireThisPhase) {
		let hits = this._reflectedDamageThisPhase; // reflected damage is always scored regardless of whether the attacking ship can fire its actual weapons
		if (canFireThisPhase) {
			this._powerSystems();
			for(let block of this._blocks) {
				if (block.key == "_wl" && block.operational && ROT.RNG.getUniform() < 1/6) hits++;
				if (block.key == "_wp" && block.operational && ROT.RNG.getUniform() < 1/3) hits++;
				if (block.key == "_wa" && block.operational && ROT.RNG.getUniform() < 1/2) hits++;
				if (block.key == "_wf" && block.operational) hits++;
				if (block.key == "_wr" && block.operational) hits += 10;
				if (block.key == "_wc" && block.operational) hits += 2;
				if (block.key == "_ws" && block.operational) hits += 20;
				if (block.key == "_wm" && block.operational) hits += 3;
				if (block.key == "_wt" && block.operational) hits += 30;
				if (block.key == "_wx" && block.operational) hits += 10;
				if (block.key == "_wb" && block.operational) hits += 1000000; // A million should be enough to destroy vulnerable targets
			}
		}
		return hits;
	}
	getAverageHitsIgnoringFighters() {
		let hits = 0;
		this._powerSystems();
		for(let block of this._blocks) {
			if (block.key == "_wl" && block.operational) hits += 1/6;
			if (block.key == "_wp" && block.operational) hits += 1/3;
			if (block.key == "_wa" && block.operational) hits += 1/2;
			if (block.key == "_wf" && block.operational) hits += 1;
			if (block.key == "_wr" && block.operational) hits += 10;
			if (block.key == "_wc" && block.operational) hits += 2;
			if (block.key == "_ws" && block.operational) hits += 20;
			if (block.key == "_wm" && block.operational) hits += 3;
			if (block.key == "_wt" && block.operational) hits += 30;
			if (block.key == "_wx" && block.operational) hits += 10;
			if (block.key == "_wb" && block.operational) hits += 1000000; // A million should be enough to destroy vulnerable targets

			if (block.key == "_dr" && block.operational) hits += 5; // We'll assume that the reflecting shield has always has half damage to reflect
		}
		return Math.floor(hits*100+0.5)/100; // Give a rounded value
	}
	getRepairCost() {
		this._cacheBlocks();
		let cost = this._blocks.reduce((cost,block) => cost+block.repairCost(), 0);
		if (this._maxFighters > 0) {
			cost += (this._maxFighters-this._fightersOnBoard) * this._design.fighterSpec().cost;
		}
		return cost;
	}
	getDestroyedBlocks() {
		this._cacheBlocks();
		return this._blocks.filter(block => block.repairCost()>0);
	}
	resolveHits(hits) {
		let report = "";
		// Save damage from reflecting shields
		if (this._reflectingShields) {
			const reflectedThisHit = Math.min(hits,this._reflectingShields);
			this._reflectedDamageNextPhase += reflectedThisHit;
			this._reflectingShields -= reflectedThisHit;
			hits -= this._reflectedDamageNextPhase;
			report += `reflects ${reflectedThisHit} hit${this._reflectedDamageNextPhase==1?"":"s"} (leaving ${this._reflectingShields} reflecting shield point${this._reflectingShields==1?"":"s"}). `;
		}
		if (this._invulnerable && hits>0) {
			report += `${hits} point${hits==1?"":"s"} of damage ignored as this ship is invulnerable to damage this round. `;
			hits = 0;
		}
		// Take the damage on the shields if possible
		if (this._shields && hits>0) {
			const points = Math.min(hits,this._shields);
			this._shields -= points;
			hits -= points;
			report += `takes ${points} point${points==1?"":"s"} of damage on its shields (leaving ${this._shields} shield point${this._shields==1?"":"s"} remaining). `;
		}
		hitLoop: while(hits > 0 && this.hasABridge()) {
			// First take out any armour
			for(let block of this._blocks) {
				if (block.key == "_da" && block.operational) {
					block.addDamage();
					hits--;
					report += "Armour block destroyed. ";
					continue hitLoop;
				}
			}

			// OK then, now hit a random block (remembering that we need to account for large blocks)
			let choices = [];
			for(let block of this._blocks) {
				for(let i=0; i<block.size; i++) {
					if (!block.destroyed) {
						choices.push(block);
					}
				}
			}
			let choice = ROT.RNG.getItem(choices);
			choice.addDamage();
			if (choice.key == "_bb") {
				this._maxMarines--;
				this._marinesOnBoard = Math.min(this._marinesOnBoard,this._maxMarines);
			}
			if (choice.key == "_df") {
				this._maxFighters--; // No need to adjust the actual fighter count - we're in combat and hence they've been launched.
			}
			hits--;
			this._powerNeedsCalculating = true;
			report += `${choice.name} hit. `;
		}
		return report;
	}
	repairAll() {
		this._cacheBlocks();
		for(let block of this._blocks) {
			block.repair();
		}
		this._maxMarines = this._design.getMarinesOnBoard(); // Barracks repaired (but the troops are not)
		this._maxFighters = this._design.getFightersOnBoard();
		this._powerNeedsCalculating = true;
	}
	launchFighters() {
		let count = this._fightersOnBoard;
		this._fightersOnBoard = 0;
		return {count: count, spec: this._design.fighterSpec()};
	}
	getFighterCount() {
		return this._fightersOnBoard;
	}
	getMaxFighters() {
		return this._maxFighters;
	}
	returnAFighter() {
		if (this._fightersOnBoard < this._maxFighters) {
			this._fightersOnBoard++;
		}
	}
	raiseShieldsAtStartOfCombat() {
		this._powerSystems();
		this._shields = 0;
		this._reflectingShields = 0;
		this._reflectedDamageThisPhase = 0;
		this._reflectedDamageNextPhase = 0;
		for(let block of this._blocks) {
			if (block.key == "_ds" && block.operational) { this._shields += 10; }
			if (block.key == "_di" && block.operational) { this._shields += 20; }
			if (block.key == "_dr" && block.operational) { this._reflectingShields += 10; }
		}
	}
	raiseShieldsAtStartOfRound() {
		this._powerSystems();
		this._invulnerable = false;
		for(let block of this._blocks) {
			if (block.key == "_dv" && block.operational) this._invulnerable = true;
		}
		let need = true;
		for(let block of this._blocks) {
			if (this._invulnerable) {
				if (need && block.key == "_dv" && block.operational) {
					block.addDamage();
					need = false;
				}
			} else if (block.key == "_dg" && block.operational) {
				this._shields += 10;
			}
		}
	}
	// Note: the following method also raises the shield as if we're at the start of a combat and round - it's designed for use by the shipdesign on a dummy ship
	getTotalShields() {
		this.raiseShieldsAtStartOfCombat();
		this.raiseShieldsAtStartOfRound();
		return this._shields;
	}
	makeVulnerable() {
		this._invulnerable = false;
	}
	shouldRetreat(ships) {
		return this._standingOrders.retreat(this,ships);
	}
	nextCombatPhase() {
		this._reflectedDamageThisPhase = this._reflectedDamageNextPhase;
		this._reflectedDamageNextPhase = 0;
	}
	get size() {
		return this._design.totalSize();
	}
	get price() {
		return this._design.totalCost();
	}
	getScrapValue() {
		return Math.floor((this._design.totalCost() - this.getRepairCost())*0.4);
	}
	get marines() {
		return this._marinesOnBoard;
	}
	set marines(marines) {
		this._marinesOnBoard = marines;
	}
	getMaxMarines() {
		return this._maxMarines;
	}
	setTractor(points) {
		this._tractoredPoints = points;
	}
	_powerSystems() {
		if (this._powerNeedsCalculating) {
			this._cacheBlocks();
			this._energyBalance = 0;
			this._energyBalanceJump = 0;
			// Compute the power generated
			for(let block of this._blocks) {
				if (block.eps > 0) {
					block.setPowered(true);
					if (block.operational) this._energyBalance += block.eps;
				}
				if (block.epsj > 0) {
					block.setPowered(true);
					if (block.operational) this._energyBalanceJump += block.epsj;
				}
			}
			// And then power blocks (in block order, so the bridge first then drives followed by weapons and others)
			for(let block of this._blocks) {
				// Initially power the block (so that it will be operational (if not damaged) when we make the test)
				let need = (block.eps == "*50" ? -50 : block.eps); let needJump = block.epsj;
				if (need < 0) {
					block.setPowered(true);
					if (block.operational && (-need) <= this._energyBalance) {
						this._energyBalance += need;
					} else {
						block.setPowered(false);
					}
				}
				if (needJump < 0) {
					block.setPowered(true);
					if (block.operational && (-needJump) <= this._energyBalanceJump) {
						this._energyBalanceJump += needJump;
					} else {
						block.setPowered(false);
					}
				}
				if (need == 0 && needJump == 0) {
					block.setPowered(true); // Since it doesn't need power
				}
			}
			this._powerNeedsCalculating = false;
		}
	}
	getEnergyBalance() {
		this._powerSystems();
		return this._energyBalance;
	}
	getEnergyBalanceJump() {
		this._powerSystems();
		return this._energyBalanceJump;
	}
	getBlocks() {
		this._powerSystems();
		return this._blocks;
	}
	_cacheBlocks() {
		// Need to deep clone the design's blocks so that we have our own copy
		if (!this._blocks) {
			this._blocks = [];
			for(let block of this._design.getDesignBlocks()) {
				this._blocks.push(new Block(block));
			}
		}
	}
	toJSON() {
		const data = {name: this._name,
			designId: this._design.id,
			loc: this._location,
			order: this._order,
			report: this._report,
			damage: null,
			retreatTo: this._retreatingTo,
			marines: this._maxMarines==0 ? null : this._marinesOnBoard+1, // To avoid the falsiness of 0
			fighters: this._maxFighters==0 ? null : this._fightersOnBoard+1, // Again to avoid falsiness of 0
			unjumped: this._unjumpedTurns,
			standing: this._standingOrders.toJSON()
		};
		if (this._retreatingTo) {
			data.retreatTo = this._retreatingTo;
		}
		if (this._blocks && this._blocks.some(block => block.damage > 0)) {
			data.damage = "";
			for(let block of this._blocks) {
				if (block.damage === 0) {
					data.damage += " ";
				} else if (block.damage < 10) {
					data.damage += String.fromCharCode(48+block.damage);
				} else {
					data.damage += String.fromCharCode(54+block.damage); // @-sign=10, A-Z=11-36 etc up to r=50
				}
			}
		}
		return data;
	}
}
Ship.fromJSON = function(data) {
	const ship =  new Ship(data.name, ShipDesign.REGISTRY[data.designId],data.loc);
	ship._order = ShipOrder.fromJSON(data.order);
	ship._report = data.report;
	if (data.damage) {
		ship._cacheBlocks();
		for(let i=0; i<data.damage.length; i++) {
			if (data.damage[i] == " ") {
				// Noop
			} else if (data.damage[i] > "0" && data.damage[i] <= "9") {
				ship._blocks[i].damage = data.damage.charCodeAt(i)-48;
				if (ship._blocks[i].key == "_bb") {
					ship._maxMarines--;
				}
				if (ship._blocks[i].key  == "_df") {
					ship._maxFighters--;
				}
			} else {
				ship._blocks[i].damage = data.damage.charCodeAt(i)-54;
			}
		}
	}
	if (data.retreatTo) {
		ship._retreatingTo = data.retreatTo;
	}
	if (data.marines) {
		ship._marinesOnBoard = data.marines-1;
	}
	if (data.fighters) {
		ship._fightersOnBoard = data.fighters-1;
	}
	if (data.standing) {
		ship._standingOrders = StandingOrders.fromJSON(data.standing);
	} // else the default from the ship's consstruction will do fine!
	ship._unjumpedTurns = data.unjumped;
	ship._powerNeedsCalculating = true;
	return ship;
};

//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
    exports.Ship = Ship;
    exports.Fighter = Fighter;
}