/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals ALLTECHS, TECHS, GameMap, ShipDesign, Ship */
/* exported Player */

// Note: this class represents the state of a player - it doesn't determine orders etc.
class Player {
	constructor(stylename,capital,indexOfFavouredTech,baseSeed) {
		this._stylename = stylename;
		this._shipdesignid = 1;
		if (baseSeed) { this._seed = baseSeed + Player.OFFSET[stylename]; }
		this._ROTRNGState = "";
		this._indexOfFavouredTech = indexOfFavouredTech;
		this._tech = ALLTECHS[indexOfFavouredTech].option;
		this._carriedProduction = 0;
		this._research = Array.from(ALLTECHS, _tech => { return 0; });
		this._production = 400;
		this._shipDesigns = [];
		this._ships = [];
		this._hasBuiltAShip = false;
		this._capital = capital;
		this._map = null;
		this._active = true;
		this._state = null;
		this._commsGridComputed = false;
		this._turnNotes= [];
	}
	init(baseMap,turn) {
		this._map = new GameMap(baseMap);
		const stylename = this._stylename;
		const map = this._map;
		map.forEachHex(function(hex) {
			if (hex.owner == stylename) {
				map.setLastScanned(hex.id,turn,"");
			}
		});

	}
	get state() {
		return this._state;
	}
	set state(s) {
		this._state = s;
	}
	getNextDesignId(mood) {
		return this._stylename + "-" + (mood ? mood+"-" : "") + this._shipdesignid++;
	}
	get designid() {
		return this._shipdesignid;
	}
	set designid(id) {
		this._shipdesignid = id;
	}
	get active() {
		return this._active;
	}
	deactivate() {
		this._active = false;
		this._ships = []; // Allow the ships to disappear (in effect on the next turn)
	}
	get indexOfFavouredTech() {
		return this._indexOfFavouredTech;
	}
	get map() {
		const map = this._map;
		const self = this;
		if (!this._commsGridComputed) {
			// Step 1 - Find all the hexes belonging to the owner
			let ownedHexes = [];
			map.forEachHex(hex => {
				if (hex.owner == self.stylename) {
					ownedHexes.push(hex.id);
				}
			});

			// Step 2 check each hex to see if it is in range of any of the owned hexes
			const commsRange = (this.researchLevel(TECHS.communications) == 8 ? 40 : this.researchLevel(TECHS.communications));
			for(let row=1; row<21; row++) {
				for(let column of "ABCDEFGHIJKLMNOP") {
					const name = column+row;
					map[name] = map[name] ? map[name] : {};
					map[name].onGrid = false;
					for(let owned of ownedHexes) {
						if (GameMap.range(name,owned) <= commsRange) {
							map[name].onGrid = true;
							break; // We don't need to look at any of the other owned hexes
						}
					}
				}
			}

			// And we're done
			this._commsGridComputed = true;
		}
		return map;
	}
	get stylename() {
		return this._stylename;
	}
	get capital() {
		return this._capital;
	}
	get production() {
		return this._production;
	}
	set production(p) {
		this._production = p;
	}
	get carriedProduction() {
		return this._carriedProduction;
	}
	set carriedProduction(cp) {
		this._carriedProduction = cp;
	}
	get tech() {
		return this._tech;
	}
	get troopValue() {
		const mults = [undefined, 1, 2, 5, 10, 20, 50, 100, 250];
		return mults[this.researchLevel(TECHS.planetaryDefences)];
	}
	get shipDesigns() {
		return this._shipDesigns;
	}
	get hasBuiltAShip() {
		return this._hasBuiltAShip;
	}
	get turnNotes() {
		return this._turnNotes;
	}
	addTurnNote(type,loc,extras) {
		let note = {event: type, location: loc};
		if (extras) Object.assign(note,extras);
		this._turnNotes.push(note);
	}
	resetReports() {
		const map = this._map;
		map.forEachHex(hex => map.setColonyReport(hex.id,null));
		this._turnNotes = [];
	}
	resetCommsGrid() {
		this._commsGridComputed = false;
	}
	addShipDesign(design) {
		this._shipDesigns.push(design);
	}
	getNumberOfDesigns() {
		return this._shipdesignid;
	}
	addShip(ship) {
		ship.report = "Construction completed at "+ship.location;
		this._ships.push(ship);
		this._hasBuiltAShip = true;
	}
	destroyShip(destroyedShip) {
		this._ships = this._ships.filter(x => x !== destroyedShip);
	}
	get ships() {
		return this._ships;
	}
	getShipTechs() {
		return { power:		this.researchLevel(TECHS.powerPlant),
				 jump:		this.researchLevel(TECHS.jumpDrive),
				 man:		this.researchLevel(TECHS.maneuverDrive),
				 size:		this.researchLevel(TECHS.shipSize),
				 weapons:	this.researchLevel(TECHS.weaponary),
				 defences:	this.researchLevel(TECHS.shipDefences),
				 computer:	this.researchLevel(TECHS.computers),
				 arch:		this.researchLevel(TECHS.archaeology),
				 planet:	this.researchLevel(TECHS.planetaryDefences)};
	}

	researchLevel(tech) {
		return ALLTECHS[tech.index].level(this._research[tech.index]);
	}
	addResearch(tech,points) {
		let oldLevel = this.researchLevel(tech);
		this._research[tech.index] += parseInt(points);
		let newLevel = this.researchLevel(tech);
		if (tech == TECHS.communications && newLevel > oldLevel) {
			this._commsGridComputed = false; // Invalidates the comms grid
		}

		return newLevel > oldLevel;
	}
	forEachTech(f) {
		let self = this;
		ALLTECHS.forEach(function(tech,index) {
			f(tech,self._research[index]);
		});
	}
	get seed() {
		if (this._seed) {
			return this._seed;
		} else {
			return undefined;
		}
	}
	set seed(seed) {
		this._seed = seed;
	}
	get ROTRNGState() {
		return this._ROTRNGState;
	}
	set ROTRNGState(state) {
		this._ROTRNGState = state;
	}
	toJSON() {
		let data = {
			tech: this._indexOfFavouredTech,
			cf: this._carriedProduction,
			r: this._research,
			prod: this._production,
			bas: this._hasBuiltAShip,
			cap: this._capital,
			map: this._map,
			offset: this._shipdesignid,
			designs: [],
			ships: [],
			ROT: this._ROTRNGState,
			notes: JSON.stringify(this._turnNotes)
		};
		if (!this._active) {
			data.dead = true;
		}
		if (this._seed) {
			data.seed = this._seed;
		}
		if (this._state) {
			if (typeof this._state === 'string')
				data.state = this._state;
			else
				data.j = JSON.stringify(this._state);
		}
		for(let design of this._shipDesigns) {
			data.designs.push(design);
		}
		for(let ship of this._ships) {
			data.ships.push(ship);
		}
		return data;
	}
	fromJSON(data,baseMap) {
		this._active = (data.dead ? false : true); // To allow active players to omit the field
		if (data.tech) { this._indexOfFavouredTech = data.tech; }
		this._production = data.prod;
		this._carriedProduction = data.cf;
		this._research = data.r;
		this._hasBuiltAShip = data.bas;
		if (data.seed) this._seed = data.seed;
		if (data.state) this._state = data.state;
		if (data.j) this._state = JSON.parse(data.j);
		if (data.notes)
			this._turnNotes = JSON.parse(data.notes);
		else
			this._turnNotes = [];
		this._ROTRNGState = data.ROT;
		this._capital = data.cap;
		if (data.map) {
			this._map = new GameMap(data.map,baseMap);
		}
		this._shipdesignid = data.offset;
		for(let design of data.designs) {
			this._shipDesigns.push(ShipDesign.fromJSON(design));
		}
		for(let ship of data.ships) {
			this._ships.push(Ship.fromJSON(ship));
		}
		this._commsGridComputed = false; // Comms grid needs recomputing
	}
}
Player.OFFSET = {'player': 2000, 'alkari': 100, 'auron':200, 'caffe':300, 'clanger':400, 'drahmini':500,
				'gobb':600, 'iyanden':700, 'jerandi':800, 'kalli':900, 'kelle':1000, 'pican':1100,
				'regul':1200, 'saggitt':1300, 'strata':1400, 'fleet':1500,'worati':1600
};