/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals ROT:true, Game:true, require, module, exports */
/* exported GameMap */

if (typeof(require) !== 'undefined') {
	Game = require('./game.js').Game;
	ROT = require('../rot/rot.min.js');
}

class GameMap {
	constructor(m,baseMap,avoidNeighbours) {
		if (m instanceof GameMap) {
			// Creating a player map and the 'm' is the baseMap
			this._baseMap = m;
		} else if (typeof m === 'object') {
			// We assume that the object came from  JSON
			if (baseMap) {
				this._baseMap = baseMap; // We were also given a basemap, so we can use that as the basemap
			} else {
				this._baseMap = this; // This is the base map
			}
			// copy across the details from the JSON version of the map
			const us = this;
			this.forEachHex(function(hex) {
				if (m[hex.id]) {
					us[hex.id] = m[hex.id];
				}
			});
		} else {
			// New base map
			this._baseMap = this; // We are the base map
			let possibles = {};
			let possibleCapitals = {};
			this.forEachHex(function(hex){
				const col = hex.id[0].charCodeAt() - 'A'.charCodeAt() + 1;
				const row = parseInt(hex.id.slice(1));
				const distance = Math.sqrt(Math.abs(row-10.5)*Math.abs(row-10.5) + Math.abs(col-8.5)*Math.abs(col-8.5) );
				possibles[hex.id] = 15-distance;
				possibleCapitals[hex.id] = possibles[hex.id];
			});
			for(let i=0; i<100; i++) {
				const choice = (i < Game.STYLENAMES.length ? ROT.RNG.getWeightedValue(possibleCapitals) : ROT.RNG.getWeightedValue(possibles));
				this._baseMap[choice] = {}; // Ensure there is an object at this hex
				this._baseMap[choice].planet = true;
				if (i < Game.STYLENAMES.length) {
					this._baseMap[choice].owner = Game.STYLENAMES[i];
					this._baseMap[choice].level = 20;
					this._baseMap[choice].capital = true;
					this._baseMap[choice].marines = 0;
				}
				delete possibles[choice];
				delete possibleCapitals[choice];
				if (avoidNeighbours) {
					const selfPossibleCapitals = possibleCapitals;
					const selfChoice = choice;
					const selfMap = GameMap;
					GameMap.forEachHexId(id => {
						if (selfMap.range(selfChoice,id) == 1) {
							delete selfPossibleCapitals[id];
						}
					});
				}
			}
		}
	}
	get baseMap() {
		return this._baseMap;
	}
	hex(name) {
		return Object.assign({planet: false, id: name, objects: []}, this._baseMap[name], this[name]);
	}
	improveColony(colony) {
		this._baseMap[colony].level++;
	}
	addMarines(colony,companies) {
		this._baseMap[colony].marines += parseInt(companies);
	}
	setMarines(colony,companies) {
		this._baseMap[colony].marines = companies;
	}
	addStarBase(location,owner) {
		this.setOwner(location,owner);
		this._baseMap[location].starbase = this._baseMap[location].starbase ? this._baseMap[location].starbase+1 : 1;
		this._baseMap[location].level = this._baseMap[location].level ? this._baseMap[location].level : 0; // Default to zero if there's no base here
	}
	removeStarBase(location) {
		if (this._baseMap[location].starbase) {
			this._baseMap[location].starbase--;
			if (this._baseMap[location].starbase == 0) {
				delete this._baseMap[location].starbase;
				if (!this._baseMap[location].planet) {
					delete this._baseMap[location].level;
					delete this._baseMap[location].owner;
				}
			}
		}
		this._baseMap[location].level = this._baseMap[location].level ? this._baseMap[location].level : 0; // Default to zero if there's no base here
	}
	forEachHex(f) {
		const self = this;
		for(let row=1; row<21; row++) {
			for(let column of "ABCDEFGHIJKLMNOP") {
				const name = column+row;
				f(self.hex(name));
			}
		}
	}
	toJSON() {
		let data = {};
		for(let row=1; row<21; row++) {
			for(let column of "ABCDEFGHIJKLMNOP") {
				const name = column+row;
				if (this[name]) data[name] = Object.assign({},this[name]);
				if (data[name]) delete data[name].onGrid; // Remove comms grid from the clone - we don't need to save it
				if (data[name] && Object.keys(data[name]).length == 0) delete data[name]; // And remove the hex's data if the comms grid was the only entry
			}
		}
		return data;
	}
	setLastScanned(hexname,turn,report) {
		this[hexname] = this[hexname] ? this[hexname] : {};
		this[hexname].lastScanned = turn;
		this[hexname].lastReport = report;
		if (this._baseMap[hexname] && this._baseMap[hexname].owner) {
			this[hexname].owner = this._baseMap[hexname].owner; // So that we don't change colour of a hex unless we scan it again
		} else {
			this[hexname].owner = "none"; // So that we don't change colour of a hex unless we scan it again
		}
	}
	setInterdicted(hexname,flag) {
		if (flag) {
			this[hexname] = this[hexname] ? this[hexname] : {};
			this._baseMap[hexname].interdicted = true;
		} else if (this._baseMap[hexname]) {
			delete this._baseMap[hexname].interdicted;
		}
	}
	setColonyReport(hexname,report) {
		this[hexname] = this[hexname] ? this[hexname] : {};
		this[hexname].colonyReport = report;
	}
	setOwner(hexname,o) {
		this._baseMap[hexname] = this.baseMap[hexname] ? this.baseMap[hexname] : {};
		this._baseMap[hexname].owner = o;
	}
	colonise(hexname,owner,startlevel) {
		// assert this._baseMap[hexname] must exist since it should be a planet!
		this._baseMap[hexname].owner = owner;
		this._baseMap[hexname].level = this._baseMap[hexname].level ? (this._baseMap[hexname].level+startlevel) : startlevel;
		this._baseMap[hexname].marines = 0;
	}
	removeColony(hexname,includeStarBase) {
		if (includeStarBase) {
			delete this._baseMap[hexname].starbase;
		}
		if (!this._baseMap[hexname].starbase) {
			delete this._baseMap[hexname].owner;
			delete this._baseMap[hexname].level;
		} else {
			this._baseMap[hexname].level = 0; // No longer a colony here (but it is still owned)
		}
		delete this._baseMap[hexname].marines;
		delete this._baseMap[hexname].capital;
	}
	destroyPlanet(hexname) {
		this.removeColony(hexname);
		delete this._baseMap[hexname].planet;
	}
	addObject(hexname,object) {
		this[hexname] = this[hexname] ? this[hexname] : {};
		this[hexname].objects = this[hexname].objects ? this[hexname].objects : [];
		if (!this[hexname].objects.includes(object)) {
			this[hexname].objects.push(object);
		}
	}
}

GameMap.forEachHexId = function(f) {
	for(let row=1; row<21; row++) {
		for(let column of "ABCDEFGHIJKLMNOP") {
			const name = column+row;
			f(name);
		}
	}
};

GameMap.range = function(src,dst) {
	const a0 = src.charCodeAt(0), b0 = parseInt(src.slice(1));
	const a1 = dst.charCodeAt(0), b1 = parseInt(dst.slice(1));
	const q0 = a0, r0=b0-(a0+(a0&1))/2;
	const q1 = a1, r1=b1-(a1+(a1&1))/2;
	return (Math.abs(q0-q1) + Math.abs(q0+r0-q1-r1) + Math.abs(r0-r1))/2;
};
//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
	exports.GameMap = GameMap;
}