/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals require, exports, module */
/* globals ROT, GameMap, Player, Util, console, ALLTECHS:true */
/* globals TurnProcessor, Order, NullOrder, PlayerOrders */
/* globals ShipOrderReport, ShipDesign, ShipDesigner, ShipOrder */
/* exported stylenames */

if (typeof(require) !== 'undefined') {
	ALLTECHS = require('./tech.js').ALLTECHS;
}

const InitialState = { version: "46", minimumVersion: "43", turn: 1 }; /* Initial Game State - version indicates which version of save data we create and can read */
const stylenames = ['player', 'alkari', 'auron', 'caffe', 'clanger', 'drahmini', 'gobb', 'iyanden', 'jerandi', 'kalli', 'kelle',
	'pican', 'regul', 'saggitt', 'strata', 'fleet', 'worati'];
const defaultTechIndexes = [0, 8, 9, 6, 6, 5, 3, 1, 8, 8, 3, 9, 4, 5, 3, 2, 0];
var Game = {
	STYLENAMES: stylenames,
	FULLNAMES: {'player': "you",
		/*1*/	'alkari': "the Alkari",
		/*2*/	'auron':"the Aurons",
		/*3*/	'caffe':"the Caffenians",
		/*4*/	'clanger':"the Clangers",
		/*5*/	'drahmini':"the Drahmini",
		/*6*/	'gobb':"the Gobbannanean Principalities",
		/*7*/	'iyanden':"the Chroniomancy of Iyanden",
		/*8*/	'jerandi':"the Jerandi People's Republic",
		/*9*/	'kalli':"the Kallikanzaroi",
		/*10*/	'kelle':"the Kellesonians",
		/*11*/	'pican':"the Picans",
		/*12*/	'regul':"the Regul",
		/*13*/	'saggitt':"the Saggittarians",
		/*14*/	'strata':"Strata Inc.",
		/*15*/	'fleet':"the Fleet",
		/*16*/	'worati':"the Worati"
	},
	State: { version: undefined, baseMap: undefined, turn: 0, playerProduction: undefined },
	_standardShipDesigns: [],
	_players: [],
	_replayTurn: 1,
	_stylenameToPlayerMapping: {},
	_computers: {},
	_orders:{},
	_gameOver: false,
	History: { avt: new Array(ALLTECHS.length).fill(1), baseMap: {}, lastTurn:0 },
	ARTIFACT_DESIGN: null,
	_showMap: function(map) {
		map.forEachHex(function(hex) {
			const hexelement = document.getElementById(hex.id);
			if (hex.lastScanned) {
				const planetElement = hexelement.childNodes[3];
				planetElement.classList.remove("starbase","planet","capital");
				planetElement.classList.add("emptyspace");
				if (hex.starbase) {
					planetElement.classList.remove("emptyspace");
					planetElement.classList.add("starbase");
				}
				if (hex.planet) {
					planetElement.classList.remove("emptyspace");
					if (hex.level > 19) {
						planetElement.classList.add("capital");
					} else {
						planetElement.classList.add("planet");
					}
				}
				hexelement.classList.remove(...stylenames);
				if (hex.owner && hex.owner !== "none") {
					hexelement.classList.add(hex.owner);
				}
				let text = "";
				hexelement.title = text + "Last report from Turn " + hex.lastScanned;
				if (hex.lastReport) {
					hexelement.title += "\n"+hex.lastReport;
				}
				hexelement.classList.remove("fogofwar");
			} else {
				hexelement.classList.add("fogofwar");
			}
		});
	},
	_generateNewGame: function() {
		const avoidNeighbours = (document.getElementById("mapGeneration").selectedIndex == 1);
		delete localStorage.domainOfTheAncientsGameOver;
		Game.State = Object.assign({}, InitialState);
		Game.State.baseMap = new GameMap(undefined,undefined,avoidNeighbours);
		Game.History = { baseMap: {}, version: Game.State.version };
		Game.State.baseMap.forEachHex(function(hex) {
			if (hex.planet) Game.History.baseMap[hex.id] = true;
		}); /* basemap shows where the planets are */

		// Now create the players
		Game._createPlayers(document.getElementById("favouredTech").selectedIndex,document.getElementById("computerTechs").selectedIndex);

		document.getElementById("welcome").classList.add("hidden");
		document.getElementById("summary").classList.remove("hidden");
		PlayerOrders.setPlayer(Game._players[0]);
		PlayerOrders.setStandardShipDesigns(Game._standardShipDesigns);
		Game.State.turn = 0;
		Game._updateHistory(["Start of a new game"]);
		Game._newTurn();
	},
	_createPlayers: function(playerTechIndex,randomCompTechs) {
		let techs = [...defaultTechIndexes];
		techs[0] = playerTechIndex ? playerTechIndex-1 : Math.floor(ROT.RNG.getUniform()*ALLTECHS.length);
		if (randomCompTechs) {
			for(let i=1; i<Game.STYLENAMES.length; i++) {
				techs[i] = Math.floor(ROT.RNG.getUniform()*ALLTECHS.length);
			}
		}
		Game._createPlayersWithTech(techs);
	},
	_createPlayersWithTech: function(techs) {
		const seed = ROT.RNG.getSeed();
		const capital = {};
		Game.State.baseMap.forEachHex(hex => {
			if (hex.owner) {
				capital[hex.owner] = hex.id;
			}
		});
		Game._players = [];
		Game._stylenameToPlayerMapping = {};
		for(let i=0; i< Game.STYLENAMES.length; i++) {
			let player = new Player(Game.STYLENAMES[i],capital[Game.STYLENAMES[i]],techs[i],seed);
			player.init(Game.State.baseMap,Game.State.turn);
			Game._players.push(player);
			Game._stylenameToPlayerMapping[Game.STYLENAMES[i]] = player;
		}
	},
	_newTurn: function() {
		Game.State.turn++;
		Game._updateProduction();
		Game._saveGameState();
		Game._startTurn();
	},
	_startTurn: function() {
		Game._pokeComputerWorkers();
		ShipOrderReport.showShipReports(Game._players[0],Game.State.baseMap); // Needed for the report on dredgers
		Game._showMap(Game._players[0].map);
		PlayerOrders.prepare(Game.State.baseMap);
		Util.replaceText("turnnumber", Game.State.turn);
	},
	_createComputerWorkers: function(computerURL) {
		Game._computers = {};
		Game._orders = {};
		for(let i=1; i<Game.STYLENAMES.length; i++) {
			let worker = new Worker(computerURL);
			worker.onmessage = Game._computerHasOrders;
			worker.onerror = Game._computerHasError;
			Game._computers[Game.STYLENAMES[i]] = worker;
		}
	},
	_pokeComputerWorkers: function() {
		for(let i=1; i<Game._players.length; i++) {
			let player = Game._players[i];
			let worker = Game._computers[player.stylename];
			worker.postMessage({stylename: player.stylename,
								baseMap: JSON.stringify(Game.State.baseMap),
								capital: player.capital,
								tech:player.indexOfFavouredTech,
								player: JSON.stringify(player),
								turn: Game.State.turn
								});
			delete player._seed; // So that the computer worker knows it has a ROTRNGState field that it should instead of the seed
		}
	},
	_updateProduction: function() {
		const productionTotal = {};
		// Colonies
		Game.State.baseMap.forEachHex(function(hex) {
			if (hex.owner && hex.level > 0 && !hex.interdicted) {
				productionTotal[hex.owner] = productionTotal[hex.owner] ? productionTotal[hex.owner]+hex.level*20 : hex.level*20;
			}
		});
		// Ships with Dredgers
		for(let player of Game._players) {
			for(let ship of player.ships) {
				if (!ship.nonDredgerUnits) {
					productionTotal[player.stylename] += ship.unjumpedTurns * ship.getDredgers();
				}
			}
		}
		// Set the total
		for(let player of Game._players) {
			const capitalHex = Game.State.baseMap.hex(player.capital);
			if (!player.active) {
				player.production = 0;
			} else if (capitalHex.interdicted) {
				player.production = capitalHex.level*20;
			} else {
				player.production = productionTotal[player.stylename];
			}
		}
	},
	_updateHistory: function(events) {
		let sheet = {};
		events = events || [];
		Game.History.avt = Game.History.avt || new Array(ALLTECHS.length).fill(1);
		let techTotals = new Array(ALLTECHS.length).fill(0);
		let activePlayers = 0;
		for(let player of Game._players) {
			if (player.active) {
				activePlayers++;
				for(let i=0; i<ALLTECHS.length; i++) {
					techTotals[i] += player.researchLevel(ALLTECHS[i]);
				}
			}
		}
		let newavt = [];
		for(let i=0; i<ALLTECHS.length; i++) {
			newavt[i] = Math.floor(techTotals[i] / activePlayers);
			if (newavt[i] > Game.History.avt[i]) {
				events.push(`The average tech level for ${ALLTECHS[i].option} has reached ${newavt[i]}`);
			}
		}
		Game.History.avt = newavt;
		if (events.length > 0) {
			sheet.events = events;
		}
		Game.State.baseMap.forEachHex(function(hex) {
			if (hex.owner) sheet[hex.id] = hex.owner; /* Each turn has a map showing who owns the hex */
		});
		Game.History[Game.State.turn] = sheet;
		Game.History.lastTurn = Game.State.turn;
		localStorage.domainOfTheAncientsHistory = JSON.stringify(Game.History);
	},
	_replayHistory() {
		localStorage.domainOfTheAncientsGameOver = "true";
		document.getElementById("gameOver").classList.remove("hidden");
		document.getElementById("resume").disabled = true;
		document.getElementById("welcome").classList.add("hidden");
		document.getElementById("history").classList.remove("hidden");
		Game.History = JSON.parse(localStorage.domainOfTheAncientsHistory);
		Game.History[0] = Game.History[0] ? Game.History[0] : Game.History[1]; // Version 43 data doesn't have the turn 0 entry'
		Game._replayTurn = 0;
		Game._displayHistoryTurn();
	},
	_previousHistory() {
		Game._replayTurn--;
		Game._displayHistoryTurn();
	},
	_nextHistory() {
		Game._replayTurn++;
		Game._displayHistoryTurn();
	},
	_endHistory() {
		GameMap.forEachHexId(id => {
			const hexelement = document.getElementById(id);
			const planetElement = hexelement.childNodes[3];
			planetElement.classList.remove("capital","starbase","planet");
			hexelement.classList.remove(...stylenames);
		});
		Util.replaceText("turnnumber", Game.State.turn);
		Game._showLoadButton();
		document.getElementById("welcome").classList.remove("hidden");
		document.getElementById("history").classList.add("hidden");
	},
	_displayHistoryTurn() {
		Util.replaceText("turnnumber",Game._replayTurn);
		document.getElementById("historyPreviousTurn").disabled = (Game._replayTurn == 0);
		document.getElementById("historyNextTurn").disabled = (Game._replayTurn == Game.History.lastTurn);
		let capitals = Object.keys(Game.History[0]);
		GameMap.forEachHexId(id => {
			const hexelement = document.getElementById(id);
			const planetElement = hexelement.childNodes[3];
			planetElement.classList.remove("capital","starbase","planet");
			if (Game.History.baseMap[id]) {
				planetElement.classList.remove("emptyspace"); // It's clearly not empty!
				if (capitals.some(capital => capital == id)) {
					planetElement.classList.add("capital");
				} else {
					planetElement.classList.add("planet");
				}
			} else {
				planetElement.classList.add("emptyspace");
			}
			hexelement.classList.remove(...stylenames);
			hexelement.title = "";
			if (Game.History[Game._replayTurn][id]) {
				hexelement.classList.add(Game.History[Game._replayTurn][id]);
				hexelement.title = "Owned by "+Game.FULLNAMES[Game.History[Game._replayTurn][id]];
				if (planetElement.classList.contains("emptyspace")) {
					planetElement.classList.remove("emptyspace");
					planetElement.classList.add("starbase");
				}
			}
		});
		Util.emptyList("historyEvents");
		if (Game.History[Game._replayTurn].events) {
			for(let event of Game.History[Game._replayTurn].events) {
				const li = document.createElement("li");
				li.append(event);
				document.getElementById("historyEvents").append(li);
			}
		}
	},
	init: function() {
		// Find the URL for the computer.js script and then create the worker threads using that url
		Game._createComputerWorkers(document.getElementById("djv_domainOfTheAncients_computer-js").src);

		ShipDesign.init();
		for(let shipdesign of Object.values(ShipDesign.REGISTRY)) { // Only valid because the players haven't been created or loaded yet!
			Game._standardShipDesigns.push(shipdesign);
		}
		Game.ARTIFACT_DESIGN = ShipDesign.REGISTRY[0];

		Game._initGameButtons();
		Game._initTwisties();
		PlayerOrders.init();
		ShipOrderReport.init();
		ShipDesigner.init();
	},
	_initGameButtons: function() {
		document.getElementById("generate").onclick = Game._generateNewGame;
		if (localStorage.domainOfTheAncients && localStorage.domainOfTheAncientsVersion && localStorage.domainOfTheAncientsSaveDate) {
			Game._showLoadButton();
			let v;
			try {
				v = JSON.parse(localStorage.domainOfTheAncientsVersion);
			} catch(_e) {
				v = null; // Clearly no version data if we can't parse it!
			}
			if (!localStorage.domainOfTheAncientsGameOver &&v && v.game && (parseInt(v.game) >= InitialState.minimumVersion)) {
				document.getElementById("resume").disabled = false;
				document.getElementById("wrongVersion").classList.add("hidden");
				document.getElementById("resume").onclick = Game._load;
			} else {
				document.getElementById("resume").disabled = true;
				if (localStorage.domainOfTheAncientsGameOver) {
					document.getElementById("gameOver").classList.remove("hidden");
				} else if (v && v.plugin) {
					Util.replaceText("incompatibleVersionNumber",v.plugin);
					document.getElementById("wrongVersion").classList.remove("hidden");
				}
			}
		} else {
			document.getElementById("resume").disabled = true;
		}
		document.getElementById("generate").disabled = false;

		Game._initHistoryButton();
		document.getElementById("replay").onclick = Game._replayHistory;
		document.getElementById("historyPreviousTurn").onclick = Game._previousHistory;
		document.getElementById("historyReturnToWelcome").onclick = Game._endHistory;
		document.getElementById("historyNextTurn").onclick = Game._nextHistory;

		document.getElementById("nextTurn").onclick = Game._humanHasOrders;
		document.getElementById("processorderread").onclick = Game._completeTurn;
	},
	_initHistoryButton: function() {
		if (localStorage.domainOfTheAncientsHistory) {
			document.getElementById("replay").disabled = false;
		} else {
			document.getElementById("replay").disabled = true;
		}
	},
	_initTwisties: function() {
		// Look for the twisties and set them up
		for(let button of document.querySelectorAll("h4 button")) {
			// Look for the child svg's 'we are interested in
			let svg1 = null, svg2 = null, followingdiv = null;
			for(let child of button.childNodes) {
				if (child.nodeName == "svg") {
					if (svg1 === null) {
						svg1 = child;
					} else {
						svg2 = child;
					}
				}
			}
			// And then the uncle's (i.e. parent's sibling) div
			let sibling = button.parentNode.nextSibling;
			while(followingdiv === null && sibling) {
				if (sibling.nodeName === "DIV") {
					followingdiv = sibling;
				} else {
					sibling = sibling.nextSibling;
				}
			}

			if (svg1 && svg2 && followingdiv) {
				const selfargs = [svg1, svg2, followingdiv];
				button.onclick = function() { Util.toggleFields(selfargs); };
			}
		}
	},
	_computerHasError: function(event) {
		document.getElementById("exception-box").handleErrorEvent(event);
		const worker = event.currentTarget;
		for(let stylename of Object.keys(Game._computers)) {
			if (Game._computers[stylename] === worker) {
				Game._orders[stylename] = [new NullOrder()];
				console.log(`It was the worker for ${stylename} that encountered ${event}`);
				if (!Game._processIfEverybodyReady()) {
					Game._updateAwaitingList();
				}
				break;
			}
		}
	},
	_computerHasOrders: function(event) {
		const worker = event.currentTarget;
		for(let stylename of Object.keys(Game._computers)) {
			if (Game._computers[stylename] === worker) {
				const player = Game._stylenameToPlayerMapping[stylename];
				const spend = JSON.parse(event.data.orders);
				Game._orders[stylename] = [];
				for(let orderdata of spend) {
					Game._orders[stylename].push(Order.fromJSON(orderdata));
				}
				const shiporders = JSON.parse(event.data.ships);
				for(let i=0; i<shiporders.length; i++) {
					player.ships[i].order = ShipOrder.fromJSON(shiporders[i]);
				}
				player.state = JSON.parse(event.data.state);
				player.designid = parseInt(event.data.designid);
				player.ROTRNGState = event.data.ROTRNGState; // Ready to resend next time
				break;
			}
		}
		if (!Game._processIfEverybodyReady()) {
			Game._updateAwaitingList();
		}
	},
	_humanHasOrders: function() {
		Game._orders[Game.STYLENAMES[0]] = PlayerOrders.getOrders();
		if (!Game._processIfEverybodyReady()) {
			// Need to wait for more orders from computer players, so display the status page
			document.getElementById("summary").classList.add("hidden");
			document.getElementById("awaitingOrders").classList.remove("hidden");
			Game._updateAwaitingList();
		}
	},
	_updateAwaitingList: function() {
		if (!document.getElementById("awaitingOrders").classList.contains("hidden")) {
			const list = document.getElementById("awaitingOrdersFromList");
			Util.emptyList(list);
			for(let stylename of Object.keys(Game._computers)) {
				if (!Game._orders[stylename]) {
					const term = document.createElement("dt");
					const def = document.createElement("dd");
					Util.replaceText(term,Game.FULLNAMES[stylename]);
					Util.replaceText(def,"");
					list.append(term,def);
				}
			}
		}
	},
	_processIfEverybodyReady: function() {
		let ready = true;
		for(let i=0; i<Game._players.length; i++) {
			if (!Game._orders[Game._players[i].stylename]) {
				ready = false; break;
			}
		}
		if (ready) {
			Game._processTurn();
			return true;
		} else {
			return false;
		}
	},
	_processTurn: function() {
		document.getElementById("awaitingOrders").classList.add("hidden");
		const summary = document.getElementById("summary");
		const proc = document.getElementById("processOrder");

		const [updates,events] = TurnProcessor.processTurn();
		Game._updateHistory(events);

		// Remove the orders from everybody in case an exception occurs beyond this point
		for(let i=0; i<Game.STYLENAMES.length; i++) {
			Game._orders[Game.STYLENAMES[i]] = null;
		}

		// Let the human know of any interesting events (if any)
		if (updates.length > 0) {
			Game._showMap(Game._players[0].map); // To show the ship discoveries
			summary.classList.add("hidden");
			proc.classList.remove("hidden");
			const list = document.getElementById("messages");
			list.replaceChildren();
			for(let update of updates) {
				const element = document.createElement("li");
				element.append(update);
				list.append(element);
			}
		} else {
			Game._completeTurn(); // Just complete the turn
		}
	},
	_completeTurn: function() {
		let summary = document.getElementById("summary");
		let proc = document.getElementById("processOrder");
		proc.classList.add("hidden");
		summary.classList.remove("hidden");
		let won = false, lost = false;
		if (Game.State.baseMap.hex(Game._players[0].capital).planet) {
			won = true;
			for(let i=1; i<Game._players.length; i++) {
				if (Game._players[i].active) {
					won = false;
					Game._newTurn();
					break;
				}
			}
		} else {
			lost = true;
		}
		if (won || lost) {
			document.getElementById("welcome").classList.remove("hidden");
			Game._initHistoryButton();
			document.getElementById("gameOver").classList.remove("hidden");
			localStorage.domainOfTheAncientsGameOver = "true";
			document.getElementById("resume").disabled = true;
			Game._showLoadButton();
			if (won) {
				document.getElementById("youwon").classList.remove("hidden");
			}
			if (lost) {
				document.getElementById("youlost").classList.remove("hidden");
			}
		}
	},
	_showLoadButton: function() {
		if (localStorage.domainOfTheAncientsSaveDate) {
			const saved = new Date(localStorage.domainOfTheAncientsSaveDate);
			const nextUpdate = Util.next_time_diff(saved);
			if (localStorage.domainOfTheAncientsTurnNumber) Util.replaceText("savegameturn",localStorage.domainOfTheAncientsTurnNumber);
			Util.replaceText("savegamedate",Util.time_diff(saved));
			// Is the welcome screen hidden?
			const hiddenWelcome = document.getElementById("welcome").classList.contains("hidden");
			if (nextUpdate && !hiddenWelcome) {
				window.setTimeout(Game._showLoadButton, nextUpdate*1000);
			}
		}
	},
	_saveGameState: function() {
		Game._players[0].ROTRNGState = JSON.stringify(ROT.RNG.getState()); // Save the ROT state for the main thread (we use it for combats)
		let data = {};
		data.turn = Game.State.turn;
		data.baseMap = Game.State.baseMap;
		data.playerTech = Game._players[0].indexOfFavouredTech;
		data.players = [];
		for(let p of Game._players) {
			data.players.push(p);
		}
		data.built = [];
		for(let design of Game._standardShipDesigns) {
			data.built.push(design._built);
		}
		data.designID = ShipDesigner._designId;
		localStorage.domainOfTheAncientsTurnNumber = Game.State.turn;
		localStorage.domainOfTheAncientsSaveDate = (new Date()).toISOString();
		let version = {game: Game.State.version, plugin:"3.218.0"};
		localStorage.domainOfTheAncientsVersion = JSON.stringify(version);
		localStorage.domainOfTheAncients = JSON.stringify(data);
	},
	_load: function() {
		const data = JSON.parse(localStorage.domainOfTheAncients);
		Game.State.turn = data.turn;
		Game.State.baseMap = new GameMap(data.baseMap);
		let techs = [...defaultTechIndexes]; // Note, if the saved player has a tech (version 44+) it will override this on the fromJSON call
		techs[0] = data.playerTech;
		Game._createPlayersWithTech(techs);
		for(let i=0; i<Game._players.length; i++) {
			Game._players[i].fromJSON(data.players[i],Game.State.baseMap);
			Game._stylenameToPlayerMapping[Game._players[i].stylename] = Game._players[i];
		}

		ROT.RNG.setState(JSON.parse(data.players[0].ROT));

		let i=0;
		for(let design of Game._standardShipDesigns) {
			design._built = data.built[i++];
		}

		PlayerOrders.setPlayer(Game._players[0]);
		PlayerOrders.setStandardShipDesigns(Game._standardShipDesigns);
		if (data.designID) { // Only exists from version 45 onwards
			ShipDesigner._designId = data.designID;
		}

		Game.State.version = InitialState.version;

		Game.History = JSON.parse(localStorage.domainOfTheAncientsHistory);

		document.getElementById("welcome").classList.add("hidden");
		document.getElementById("summary").classList.remove("hidden");
		Game._startTurn();
	},
};

if (typeof window !== "undefined") {
	window.onload = function() {
		Game.init();
	};
}

//For testing via QUnit
if (typeof module !== "undefined" && module.exports) {
	exports.Game = Game;
}