/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals self, postMessage, importScripts, GameMap, Player, ROT, TECHS, ShipDesign, ShipDesignType */
/* globals ResearchOrder, CarryForwardOrder, ImproveColonyOrder, AddMarinesOrder, RegisterShipDesign, NullOrder */
/* globals BuyVehicleOrder, BuyStarBaseOrder */
/* globals ShipOrder, RepairShipOrder, ScrapShipOrder */
/* globals console */
/* exported onmessage */

if (!self.document) {
	importScripts('map.js?' + location.search);
	importScripts('ship.js?' + location.search);
	importScripts('shipdesign.js?' + location.search);
	importScripts('shiporder.js?' + location.search);
	importScripts('tech.js?' + location.search);
	importScripts('player.js?' + location.search);
	importScripts('order.js?' + location.search);
	importScripts('../rot/rot.min.js?' + location.search);

	ShipDesign.init();

	const MOODS = ["explore","expand","exploit","exterminate","invade","defend","research"];
	const debug = location.hostname == 'virtual.internal';

	const designPoolSize = 100; // size of the pool of potential new designs - must be even for the GA algorithm below to work
	const designLoops = 25;    // Number of iterations for the GA

	var onmessage = function(message) {
		const datastylename = message.data.stylename;
		const databasemap = JSON.parse(message.data.baseMap);
		const datacapital = message.data.capital;
		const datatech = message.data.tech;
		const dataplayer = JSON.parse(message.data.player);

		const turn = parseInt(message.data.turn);
		const basemap = new GameMap(databasemap);
		const player = new Player(datastylename,datacapital,datatech);

		player.fromJSON(dataplayer,basemap);
		if (player.state) {
			if (typeof player.state === 'string') {
				player.state = { mood: player.state };
			}
		} else {
			player.state = { mood: null };
		}
		if (player.seed) {
			ROT.RNG.setSeed(player.seed); // Set ROT's seed on this computer worker thread
			for(let i=1; i<10; i++) {
				ROT.RNG.getUniform(); // Pull some numbers to get into a more random space
			}
		} else {
			ROT.RNG.setState(JSON.parse(player.ROTRNGState));
		}

		let orders = [new NullOrder()];
		let shiporders = [];

		if (player.active) {
			changeState(turn,player);

			moveShips(player);
			orders = spendMoney(player);

			for(let ship of player.ships) {
				shiporders.push(ship.order);
			}
		}

		let reply = {orders: JSON.stringify(orders),
					 ships: JSON.stringify(shiporders),
					 state: JSON.stringify(player.state),
					 designid: player.designid,
					 ROTRNGState: JSON.stringify(ROT.RNG.getState()) };
		postMessage(reply);
	};

	var changeState = function(turn, player) {
		const oldMood = player.state.mood;
		if (oldMood && !player.turnNotes.length && ROT.RNG.getPercentage() > 15) return oldMood; // No need to change state

		let possibles = {};
		for(let actionName of MOODS) {
			possibles[actionName] = 20;
		}
		// Scan the player's map - unscanned hexes add twenty to explore - add one per missed turn for other hexes to explore.
		// Uncolonised worlds add 50 to expand
		player.map.forEachHex(hex => {
			if (hex.onGrid) {
				if (hex.lastScanned) {
					if (hex.planet && hex.level == 0) {
						possibles.expand += 50;
					}
					possibles.explore += (hex.lastScanned - turn);
				} else {
					possibles.explore += 20;
				}
				if (hex.owner && hex.owner !== player.stylename) {
					possibles.invade += 30;
					possibles.exterminate += 30;
				}
			}
		});

		// Also check the turn notes from the Turn Processor - they will cause big changes to the possibles
		let targets = [];
		for(let note of player.turnNotes) {
			switch(note.event) {
				case "planetBusted": possibles.defend += 100; break;
				case "colonised": possibles.defend -= 20; possibles.exterminate -= 20; break; // To balance out the ship destroyed event we'll also be seeing
				case "interdicted": possibles.defend += 100; possibles.exterminate += 50; targets.push(note.location); break;
				case "invaded": possibles.defend += 100; targets.push(note.location); break;
				case "shipCombat": possibles.defend += 10; possibles.exterminate += 10; targets.push(note.location); break;
				case "shipDestroyed": possibles.defend += 20; possibles.exterminate += 20; targets.push(note.location); break;
				default: break;
			}
		}

		const newMood = ROT.RNG.getWeightedValue(possibles);
		player.map.forEachHex( hex => {
			if (hex.onGrid) {
				switch(newMood) {
					case "explore": if (!hex.lastScanned || (turn - hex.lastScanned > 5)) { targets.push(hex.id); } break;
					case "expand": if (!hex.owner || (hex.owner == player.stylename && hex.level < 5)) { targets.push(hex.id); } break;
					case "invade": if (hex.owner && hex.owner !== player.stylename) { targets.push(hex.id); } break;
					default: break;
				}
			}

		});
		if (targets.length)
			player.state.target = ROT.RNG.getItem(targets);
		else
			player.state.target = null;

		if (debug && newMood !== oldMood) {
			console.info(`${player.stylename} changing to ${newMood}${player.state.target ? `(target hex ${player.state.target})` : ''}`);
		}

		player.state.mood = newMood;
	};

	var moveShips = function(player) {
		var shipyards = [];
		var colonisable = [];
		var colonies = [];
		var coloniesWithDetails = [];
		player.map.forEachHex(hex => {
			if (hex.owner == player.stylename && hex.planet && hex.level) {
				colonies.push(hex.id);
				coloniesWithDetails.push({id: hex.id, level: hex.level, defence: 0});
				if (hex.level > 19)
					shipyards.push(hex.id);
				else if (hex.level < 5) {
					colonisable.push({id: hex.id, missing:5-hex.level});
				}
			} else if (!hex.owner && hex.planet) {
				colonisable.push({id: hex.id, missing:5});
			}
		});
		for(let ship of player.ships) {
			const selfShipyards = shipyards;
			const selfShip = ship;
			if (ship.retreatingTo) {
				ship.order = new ShipOrder();
				ship.order.retreating= true;
				if (ship.retreatingTo !== ship.location) ship.order.jump = ship.retreatingTo;
			} else if (ship.getRepairCost() > 0 && ship.design.type !== ShipDesignType.Ship) {
				ship.order = new RepairShipOrder(ship.getRepairCost());
			} else if (ship.getRepairCost() > 0 && selfShipyards.some(hexid => hexid == selfShip.location)) {
				ship.order = new RepairShipOrder(ship.getRepairCost());
			} else if (ship.getRepairCost() > 0 && ship.getJumpRange() > 0) {
				ship.order = new ShipOrder();
				const target = chooseClosestHexToDestination(player,ship.location,shipyards,ship.getJumpRange());
				if (target !== ship.location) ship.order.jump = target;
			} else if (ship.getRepairCost() > 0){
				// It's damaged but can't jump (and we can't repair it since it's not at a shipyard (and not a base)) - scrap it!
				ship.order = new ScrapShipOrder(ship.getScrapValue());
			} else if (shouldScrap(player,ship)) {
				ship.order = new ScrapShipOrder(ship.getScrapValue());
			} else if (ship.getColonyImprovement() > 0 && ship.getJumpRange() > 0) {
				const possibleColonies = colonisable.filter(element => GameMap.range(selfShip.location,element.id) <= selfShip.getJumpRange());
				if (possibleColonies.length) { // i.e. there's at least one colony
					const destColony = ROT.RNG.getItem(possibleColonies);
					ship.order = new ShipOrder();
					if (destColony.id !== ship.location) ship.order.jump = destColony.id;
					ship.order.colonise = true;
					destColony.level--;
					if (destColony.level == 0) {
						const index = possibleColonies.findIndex(element => element.id = destColony.id);
						colonisable.splice(index,1);
					}
				} else {
					ship.order = new ShipOrder();
				}
			} else if (ship.marines && player.state.mood == "invade" && player.state.target) {
				const jumpTarget = chooseClosestHexToDestination(player,ship.location,[player.state.target],ship.getJumpRange());
				ship.order = new ShipOrder();
				if (jumpTarget !== ship.location) ship.order.jump = jumpTarget;
				if (jumpTarget == player.state.target) {
					ship.order.setMarinesToDisembark();
				}
			} else if (ship.marines) {
				// no invasion target at present - hold position
				ship.order = new ShipOrder();
				if (ship.marines < ship.getMaxMarines())
					ship.order.setMarinesToEmbark(); // But pick up marines from current location if possible
			} else if (ship.getMaxMarines()) {
				const jumpTarget =chooseClosestHexToDestination(player,ship.location,colonies,ship.getJumpRange());
				ship.order = new ShipOrder();
				if (jumpTarget !== ship.location) ship.order.jump = jumpTarget;
				ship.order.setMarinesToEmbark(); // But pick up marines from current location if possible
			} else if (ship.getJumpRange() == 0) {
				// Active non jumping ship
				ship.order = new ShipOrder();
				const colonyIndex = coloniesWithDetails.findIndex(element => element.id == selfShip.location);
				if (colonyIndex !== -1) {
					coloniesWithDetails[colonyIndex].defence += ship.design.getRatingDefence(); // Assumes ship/base is undamaged
				}
			} else if ((player.state.mood == "exterminate" || player.state.mood == "invade" || player.state.mood == "defend") && player.state.target) {
				ship.order = new ShipOrder();
				const target = chooseClosestHexToDestination(player,ship.location,[player.state.target],ship.getJumpRange());
				if (target !== ship.location) ship.order.jump = target;
			} else if (ROT.RNG.getPercentage() > 40 || player.state.mood == "explore") {
				ship.order = new ShipOrder();
				const possibleHexes = [];
				for(let row=1; row<21; row++) {
					for(let column of "ABCDEFGHIJKLMNOP") {
						const name = column+row;
						if (GameMap.range(ship.location,name) <= ship.getJumpRange() && player.map.hex(name).onGrid) {
							possibleHexes.push(name);
						}
					}
				}
				const target = ROT.RNG.getItem(possibleHexes);
				if (target !==  ship.location) ship.order.jump = target;
			} else {
				ship.order = new ShipOrder();
			}
		}

		// Okay we should have a list of active combat ships and a list of our colonies with the amount of defence from undamaged ships and bases


		for(let ship of player.ships) {
			if (debug) console.info(`${player.stylename}: ${ship.name} at ${ship.location} ordered to ${ship.order.toString()}`);
		}
	};

	var chooseClosestHexToDestination = function(player,src,dests,jump) {
		var choices = [];
		var closest = 50;
		for(let row=1; row<21; row++) {
			for(let column of "ABCDEFGHIJKLMNOP") {
				const name = column+row;
				if (GameMap.range(src,name) <= jump && player.map.hex(name).onGrid) {
					for(let dest of dests) {
						const range = GameMap.range(name,dest);
						if (range < closest) {
							choices = [];
							closest = range;
						}
						if (range === closest) {
							choices.push(name);
						}
					}
				}
			}
		}
		return ROT.RNG.getItem(choices);
	};

	var shouldScrap = function(player,ship) {
		// Is the design a current best design? (if so we don't need to scrap it)
		var bestDesign = determineBestShipDesigns(player);

		for (let state of MOODS) {
			if (bestDesign[state] == ship.design) return false;
		}

		// Still here? It is obsolete so 50/50 whether to scrap it
		return ROT.RNG.getPercentage() < 51;
	};

	var evaluateDesign = function(player,design,mood,budget) {
		if (design === null) return -1e9;
		if (!design.isValid()) return -1e9;
		if (!design.isTechSufficent(player.getShipTechs())) return -1e6-design.totalCost(); // Better than unbuildable designs but worse than merely overbudget designs
		if (design.type === ShipDesignType.SHIP && design.getJumpRange()===0 && mood !== "defend") return -1e3-design.totalCost(); // Ships really ought to jump
		const cost = design.totalCost()*builtFactor(player,design,[]);
		if (cost > budget) return -cost; // Rate according to negative price if over budget (i.e. it's better than unbuilable designs)
		// Note: the rating already inlcudes a component for the cost
		switch(mood) {
			case "exploit": return design.getRatingProduction();
			case "explore": return (design.getJumpRange() > player.researchLevel(TECHS.communications) ? player.researchLevel(TECHS.communications) : design.getJumpRange());
			case "expand":  return design.getRatingColony();
			case "exterminate": return design.getRatingAttack();
			case "invade": return design.getRatingTroops();
			case "defend": return design.getRatingDefence();
			default: return 0; // I.e. don't have a best design for this mood
		}
	};

	var designNewShip = function(player,type,budget) {
		if (budget < 100) return null; // The cheapest possible ship is 50 PPs (which will be doubled for the first one) so if the proposed budget isn't even that high - give up now!
		if (player.state.mood == "exploit" && player.researchLevel(TECHS.archaeology) < 2) return null; // Need jump space dredgers to be able to have designs that can exploit
		if (player.state.mood == "research") return null; // Not going to bother if we are supposed to be researching
		if (player.state.mood == "explore" && player.researchLevel(TECHS.communications) < 3) return null; // The TL1 scout is already Jump-2
		if (player.state.mood == "expand" && player.researchLevel(TECHS.communications) < 2 && player.researchLevel(TECHS.shipSize) == 1) return null; // The TL1 colony ship is already Jump-1
		if (player.state.mood == "invade" && player.researchLevel(TECHS.communications) < 2  && player.researchLevel(TECHS.shipSize) == 1) return null; // The TL1 troop transport is already Jump-1
		// First what's the best ship design so far
		let currentDesignsInBudget = Object.values(ShipDesign.REGISTRY)
			.filter(design => design.type == type)
			.filter(design => design.isTechSufficent({ power:	player.researchLevel(TECHS.powerPlant),
					jump:		player.researchLevel(TECHS.jumpDrive),
					man:		player.researchLevel(TECHS.maneuverDrive),
					size:		player.researchLevel(TECHS.shipSize),
					weapons:	player.researchLevel(TECHS.weaponary),
					defences:	player.researchLevel(TECHS.shipDefences),
					computer:	player.researchLevel(TECHS.computers),
					arch:		player.researchLevel(TECHS.archaeology),
					planet:	player.researchLevel(TECHS.planetaryDefences)}))
			.filter(design => design.totalCost()*builtFactor(player,design,[]) <= budget)
			.sort((a,b) => {
				return evaluateDesign(player,b,player.state.mood,budget) - evaluateDesign(player,a,player.state.mood,budget);
			});
		const currentBest = (currentDesignsInBudget.length ? currentDesignsInBudget[0] : null);
		const currentBestScore = evaluateDesign(player,currentBest,player.state.mood,budget);
		if (debug) console.info(`${player.stylename}: designNewShip: budget=${budget} type=${type.toString()} priority=${player.state.mood} currentBest=${currentBest ? currentBest.name : "none"} score=${currentBestScore}`);
		let pool = [];
		for(let i=0; i<designPoolSize; i++) {
			pool[i] = getNewRandomVesselDesign(player,player.state.mood,type,budget);
		}
		for(let g=0; g<designLoops; g++) {
			const selfEval = evaluateDesign;
			const selfPlayer = player;
			const selfBudget = budget;
			pool.sort((a,b) => selfEval(selfPlayer,b,selfPlayer.state.mood,selfBudget) - selfEval(selfPlayer,a,selfPlayer.state.mood,selfBudget));
			if (debug) console.info(`${player.stylename}: generation ${g} Best New Design: ${JSON.stringify(pool[0])} scoring ${selfEval(selfPlayer,pool[0],selfPlayer.state.mood,selfBudget)}`);

			for(let i=pool.length/2; i<pool.length; i++) {
				pool[i].clone(pool[i-pool.length/2]);
				if (g < ROT.RNG.getUniform()*10) pool[i].cross(pool[i-pool.length/2+1]);
				pool[i].mutate();
			}
		}
		pool.sort((a,b) => evaluateDesign(player,b,player.state.mood,budget) - evaluateDesign(player,a,player.state.mood,budget));
		const proposedDesign = pool[0];
		for(let i=1; i<pool.length; i++) {
			pool[i].delete(); // These designs definitely didn't make the cut
		}
		const proposedScore = evaluateDesign(player,proposedDesign,player.state.mood,budget);
		if (proposedScore <= currentBestScore) {
			proposedDesign.delete();
			return null;
		}
		if (proposedDesign.totalCost()*builtFactor(player,proposedDesign,[]) > budget) {
			if (debug) console.info(`${player.stylename}: design rejected - too expensive at ${proposedDesign.totalCost()*builtFactor(player,proposedDesign,[])}`);
			proposedDesign.delete();
			return null;
		}
		if (debug) console.info(`${player.stylename}: New Design: ${JSON.stringify(proposedDesign)} scores ${proposedScore}`);

		return proposedDesign;
	};

	var getNewRandomVesselDesign = function(player,mood,type,budget) {
		// Always start with a bridge
		let blocks = ["_b0","_b1","_b2","_b3","_b4","_b5","_b6","_b7","_b8","_b9"].slice(0,player.researchLevel(TECHS.computers)+1);
		const name = player.getNextDesignId(mood);
		const design = new ShipDesign(name,type,name);
		design.setFighterTechs(player.getShipTechs());
		design.addBlock(ROT.RNG.getItem(blocks));

		// Also always need a power for a useful vessel - so we will give it one now
		let powerBlocks = ["_p1","_p2","_p3","_p4","_p5","_p6","_p7"].slice(0,player.researchLevel(TECHS.powerPlant));
		design.addBlock(ROT.RNG.getItem(powerBlocks));

		// Now add in all the other block types
		blocks = blocks.concat(powerBlocks);
		if (type == ShipDesignType.Ship) {
			blocks = blocks.concat(["_ja","_jb","_jc","_jd","_je","_jf","_jg","_jh"].slice(0,player.researchLevel(TECHS.jumpDrive)));
			blocks = blocks.concat(["_ma","_mb","_mc","_md","_me","_mf","_mg","_mh"].slice(0,player.researchLevel(TECHS.maneuverDrive)));

			if (player.state.mood == "expand") {
				blocks.push("_cb");
				design.addBlock("_cb"); // It'll need at least one to stand a chance of being a 'good' design
			}
			if (player.state.mood == "invade") {
				blocks.push("_bb");
				design.addBlock("_bb"); // It'll need at least one (and probably more!) to stand a chance of being a 'good' design
			}
		}
		blocks = blocks.concat(["_wl","_wp","_wa","_wf","_wr","_wc","_ws","_wm","_wt","_wx","_wb"].slice(0,player.researchLevel(TECHS.weaponary)));
		blocks = blocks.concat(["_da","_dc","_ds","_df","_di","_dg","_dr","_di"].slice(0,player.researchLevel(TECHS.shipDefences)));
		if (player.researchLevel(TECHS.archaeology) > 1) blocks = blocks.concat(["_an"]);
		if (player.researchLevel(TECHS.archaeology) > 2) blocks = blocks.concat(["_ad"]);
		if (player.researchLevel(TECHS.archaeology) > 3) blocks = blocks.concat(["_at"]);
		if (player.researchLevel(TECHS.archaeology) > 4) blocks = blocks.concat(["_as"]);

		const maxSizes = [0,10,20,30,40,60,80,100,150,1e9];
		const maxSize = maxSizes[player.researchLevel(TECHS.shipSize)];
		while(design.totalSize() < maxSize && (ROT.RNG.getUniform() > (2/maxSize)) && design.totalCost()*builtFactor(player,design,[]) < budget) {
			design.addBlock(ROT.RNG.getItem(blocks),player.getShipTechs());
		}
		return design;
	};

	var spendMoney = function(player) {
		const orders = [];
		const colonies = [];
		const allHexesOnGrid = [];
		let money = player.production+player.carriedProduction;
		player.map.forEachHex(function(hex) {
			if (hex.owner == player.stylename && hex.level > 0) {
				colonies.push(hex.id);
			}
			if (hex.onGrid) {
				allHexesOnGrid.push(hex.id);
			}
		});

		if (ROT.RNG.getPercentage() < 21) {
			const design = designNewShip(player,ShipDesignType.Ship,money);
			if (design) {
				const order =  new RegisterShipDesign(design);
				const colony = ROT.RNG.getItem(colonies.filter(colony => player.map.hex(colony).level > 19));
				const name = design.name+"-"+getRandom(0,99999999);
				const buyorder = new BuyVehicleOrder(design,name,colony);
				const amount = design.totalCost()*builtFactor(player,design,orders);
				orders.push(order);
				orders.push(buyorder);
				money -= amount;
			}
		}

		if (ROT.RNG.getPercentage() < 11 && player.researchLevel(TECHS.planetaryDefences) == 3) {
			const design = designNewShip(player,ShipDesignType.SpaceBase,money);
			if (design) {
				const order =  new RegisterShipDesign(design);
				const colony = ROT.RNG.getItem(colonies.filter(colony => player.map.hex(colony).level > 19));
				const name = design.name+" "+getRandom(0,99999999);
				const buyorder = new BuyVehicleOrder(design,name,colony);
				const amount = design.totalCost()*builtFactor(player,design,orders);
				orders.push(order);
				orders.push(buyorder);
				money -= amount;
			}
		}

		if (ROT.RNG.getPercentage() < 11 && player.researchLevel(TECHS.planetaryDefences) > 3) {
			const design = designNewShip(player,ShipDesignType.StarBase,money);
			if (design) {
				const order =  new RegisterShipDesign(design);
				const colony = ROT.RNG.getItem(allHexesOnGrid);
				const name = design.name+" "+getRandom(0,99999999);
				const buyorder = new BuyVehicleOrder(design,name,colony);
				const amount = design.totalCost()*builtFactor(player,design,orders);
				orders.push(order);
				orders.push(buyorder);
				money -= amount;
			}
		}

		var bestDesigns = determineBestShipDesigns(player);

		let spendMoneyOnResearch = function(_player,money,mood) {
			const amount = getRandom(1,money);

			let weighted ={};
			for(let tech of Object.keys(TECHS)) {
				weighted[tech] = 10;
				if (TECHS[tech].index == player.indexOfFavouredTech) tech[tech] += 20;
			}

			switch(mood) {
				case "explore": weighted.jumpDrive += 5; weighted.powerPlant += 5; weighted.communications += 5; break;
				case "expand":  weighted.jumpDrive += 5; weighted.powerPlant += 5; break;
				case "exterminate": weighted.powerPlant += 5; weighted.weaponary += 10; weighted.computers += 5; weighted.maneuverDrive += 5; break;
				case "invade": weighted.planetaryDefences += 20; break;
				case "defend": weighted.planetaryDefences += 10; weighted.powerPlant += 5; weighted.weaponary +=5; weighted.maneuverDrive += 5; break;
				default: break;
			}

			const techname = ROT.RNG.getWeightedValue(weighted);
			const order = new ResearchOrder(TECHS[techname],TECHS[techname].index === player.indexOfFavouredTech ? amount*2 : amount);
			return [order,amount];
		};
		let improvedColonies = {};
		let spendMoneyOnImprovingColony = function(_player,money,_priority) {
			const colony = ROT.RNG.getItem(colonies);
			let order = null;
			let amount = 0;
			const current = player.map.hex(colony).level;
			if (current > 4 && current < 20 && current <= money/20) {
				if (improvedColonies[colony]) {
					// No order
				} else {
					order = new ImproveColonyOrder(colony);
					amount = player.map.hex(colony).level*20;
					improvedColonies[colony] = true;
				}
			}
			return [order,amount];
		};
		let marinedColonies = {};
		let spendMoneyOnMarines = function(_player,money,_priority) {
			let order = null;
			let amount = 0;
			if (money > 9) {
				const colony = ROT.RNG.getItem(colonies);
				if (marinedColonies[colony]) {
					// No order
				} else {
					const companies = getRandom(1,Math.floor(money/10));
					order = new AddMarinesOrder(colony,companies);
					amount = companies*10;
					marinedColonies[colony] = true;
				}
			}
			return [order,amount];
		};
		let spendMoneyOnShip = function(player,money,mood) {
			let order = null;
			let amount = 0;
			const colony = ROT.RNG.getItem(colonies.filter(colony => player.map.hex(colony).level > 19));
			const design = bestDesigns[mood];
			if (design && design.totalCost()*builtFactor(player,design,orders) <= money) {
				order = new BuyVehicleOrder(design,newShipName(player,design,location),colony);
				amount = design.totalCost()*builtFactor(player,design,orders);
			}
			return [order,amount];
		};
		let spendMoneyOnSpaceBase = function(player,money,_mood) {
			let order = null;
			let amount = 0;
			const location = ROT.RNG.getItem(colonies);
			const design = bestDesigns.spaceBase;
			if (design && design.totalCost()*builtFactor(player,design,orders) <= money) {
				order = new BuyStarBaseOrder(design,newShipName(player,design,location),location);
				amount = design.totalCost()*builtFactor(player,design,orders);
			}
			return [order,amount];
		};
		let spendMoneyOnStarBase = function(player,money,_mood) {
			let order = null;
			let amount = 0;
			const location = ROT.RNG.getItem(allHexesOnGrid);
			const design = bestDesigns.starBase;
			if (design && design.totalCost()*builtFactor(player,design,orders) <= money) {
				order = new BuyStarBaseOrder(design,newShipName(player,design,location),location);
				amount = design.totalCost()*builtFactor(player,design,orders);
			}
			return [order,amount];
		};

		// Decide on where to spend money
		let functions = {research: spendMoneyOnResearch,
			improveColony: spendMoneyOnImprovingColony,
			marines: spendMoneyOnMarines,
			buildShip: spendMoneyOnShip,
			buildSpaceBase: spendMoneyOnSpaceBase,
			buildStarBase: spendMoneyOnStarBase
		};
		let options = {research: 34, improveColony: 11, marines: 16, buildShip: 39, buildSpaceBase: 5, buildStarBase: 5 };
		if (player.state.mood == "explore") options.buildShip += 20;
		if (player.state.mood == "expand") options.buildShip += 20;
		if (player.state.mood == "research") options.research += 20;
		if (player.state.mood == "exploit") options.improveColony += 20;
		if (player.state.mood == "defend") { options.marines += 30; options.buildSpaceBase += 30; options.buildStarBase += 30; }
		if (player.state.mood == "exterminate") options.buildShip += 20;
		if (player.state.mood == "invade") { options.marines += 10; options.buildShip += 10; }

		while(money > 0 && ROT.RNG.getPercentage() > 16) {
			const choice = ROT.RNG.getWeightedValue(options);
			const [newOrder,deduction] = [ ...functions[choice](player,money,player.state.mood) ];
			if (newOrder) {
				orders.push(newOrder);
				money -= deduction;
			}
		}
		orders.push(new CarryForwardOrder(money));
		if (debug) console.info(`${player.stylename} orders are [${orders}]`);
		return orders;
	};

	var newShipName = function(player,design,location) {
		let name = design.name;
		if (!design.name.includes(player.stylename)) name += "("+player.stylename+")";
		if (design.type !== ShipDesignType.Ship) name += "-"+location;
		return name+"-"+getRandom(0,99999999);
	};

	var builtFactor = function(player,design,orders) {
		let number = design.numberOfClassBuilt(player)+1;
		// Scan the orders looking for orders of this ship class - if so bump the number by one
		for(let order of orders) {
			if ((order instanceof BuyVehicleOrder) && (order.design === design)) { number++; }
		}
		// If the number is 1 AND this is a second or later ship we must double the price
		if (number === 1 && hasEverBuiltAShip(player,orders)) {
			return 2;
		}
		return 1;
	};

	var hasEverBuiltAShip = function(player,orders) {
		if (player.hasBuiltAShip) return true;
		for(let order of orders) {
			if (order instanceof BuyVehicleOrder) return true;
		}
		return false;

	};

	var getRandom = function(min,max) {
		return Math.floor(ROT.RNG.getUniform()*(max-min+1))+min;
	};

	var determineBestShipDesigns = function(player) {
		let result = {};
		let bscore = {};
		for (let state of MOODS) {
			result[state] = null; // Start with no design
			bscore[state] = 0;
		}
		result.spaceBase = null;
		bscore.spaceBase = 0;
		result.starBase = null;
		bscore.starBase = 0;

		const playerTechs = player.getShipTechs();
		for(let design of Object.values(ShipDesign.REGISTRY)) {
			if (design.isTechSufficent(playerTechs)) {
				if (design.type == ShipDesignType.Ship) {
					for (let state of MOODS) {
						const rating = evaluateDesign(player,design,state,1e9);
						if (rating > bscore[state]) {
							result[state] = design;
							bscore[state] = rating;
						}
					}
				} else if (design.type == ShipDesignType.SpaceBase){
					const rating = evaluateDesign(player,design,"defence",1e9);
					if (rating > bscore.spaceBase) {
						result.spaceBase = design;
						bscore.spaceBase = rating;
					}
				} else /* it's a star base */ {
					const rating = evaluateDesign(player,design,"defence",1e9);
					if (rating > bscore.spaceBase) {
						result.spaceBase = design;
						bscore.spaceBase = rating;
					}
					if (rating > bscore.starBase) {
						result.starBase = design;
						bscore.starBase = rating;
					}
				}
			}
		}

		return result;
	};
}