/*!
 * © 2022 David Vines
 *
 * domainOfTheAncients is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * domainOfTheAncients is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with domainOfTheAncients. If not, see https://www.gnu.org/licenses/gpl-2.0.html.
 */
/* globals Util, ShipDesigner, ShipDesignType, PlayerOrders */
/* exported BuildVessel */
class BuildVessel {
	constructor(typeMarker,designType,movableLocation,placeOrderFunction) {
		const self = this;
		self._modal = document.getElementById(`build${typeMarker}Modal`);
		self._locationSpan = document.getElementById(`build${typeMarker}ModalLocation`);
		self._choice = document.getElementById(`buy-${typeMarker}-choice`);
		self._name = document.getElementById(`new-${typeMarker}-name`);
		self._review = document.getElementById(`review-${typeMarker}-design`);
		self._design = document.getElementById(`design-new-${typeMarker}-class`);
		self._doit = document.getElementById(`buy-${typeMarker}-doit`);
		self._cost = document.getElementById(`buy-${typeMarker}-cost`);
		self._cancelPurchase = document.getElementById(`buy-${typeMarker}-cancel`);

		self._designType = designType;
		self._placeOrderFunction = placeOrderFunction;
		self._designs = [];

		self._choice.onchange = self._determineVehicleNameAndCost.bind(self);
		self._design.onclick = self._handleDesignNewVehicle.bind(self);
		self._review.onclick = self._handleReviewVehicleDesign.bind(self);

		self._doit.onclick = function() {
			const shipindex = self._choice.selectedIndex;
			const ship = self._designs[shipindex];
			const shipname = self._name.value;
			const location =self._locationSpan.innerHTML;

			self._placeOrderFunction(ship,shipname,location,self._cost.dataset.first,self._cost.innerText);

			self._modal.close();
		};
		self._cancelPurchase.onclick = function() {
			self._modal.close();
		};

		if (movableLocation) {
			self._locationButton = document.getElementById(`build${typeMarker}ChangeLocation`);
			self._locationInstructions = document.getElementById(`${typeMarker}Instructions`);

			self._locationButton.onclick = function() {
				self._locationInstructions.classList.remove("hidden");
				for(let row=1; row<21; row++) {
					for(let column of "ABCDEFGHIJKLMNOP") {
						const name = column+row;
						const hexelement = document.getElementById(name);
						if (self._player.map.hex(name).onGrid) {
							hexelement.onclick = function() { self._selectLocation(name); };
						} else {
							hexelement.classList.add("greyout");
						}
					}
				}
				self._modal.onclose = self._close.bind(self);
			};
		} else {
			self._locationButton = null;
		}
	}
	_selectLocation(location) {
		Util.replaceText(this._locationSpan,location);
		this._finishLocation();
	}
	_finishLocation() {
		this._locationInstructions.classList.add("hidden");
		for(let row=1; row<21; row++) {
			for(let column of "ABCDEFGHIJKLMNOP") {
				const name = column+row;
				const hexelement = document.getElementById(name);
				hexelement.onclick = null;
				hexelement.classList.remove("greyout");
			}
		}
		this._modal.onclose = null;
	}
	setPlayer(player) {
		this._player = player;
	}
	open(location) {
		Util.replaceText(this._locationSpan,location);
		this._listAvailableVehicleDesigns();
		this._modal.open();
		if (this._locationButton) {
			this._locationButton.onclick(); // Fake a click to get the map location selectable
		}
	}
	_close() {
		if (this._locationButton) {
			this._finishLocation();
		}
	}
	setStandardShipDesigns(list) {
		this._standardShipDesigns = list;
	}
	_listAvailableVehicleDesigns() {
		Util.emptyList(this._choice);

		const techs = this._player.getShipTechs();
		let validDesignTypes = [this._designType];
		if (this._designType == ShipDesignType.Ship) validDesignTypes.push(ShipDesignType.SpaceBase);
		if (this._designType == ShipDesignType.Ship || this._designType == ShipDesignType.SpaceBase) validDesignTypes.push(ShipDesignType.StarBase);
		this._designs = this._standardShipDesigns
			.concat(this._player.shipDesigns)
			.filter(design => validDesignTypes.includes(design.type))
			.filter(design => design.isTechSufficent(techs))
			.reverse();

		for(let design of this._designs) {
			Util.addOption(this._choice,design.name);
		}
		this._choice.selectedIndex = 0;
		this._determineVehicleNameAndCost();
	}
	_handleDesignNewVehicle() {
		ShipDesigner.newDesign(this._player.getNextDesignId(),
			this._player.getShipTechs(),
			this._hasEverBuiltAShip(),
			this._callbackWhenDesignComplete.bind(this));
	}
	_handleReviewVehicleDesign() {
		const vehicleIndex = this._choice.selectedIndex;
		const vehicle = this._designs[vehicleIndex];
		ShipDesigner.review(vehicle,this._getVehicleCost());
	}
	_hasEverBuiltAShip() {
		if (this._player.hasBuiltAShip) return true;
		return PlayerOrders.isAVesselOnOrder();
	}
	_callbackWhenDesignComplete(newDesign) {
		if (newDesign) {
			this._choice.selectedIndex = 0; // Reset to the first entry
			this._player.addShipDesign(newDesign);
			this._listAvailableVehicleDesigns();
		}
	}
	_determineVehicleNameAndCost() {
		this._determineVehicleName();
		this._determineVehicleCost();
	}
	_determineVehicleName() {
		const index = this._choice.selectedIndex;
		if (index !== -1) {
			let number = this._designs[index].numberOfClassBuilt(this._player) + 1 + PlayerOrders.getNumberOfVesselsOrdered(this._designs[index]);
			// Now check to see if we need to bump the number to avoid the name of any existing ship (or ship on order)
			let duplicate = true;
			let proposedName = this._designs[index].name + " " + number;
			while(duplicate) {
				duplicate = false;
				for(let ship of this._player.ships) {
					if (ship.name == proposedName) {
						duplicate = true; break;
					}
				}
				if (!duplicate) {
					duplicate = PlayerOrders.isNameOnAnOrder(proposedName);
				}
				if (duplicate) {
					number++;
					proposedName = this._designs[index].name + " " + number;
				}
			}
			this._name.value = proposedName;
		}
	}
	_determineVehicleCost() {
		const [cost,firstOfDesign] = this._getVehicleCost();
		Util.replaceText(this._cost,cost);
		if (firstOfDesign) {
			this._cost.dataset.first = "first";
		} else if (this._cost.dataset.first) {
			delete this._cost.dataset.first;
		}
	}
	_getVehicleCost() {
		const index = this._choice.selectedIndex;
		let cost = 0;
		let firstOfDesign = false;
		if (index !== -1) {
			cost = this._designs[index].totalCost();
			let number = this._designs[index].numberOfClassBuilt(this._player) + 1 + PlayerOrders.getNumberOfVesselsOrdered(this._designs[index]);
			// If the number is 1 AND this is a second or later ship we must double the price
			if (number === 1 && this._hasEverBuiltAShip()) {
				cost *= 2; firstOfDesign = true;
			}
		}
		return [cost, firstOfDesign];
	}
}