/*!
 * © 2021 David Vines
 *
 * Creative Commons License
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License.
 */

/* globals ROT */
/* exported OpeningMove, Contination, Bout */
class OpeningMove {
	constructor(name) {
		this._name = name;
	}
	toString() {
		return this._name;
	}
	getOdds(theirMove,us,them) {
		throw new Error("getOdds not provided for "+this.toString()+".getOdds("+theirMove.toString()+","+us.toString()+","+them.toString()+")");
	}
}
OpeningMove.Charge = new OpeningMove("Charge");
OpeningMove.Charge.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === OpeningMove.Charge:
			odds = 65; break;
		case theirMove === OpeningMove.Attack:
			odds = 30; break;
		case theirMove === OpeningMove.StepAside:
			odds = 90; break;
		case theirMove === OpeningMove.Slap:
			odds = 45; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for OpeningMove.Charge");
	}
	odds += (us.getSpeed() - them.getWeight())/5;
	return odds;
};
OpeningMove.Charge.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" charges "+them.toString();
};
OpeningMove.Attack = new OpeningMove("Attack");
OpeningMove.Attack.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === OpeningMove.Charge:
			odds = 90; break;
		case theirMove === OpeningMove.Attack:
			odds = 65; break;
		case theirMove === OpeningMove.StepAside:
			odds = 30; break;
		case theirMove === OpeningMove.Slap:
			odds = 60; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for OpeningMove.Attack");
	}
	odds += (us.getStrength() - them.getStrength())/5;
	return odds;
};
OpeningMove.Attack.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" attacks "+them.toString();
};
OpeningMove.StepAside = new OpeningMove("Step Aside");
OpeningMove.StepAside.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === OpeningMove.Charge:
			odds = 30; break;
		case theirMove === OpeningMove.Attack:
			odds = 90; break;
		case theirMove === OpeningMove.StepAside:
			odds = 65; break;
		case theirMove === OpeningMove.Slap:
			odds = 75; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for OpeningMove.StepAside");
	}
	odds += (us.getAgility() - them.getSpeed())/5;
	return odds;
};
OpeningMove.StepAside.getSuccessfulDescription = function(us,them,theirMove) {
	return us.toString()+" steps to the side and then pushes "+them.toString()+" from behind";
};
OpeningMove.Slap = new OpeningMove("Slap");
OpeningMove.Slap.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === OpeningMove.Charge:
			odds = 75; break;
		case theirMove === OpeningMove.Attack:
			odds = 60; break;
		case theirMove === OpeningMove.StepAside:
			odds = 45; break;
		case theirMove === OpeningMove.Slap:
			odds = 65; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for OpeningMove.Slap");
	}
	odds += (us.getStrength() - them.getWeight())/5;
	return odds;
};
OpeningMove.Slap.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" slaps down "+them.toString();
};

class Continuation {
	constructor(name) {
		this._name = name;
	}
	toString() {
		return this._name;
	}
	getOdds(theirMove,us,them) {
		throw new Error("getOdds not provided for "+this.toString()+".getOdds("+theirMove.toString()+","+us.toString()+","+them.toString()+")");
	}
}
Continuation.Lever = new Continuation("Lever");
Continuation.Lever.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === Continuation.Lever:
			odds = 65; break;
		case theirMove === Continuation.Lift:
			odds = 90; break;
		case theirMove === Continuation.Push:
			odds = 30; break;
		case theirMove === Continuation.Pull:
			odds = 45; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for Continuation.Lever");
	}
	odds += (us.getAgility() - them.getStrength())/5;
	return odds;
};
Continuation.Lever.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" levers "+them.toString();
};
Continuation.Lift = new Continuation("Lift");
Continuation.Lift.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === Continuation.Lever:
			odds = 30; break;
		case theirMove === Continuation.Lift:
			odds = 65; break;
		case theirMove === Continuation.Push:
			odds = 90; break;
		case theirMove === Continuation.Pull:
			odds = 60; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for Continuation.Lift");
	}
	odds += (us.getStrength() - them.getWeight())/5;
	return odds;
};
Continuation.Lift.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" lifts "+them.toString();
};
Continuation.Push = new Continuation("Push");
Continuation.Push.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === Continuation.Lever:
			odds = 90; break;
		case theirMove === Continuation.Lift:
			odds = 30; break;
		case theirMove === Continuation.Push:
			odds = 65; break;
		case theirMove === Continuation.Pull:
			odds = 75; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for Continuation.Push");
	}
	odds += (us.getWeight() - them.getWeight())/5;
	return odds;
};
Continuation.Push.getSuccessfulDescription = function(us,them,theirMove) {
	return us.toString()+" pushes "+them.toString();
};
Continuation.Pull = new Continuation("Pull");
Continuation.Pull.getOdds = function(theirMove,us,them) {
	let odds;
	switch(true) {
		case theirMove === Continuation.Lever:
			odds = 75; break;
		case theirMove === Continuation.Lift:
			odds = 60; break;
		case theirMove === Continuation.Push:
			odds = 45; break;
		case theirMove === Continuation.Pull:
			odds = 65; break;
		default:
			throw new Error("Unknown countermove "+theirMove.toString()+" when determing odds for Continuation.Pull");
	}
	odds += (us.getStrength() - them.getWeight())/5;
	return odds;
};
Continuation.Pull.getSuccessfulDescription = function(us,them,_theirMove) {
	return us.toString()+" pulls "+them.toString();
};

class Result {
	constructor() {
		this._winner = null;
		this._loser = null;
		this._text = [];
	}
	setWinner(winner) {
		this._winner = winner;
	}
	getWinner() {
		return this._winner;
	}
	setLoser(loser) {
		this._loser = loser;
	}
	getLoser() {
		return this._loser;
	}
	getText() {
		return this._text;
	}
	addText(text) {
		this._text.push(text);
	}
}

class Bout {
	static resolve(east,west) {
		let result = Bout._resolveOpening(east,west);
		while(result.getWinner() === null) {
			Bout._resolveContination(result,east,west);
		}
		return result;
	}
	static _resolveOpening(east,west) {
		let result = new Result();
		const eastopen = east.getOpeningMove(west);
		const westopen = west.getOpeningMove(east);
		const eastopensuccess = (ROT.RNG.getPercentage() <= eastopen.getOdds(westopen,east,west));
		const westopensuccess = (ROT.RNG.getPercentage() <= eastopen.getOdds(westopen,east,west));

		if (eastopensuccess && !westopensuccess) {
			if (west.saves()) {
				result.addText(eastopen.getSuccessfulDescription(east,west,westopen)+" but "+west.toString()+" recovers.");
			} else {
				result.setWinner(east);	result.setLoser(west);
				result.addText(eastopen.getSuccessfulDescription(east,west,westopen)+" winning the bout!");
			}
		}
		if (westopensuccess && !eastopensuccess) {
			if (east.saves()) {
				result.addText(westopen.getSuccessfulDescription(west,east,eastopen)+" but "+east.toString()+" recovers.");
			} else {
				result.setWinner(west); result.setLoser(east);
				result.addText(westopen.getSuccessfulDescription(west,east,eastopen)+" winning the bout!");
			}
		}

		if (!eastopensuccess && !westopensuccess) {
			if (eastopen !== OpeningMove.StepAside && westopen !== OpeningMove.StepAside) {
				result.addText(east.toString()+" tries to "+eastopen.toString()+", while "+west.toString()+" tries to "+westopen.toString()+". Neither man succeeds.");
			} else if (eastopen === OpeningMove.StepAside && westopen === OpeningMove.StepAside) {
				result.addText("Both "+east.toString()+" and "+west.toString()+" try to dodge each other's opening, result in some circling of the dohyō.");
			} else if (eastopen === OpeningMove.StepAside){
				result.addText(east.toString()+" steps to the side as "+west.toString()+" tries to "+westopen.toString()+".");
			} else {
				result.addText(east.toString()+" tries to "+eastopen.toString()+" while "+west.toString()+" steps to the side.");
			}
		}

		if (eastopensuccess && westopensuccess) {
			const eastsaves = east.saves();
			const westsaves = west.saves();
			const initialText = eastopen.getSuccessfulDescription(east,west,westopen) + " and "+westopen.getSuccessfulDescription(west,east,eastopen)+".";
			if (eastsaves && !westsaves) {
				result.setWinner(east); result.setLoser(west);
				result.addText(initialText+" Only "+east.toString()+" recovers and thus wins the bout!");
			}
			if (!eastsaves && westsaves) {
				result.setWinner(west); result.setLoser(east);
				result.addText(initialText+" Only "+west.toString()+" recovers and thus wins the bout!");
			}
			if (eastsaves && westsaves) {
				result.addText(initialText+" Both rikishi recover.");
			}
			if (!eastsaves && !westsaves) {
				let eastdelay = ROT.RNG.getPercentage()+east.getBalance()/5+ROT.RNG.getUniform()/5;
				let westdelay = ROT.RNG.getPercentage()+west.getBalance()/5+ROT.RNG.getUniform()/5;
				if (eastdelay > westdelay) {
					result.setWinner(east); result.setLoser(west);
					result.addText(initialText+" "+west.toString()+" hits the clay first and so "+east.toString()+" wins the bout!");
				} else {
					result.setWinner(west); result.setLoser(east);
					result.addText(initialText+" "+east.toString()+" hits the clay first and so "+west.toString()+" wins the bout!");
				}
			}
		}

		return result;
	}
	static _resolveContination(result,east,west) {
		const eastcont = east.getContinuation(west);
		const westcont = west.getContinuation(east);
		const eastcontsuccess = (ROT.RNG.getPercentage() <= eastcont.getOdds(westcont,east,west));
		const westcontsuccess = (ROT.RNG.getPercentage() <= eastcont.getOdds(westcont,east,west));

		if (eastcontsuccess && !westcontsuccess) {
			if (west.saves()) {
				result.addText(eastcont.getSuccessfulDescription(east,west,westcont)+" but "+west.toString()+" recovers.");
			} else {
				result.setWinner(east); result.setLoser(west);
				result.addText(eastcont.getSuccessfulDescription(east,west,westcont)+" winning the bout!");
			}
		}
		if (westcontsuccess && !eastcontsuccess) {
			if (east.saves()) {
				result.addText(westcont.getSuccessfulDescription(west,east,eastcont)+" but "+east.toString()+" recovers.");
			} else {
				result.setWinner(west); result.setLoser(east);
				result.addText(westcont.getSuccessfulDescription(west,east,eastcont)+" winning the bout!");
			}
		}

		if (!eastcontsuccess && !westcontsuccess) {
			result.addText(east.toString()+" tries to "+eastcont.toString()+", while "+west.toString()+" tries to "+westcont.toString()+". Neither man succeeds.");
		}

		if (eastcontsuccess && westcontsuccess) {
			const eastsaves = east.saves();
			const westsaves = west.saves();
			const initialText = eastcont.getSuccessfulDescription(east,west,westcont) + " and "+westcont.getSuccessfulDescription(west,east,eastcont)+".";
			if (eastsaves && !westsaves) {
				result.setWinner(east); result.setLoser(west);
				result.addText(initialText+" Only "+east.toString()+" recovers and thus wins the bout!");
			}
			if (!eastsaves && westsaves) {
				result.setWinner(west); result.setLoser(east);
				result.addText(initialText+" Only "+west.toString()+" recovers and thus wins the bout!");
			}
			if (eastsaves && westsaves) {
				result.addText(initialText+" Both rikishi recover.");
			}
			if (!eastsaves && !westsaves) {
				let eastdelay = ROT.RNG.getPercentage()+east.getBalance()/5+ROT.RNG.getUniform()/5;
				let westdelay = ROT.RNG.getPercentage()+west.getBalance()/5+ROT.RNG.getUniform()/5;
				if (eastdelay > westdelay) {
					result.setWinner(east); result.setLoser(west);
					result.addText(initialText+" "+west.toString()+" hits the clay first and so "+east.toString()+" wins the bout!");
				} else {
					result.setWinner(west); result.setLoser(east);
					result.addText(initialText+" "+east.toString()+" hits the clay first and so "+west.toString()+" wins the bout!");
				}
			}
		}

		return result;
	}
}