<?php
/*
Plugin Name: Linrodeth Character Sheet
Plugin URI: http://blog.tremlas.com/linrodeth-character-sheet-plugin-details/
Description: Adds LinrodethCharacterSheet as a post type and adds display logic
Version: 3.26.0
Author: David Vines
Author URI: http://tremlas.com
License:  Creative Commons License
*/
/* © Copyright 2014, 2017 David Vines

   Creative Commons License

   This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License.
*/

// Debugging function in case we need it
if ( !function_exists('djv_write_log') ) {
	function djv_write_log ( $log )  {
		if ( true === WP_DEBUG ) {
			if ( is_array( $log ) || is_object( $log ) ) {
				error_log( print_r( $log, true ) );
			} else {
				error_log( $log );
			}
		}
	}
}

// Initial option values
add_option( 'djv_linrodethCharacterSheet_options', array(
		'edit_preamble' => '<ul><li><b>Note:</b> A work in progress - Now on version 3.26.0. Based on Linrodeth rule set 2014a.</li><li><b>Note:</b> Each period needs a different title</li></ul>',
		'view_preamble' => '<ul><li><b>Note:</b> A work in progress - Now on version 3.26.0. Based on Linrodeth rule set 2014a.</li></ul>',
		'default_genre' => 'Linrodeth',
		'gametime' => '1222869228031'
	)
);

function djv_linrodethCharacterSheet_upgrade_option() {
 $changed = false;
 $genre_timefields = array( 'lin_gametime', 'luna_gametime', 'con_gametime' );
 $options = get_option('djv_linrodethCharacterSheet_options');
 
 if ( !isset( $options['gametime'] ) ) {
  $options['gametime'] = '1222869228031'; // Will be upgraded in the next step!
  $changed = true;
 }
 
 if ( isset( $options['gametime'] ) ) {
  $was = $options['gametime'];
  foreach( $genre_timefields as $field ) {
   if (!isset( $options[$field] ) ) {
    $options[$field] = $was;
    $changed = true; 
   }
  }
 }
 if ( $changed ) {
  update_option( 'djv_linrodethCharacterSheet_options', $options );
 }
}
register_activation_hook( __FILE__, 'djv_linrodethCharacterSheet_upgrade_option');

// Create a custom post type
function djv_linrodethCharacterSheet_register_post_types() {
	$character_args = array(
			'public' => true,
			'query_var' => 'character_sheet',
			'rewrite' => array(
					'slug' => 'character',
					'with_front' => false
			),
			'supports' => array(
					'title',
					'author',
					'revisions',
					'editor',
					'comments',
					'excerpt',
					'tags'
			),
			'labels' => array(
					'name' => 'Characters',
					'singular_name' => 'Character',
					'add_new' => 'Add New Character',
					'add_new_item' => 'Add New Character',
					'edit_item' => 'Edit Character',
					'new_item' => 'New Character',
					'view_item' => 'View Character',
					'search_items' => 'Search Characters',
					'not_found' => 'No Characters found',
					'not_found_in_trash' => 'No Characters found in trash'
			),
			'taxonomies' => array('post_tag')
	);
	
	register_post_type( 'linrodeth_characters', $character_args);
}

// and register it on initialisation
add_action( 'init', 'djv_linrodethCharacterSheet_register_post_types' );

// A widget that shows the latest (5 by default) characters
function djv_linrodethCharacterSheet_register_widgets() {
	register_widget('djv_linrodethCharacterSheet_listLatest_widget');
}

class djv_linrodethCharacterSheet_listLatest_widget extends WP_Widget {
	function __construct() {
		$widget_ops = array(
				'classname' => 'djv_linrodethCharacterSheet_listLatest_widget_class',
				'description' => 'Displays a list of the most recent changed linrodeth characters'
		);
		parent::__construct('djv_linrodethCharacterSheet_listLatest_widget', 'Latest Linrodeth Characters Widget', $widget_ops);
	}
	
	function form($instance) {
		$defaults = array( 'title' => 'Recent Linrodeth Characters', 'numberOfCharacters' => '5');
		$instance = wp_parse_args( (array) $instance, $defaults);
		$title = $instance['title'];
		$numberOfCharacters = $instance['numberOfCharacters'];
		$addDate = $instance['addDate'];
		?>
			<p>Title: <input class="widefat" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" /></p>
			<p>Number of Characters: <input class="widefat" name="<?php echo $this->get_field_name('numberOfCharacters'); ?>" type="text" value="<?php echo esc_attr( $numberOfCharacters ); ?>" /></p>
			<p>Append date of last update: <input name="<?php echo $this->get_field_name('addDate'); ?>" type="checkbox" <?php checked( $addDate , 'on'); ?>/></p>
		<?php
	}
	
	function update($new_instance, $old_instance) {
		$instance = $old_instance;
		$instance['title'] = strip_tags($new_instance['title']);
		$instance['numberOfCharacters'] = strip_tags($new_instance['numberOfCharacters']);
		$instance['addDate'] = strip_tags($new_instance['addDate']);
		return $instance;
	}
	
	function widget($args, $instance) {
		extract($args);
		$title = apply_filters('widget_title', $instance['title']);
		$numberOfCharacters = $instance['numberOfCharacters'];
		$addDate = $instance['addDate'];
		echo $before_widget;
		if ( $title ) {
			echo $before_title . $title . $after_title;
		}
		$this->getCharacters($numberOfCharacters , $addDate);
		echo $after_widget;
	}
	
	function getCharacters($numberOfCharacters , $addDate) {
		global $post;
		$characters = new WP_Query();
		$characters->query('post_type=linrodeth_characters&posts_per_page=' . $numberOfCharacters );
		if($characters->found_posts > 0) {
			echo '<ul class="character_widget">';
			while ($characters->have_posts()) {
				$characters->the_post();
				$listItem = '<li>';
				$listItem .= '<a href="' . get_permalink() . '" title="' . esc_attr( get_the_title() ? get_the_title() : get_the_ID() ) . '">';
				if ( get_the_title() )
					$listItem .= get_the_title();
				else
					$listItem .= get_the_ID();
				$listItem .= '</a>';
				if ( $addDate ) {
					$listItem .= '<span>Added ' . get_the_date() . '</span></li>';
				}
				echo $listItem;
			}
			echo '</ul>';
			wp_reset_postdata();
		}else{
			echo '<p style="padding:25px;">No characters found</p>';
		}
	}
}

// Register the latest characters widget when the widgets get initialised
add_action( 'widgets_init', 'djv_linrodethCharacterSheet_register_widgets' );

// An admin page to allow the preamble texts (and other settings to be modified
class DjvLinrodethCharacterSheetSettingsPage
{
	private $options;
	private $genres = array( 'Linrodeth',    'Luna Romana',   'Contemporary' );
	private $fields = array( 'lin_gametime', 'luna_gametime', 'con_gametime' );
	
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_plugin_page' ) );
		add_action( 'admin_init', array( $this, 'page_init' ) );
	}
	
	public function add_plugin_page() {
		add_options_page(
			'Linrodeth Character Sheet Admin',
			'Linrodeth Character Sheet',
			'manage_options',
			'djv-linrodethcharactersheet-setting-admin',
			array( $this, 'create_admin_page' ) 
		);
	}
	
	public function create_admin_page() {
		$this->options = get_option('djv_linrodethCharacterSheet_options');
		?>
		<div class="wrap">
			<?php screen_icon(); ?>
			<h2>Linrodeth Character Sheet Options</h2>
			<form method="post" action="options.php">
			<?php
				settings_fields('djv_linrodethCharacterSheet_option_group');
				do_settings_sections('djv-linrodethcharactersheet-setting-admin');
				submit_button(); 
			?>
			</form>
		</div>
		<?php
	}
	
	public function page_init() {
		register_setting(
			'djv_linrodethCharacterSheet_option_group',
			'djv_linrodethCharacterSheet_options',
			array( $this, 'sanitize' )
		);
		add_settings_section(
			'djv_linrodethCharacterSheet_settings_section_id',
			'Linrodeth Character Sheet Settings',
			array( $this, 'print_section_info' ),
			'djv-linrodethcharactersheet-setting-admin'
		);
		add_settings_field(
			'view_preamble',
			'Preamble Text when viewing',
			array( $this, 'view_preamble_callback' ),
			'djv-linrodethcharactersheet-setting-admin',
			'djv_linrodethCharacterSheet_settings_section_id'
		);
		add_settings_field(
			'edit_preamble',
			'Preamble Text when editing',
			array( $this, 'edit_preamble_callback' ),
			'djv-linrodethcharactersheet-setting-admin',
			'djv_linrodethCharacterSheet_settings_section_id'
		);
		add_settings_field(
			'default_genre',
			'Default genre for new characters',
			array( $this, 'default_genre_callback' ),
			'djv-linrodethcharactersheet-setting-admin',
			'djv_linrodethCharacterSheet_settings_section_id'
		);
		for( $i = 0, $size = count($this->genres); $i < $size; ++$i) {
		    add_settings_field(
			    $this->fields[$i],
			    $this->genres[$i] . ' Game Date',
			    array( $this, 'gametime_callback' ),
			    'djv-linrodethcharactersheet-setting-admin',
			    'djv_linrodethCharacterSheet_settings_section_id',
		        array( 'genre' => $this->genres[$i], 'field' => $this->fields[$i] )
		    );
		}
	}
	
	public function sanitize( $input ) {
		$new_input = array();
		if ( isset( $input['view_preamble'] ) ) {
			$new_input['view_preamble'] = $input['view_preamble'];
		}
		if ( isset( $input['edit_preamble'] ) ) {
			$new_input['edit_preamble'] = $input['edit_preamble'];
		}
		if ( isset( $input['default_genre'] ) ) {
			$new_input['default_genre'] = 'Linrodeth'; // default if not understood
			for( $i = 0; $i < count( $this->genres ); $i++ ) {
				if ( $input['default_genre'] == $this->genres[ $i ] ) {
					$new_input['default_genre'] = $this->genres[ $i ];
				}
			}
		}
		
		for( $i = 0, $size = count($this->fields); $i < $size; ++$i) {
		    if ( isset( $input[$this->fields[$i]] ) && is_numeric( $input[$this->fields[$i]] ) ) {
		        $new_input[$this->fields[$i]] = $input[$this->fields[$i]];
		    }
		}
		return $new_input;
	}
	
	public function print_section_info() {
		print "Enter the settings for the linrodeth character plugin. Preamble text is the text that appears before the actual character sheet.";
	}
	
	public function view_preamble_callback() {
		$settings = array(
			'textarea_name' => 'djv_linrodethCharacterSheet_options[view_preamble]',
			'teeny' => true,
			'textarea_rows' => 6,
			'tabindex' => 1,
			'media_buttons' => false
		);
		wp_editor(
			isset( $this->options['view_preamble'] ) ? $this->options['view_preamble'] : '',
			'view_preamble',
			$settings
		);
	}
	
	public function edit_preamble_callback() {
		$settings = array(
			'textarea_name' => 'djv_linrodethCharacterSheet_options[edit_preamble]',
			'teeny' => true,
			'textarea_rows' => 6,
			'tabindex' => 1,
			'media_buttons' => false
		);
		wp_editor(
			isset( $this->options['edit_preamble'] ) ? $this->options['edit_preamble'] : '',
			'edit_preamble',
			$settings
		);
	}
	
	public function default_genre_callback() {
		$value = isset( $this->options['default_genre'] ) ? $this->options['default_genre'] : 'Linrodeth';
		$html = '<select name="djv_linrodethCharacterSheet_options[default_genre]" id="djv_genre">'  . PHP_EOL;
		for( $i = 0; $i < count( $this->genres ); $i++ ) {
			if ( $value == $this->genres[ $i ] ) {
				$html .= '<option selected>';
			} else {
				$html .= '<option>';
			}
			$html .=  $this->genres[ $i ] . '</option>'  . PHP_EOL;
		} 
		$html .= '</select>'  . PHP_EOL;
		echo $html;
	}
	
	public function gametime_callback( $args ) {
	    $genre = $args['genre'];
	    $field = $args['field'];
	    
		$plugin_url = plugin_dir_url(__FILE__);
		wp_enqueue_script('djv_linrodethCharacterSheet_lindate', $plugin_url.'js/linrodeth/lindate.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_genre', $plugin_url.'js/linrodethchargen/genre.js', array( 'djv_linrodethCharacterSheet_lindate' ) );	
		wp_enqueue_script('djv_linrodethCharacterSheet_editgamedate', $plugin_url.'js/linrodethchargen/editgamedate.js', array( 'djv_linrodethCharacterSheet_genre' ) );	
		$value = isset( $this->options[$field] ) ? $this->options[$field] : time()*1000;
		echo '<button type="submit" id="' . $field .'-gamedayminus1year">&lt;&lt;&lt;</button>' . PHP_EOL;
		echo '<button type="submit" id="' . $field .'-gamedayminus1month">&lt;&lt;</button>' . PHP_EOL;
		echo '<button type="submit" id="' . $field .'-gamedayminus1day">&lt;</button>' . PHP_EOL;
		echo '<span id="' . $field .'-gameday" style="display: inline-block; width: 32ch;"></span>' . PHP_EOL;
		echo '<script type="text/javascript">' . PHP_EOL;
		echo 'var gametime = gametime || {};' . PHP_EOL;
		echo 'gametime.' . $field . ' = ' . $value . ';' . PHP_EOL;
		echo 'var genrenames = genrenames || {};' . PHP_EOL;
		echo 'genrenames.' . $field . ' = "' . $genre . '";' . PHP_EOL;
		echo '</script>' . PHP_EOL;
		echo '<button type="submit" id="' . $field .'-gamedayplus1day">&gt;</button>' . PHP_EOL;
		echo '<button type="submit" id="' . $field .'-gamedayplus1month">&gt;&gt;</button>' . PHP_EOL;
		echo '<button type="submit" id="' . $field .'-gamedayplus1year">&gt;&gt;&gt;</button>' . PHP_EOL;
		echo '<div style="display: none;">' . PHP_EOL;
		echo '<textarea id="' . $field .'-newgametime" name="djv_linrodethCharacterSheet_options[' . $field . ']">' . $value . '</textarea>' . PHP_EOL;
		echo '</div>' . PHP_EOL;
	}
}
if (is_admin()) {
	$djv_linrodethCharacterSheet_settings_page = new DjvLinrodethCharacterSheetSettingsPage();
}

// Also provide a link on the plugins listings page
function djv_add_plugin_action_links( $links ) {
	return array_merge(
			array( 'settings' => '<a href="' . get_bloginfo( 'wpurl' ) . '/wp-admin/options-general.php?page=djv-linrodethcharactersheet-setting-admin">Settings</a>'
			),
			$links
		);
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'djv_add_plugin_action_links' );

// Use our javascript when displaying characters
function djv_linrodethCharacterSheet_empty_the_content($content) {
	if ( get_post()->post_type !== 'linrodeth_characters' )
		return $content;
	return ""; // Skip the contents for now, we'll add some real stuff once all the other the_contents filters have run
}

function djv_linrodethCharacterSheet_the_content($content) {
	if ( get_post()->post_type !== 'linrodeth_characters' )
			return $content;

    $genres = array( 'Linrodeth',    'Luna Romana',   'Contemporary' );
    $fields = array( 'lin_gametime', 'luna_gametime', 'con_gametime' );
	
    static $id = 0; $id++;
	
	$plugin_url = plugin_dir_url(__FILE__);
	wp_enqueue_script('djv_linrodethCharacterSheet_blob', $plugin_url.'js/blob/Blob.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_filesaver', $plugin_url.'js/filesaver/FileSaver.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_djvtabber', $plugin_url.'js/djvtabber/djvtabber.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF', $plugin_url.'js/jsPDF/jspdf.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_metrics', $plugin_url.'js/jsPDF/jspdf.plugin.standard_fonts_metrics.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_split', $plugin_url.'js/jsPDF/jspdf.plugin.split_text_to_size.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_djv', $plugin_url.'js/jsPDF/jspdf.plugin.djv.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_lindate', $plugin_url.'js/linrodeth/lindate.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_rules', $plugin_url.'js/linrodethchargen/rules.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_genre', $plugin_url.'js/linrodethchargen/genre.js' );	
	wp_enqueue_script('djv_linrodethCharacterSheet_common', $plugin_url.'js/linrodethchargen/common.js',[],'',true );
	wp_enqueue_script('djv_linrodethCharacterSheet_character', $plugin_url.'js/linrodethchargen/viewcharacter.js',[],'',true );
	
	wp_enqueue_style('djv_linrodethCharacterSheet_djvstyle', $plugin_url.'css/djvstyle.css');
	wp_enqueue_style('djv_linrodethCharacterSheet_djvtabber', $plugin_url.'js/djvtabber/djvtabber.css');
	
	$reply = '<script id="c' . $id . '-character-data" type="application/json">' . PHP_EOL;
    $reply .= get_post()->post_content . PHP_EOL; // We need the raw content, not the html-ified version - luckily we're throwing all that away :)
    $reply .= '</script>' . PHP_EOL;
    
	$reply .= '<script type="text/javascript">' . PHP_EOL;
	$reply .= 'var options = options || {};' . PHP_EOL;
	$reply .= 'options.ids = options.ids || [];' . PHP_EOL;
	$reply .= 'options.ids.push(' . $id . ');' . PHP_EOL;
	$reply .= 'options.gametime = options.gametime || {};' . PHP_EOL;
	for( $i = 0, $size = count($genres); $i < $size; ++$i) {
	 $reply .= 'options.gametime["' . $genres[$i] . '"] = ' . get_option('djv_linrodethCharacterSheet_options')[$fields[$i]] . ';' . PHP_EOL;
	}
	$reply .= 'options.default_genre = "' . get_option('djv_linrodethCharacterSheet_options')['default_genre'] . '";' . PHP_EOL;
	$reply .= '</script>' . PHP_EOL;
    	
	$file = plugin_dir_path(__FILE__);
	$filecontent = file_get_contents($file.'/html/view.html');
	$filecontent = str_replace('@@@VIEWPREAMBLE@@@',get_option('djv_linrodethCharacterSheet_options')['view_preamble'],$filecontent);
	$filecontent = str_replace('@@@AUTHORNAME@@@',get_the_author_meta("user_nicename"),$filecontent);
	$filecontent = str_replace('@@@ID@@@', $id, $filecontent );
	
	$reply .= $filecontent;

	return $reply;
}

add_filter( 'the_content', 'djv_linrodethCharacterSheet_empty_the_content', 9);
add_filter( 'the_content', 'djv_linrodethCharacterSheet_the_content', 120);

function djv_linrodethCharacterSheet_get_the_excerpt( $output ) {
	if ( get_post()->post_type !== 'linrodeth_characters' )
		return $output;
	
	if (! has_excerpt() ) {
		// No manual excerpt, so build one from the character description
		// First find the schemaVersion
		$output = get_post()->post_content;
		$output = substr($output, strpos($output, "schemaVersion")+13);
		// Now skip past the next description
		$output = substr($output, strpos($output, "description")+11);
		// $output should now start with ": "text... - but the space is optional (some characters stored with no white space in their JSON)
		$output = substr(trim(substr($output, 2)),1);
		// If there's a quote, just take the text up to that quote
		if (strpos($output, '"') !== FALSE) {
			$output = substr($output, 0 , strpos($output, '"'));
		} else {
			// Just take the text up to just before the next comma
			$output = substr($output, 0 , strpos($output, ',')-1);
		}
		$output = '<p>' . $output . '</p>'; 
	}
	
	// And in both cases add a link to the character sheet
	$output .= '<p><a href="' . get_permalink() . '">Character Sheet</a></p>';
	return $output;
}

add_filter( 'get_the_excerpt', 'djv_linrodethCharacterSheet_get_the_excerpt');

// Add a meta box for the charater editor page
function djv_linrodethCharacterSheet_addScriptsAndStyles_meta() {
	global $typenow;
	if ( $typenow == 'linrodeth_characters' ) {
		$plugin_url = plugin_dir_url(__FILE__);
		wp_enqueue_script('djv_linrodethCharacterSheet_djvpreedit', $plugin_url.'js/linrodethchargen/preeditcharacter.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_blob', $plugin_url.'js/blob/Blob.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_filesaver', $plugin_url.'js/filesaver/FileSaver.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_djvtabber', $plugin_url.'js/djvtabber/djvtabber.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF', $plugin_url.'js/jsPDF/jspdf.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_metrics', $plugin_url.'js/jsPDF/jspdf.plugin.standard_fonts_metrics.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_split', $plugin_url.'js/jsPDF/jspdf.plugin.split_text_to_size.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_jsPDF_djv', $plugin_url.'js/jsPDF/jspdf.plugin.djv.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_lindate', $plugin_url.'js/linrodeth/lindate.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_rules', $plugin_url.'js/linrodethchargen/rules.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_genre', $plugin_url.'js/linrodethchargen/genre.js' );
		wp_enqueue_script('djv_linrodethCharacterSheet_common', $plugin_url.'js/linrodethchargen/common.js',[],'',true );
		wp_enqueue_script('djv_linrodethCharacterSheet_character', $plugin_url.'js/linrodethchargen/editcharacter.js',[],'',true );
		
		wp_enqueue_style('djv_linrodethCharacterSheet_djvstyle', $plugin_url.'css/djvstyle.css');
		wp_enqueue_style('djv_linrodethCharacterSheet_djvtabber', $plugin_url.'js/djvtabber/djvtabber.css');
	}
}

add_action( 'admin_print_styles', 'djv_linrodethCharacterSheet_addScriptsAndStyles_meta' );

// And the contents of the meta box
function djv_linrodethCharacterSheet_meta( $post ) {
    $genres = array( 'Linrodeth',    'Luna Romana',   'Contemporary' );
    $fields = array( 'lin_gametime', 'luna_gametime', 'con_gametime' );
 
	static $id = 0; $id++;
	
	$content = '<script id="c' . $id . '-character-data" type="application/json">' . PHP_EOL;
	$content .= $post->post_content . PHP_EOL;
	$content .= '</script>' . PHP_EOL;
	
	$content .= '<script type="text/javascript">' . PHP_EOL;
	$content .= 'var options = options || {};' . PHP_EOL;
	$content .= 'options.ids = options.ids || [];' . PHP_EOL;
	$content .= 'options.ids.push(' . $id . ');' . PHP_EOL;
	$content .= 'options.gametime = options.gametime || {};' . PHP_EOL;
	for( $i = 0, $size = count($genres); $i < $size; ++$i) {
	 $content .= 'options.gametime["' . $genres[$i] . '"] = ' . get_option('djv_linrodethCharacterSheet_options')[$fields[$i]] . ';' . PHP_EOL;
	}
	$content .= 'options.default_genre = "' . get_option('djv_linrodethCharacterSheet_options')['default_genre'] . '";' . PHP_EOL;
	$content .= '</script>' . PHP_EOL;
	
	$file = plugin_dir_path(__FILE__);
	$filecontent = file_get_contents($file.'/html/edit.html');
	$filecontent = str_replace('@@@EDITPREAMBLE@@@',get_option('djv_linrodethCharacterSheet_options')['edit_preamble'],$filecontent);
	$filecontent = str_replace('@@@AUTHORNAME@@@',get_the_author_meta("user_nicename",$post->post_author),$filecontent);
	$filecontent = str_replace('@@@ID@@@', $id, $filecontent );
	$content .= $filecontent;
		
	echo $content;
}

function djv_linrodethCharacterSheet_page_meta_boxes() {
	global $_wp_post_type_features;
	
	add_meta_box('djv_linrodethCharacterSheet_editor','Character Sheet','djv_linrodethCharacterSheet_meta', 'linrodeth_characters', 'normal', 'high',
        array(
            '__block_editor_compatible_meta_box' => false,
        )
	);
	
	if (isset($_wp_post_type_features['linrodeth_characters']['editor']) && $_wp_post_type_features['linrodeth_characters']['editor']) {
		unset($_wp_post_type_features['linrodeth_characters']['editor']);
	}
}

add_action( 'add_meta_boxes', 'djv_linrodethCharacterSheet_page_meta_boxes' );

// And now the processing of the meta box - note it changes the post itself to the JSON
function djv_linrodethCharacterSheet_save_meta( $post_id ) {
	if ( isset($_POST['post_type']) && $_POST['post_type'] == 'linrodeth_characters') {
		if ( isset( $_POST['djv_linrodethCharacterSheet_JSON'] ) ) {
			$post = array(
				'ID' => $post_id,
				'post_content' => strip_tags( $_POST['djv_linrodethCharacterSheet_JSON'])
			);
			// unhook this function so it doesn't loop infinitely
			remove_action( 'save_post', 'djv_linrodethCharacterSheet_save_meta' );
			wp_update_post( $post );
			add_action( 'save_post', 'djv_linrodethCharacterSheet_save_meta' );
		}
		if ( isset( $_POST['djv_linrodethCharacterSheet_genre'] ) ) {
			update_post_meta( $post_id, '_djv_linrodethCharacterSheet_genre', strip_tags( $_POST['djv_linrodethCharacterSheet_genre']));
		}
			if ( isset( $_POST['djv_linrodethCharacterSheet_name'] ) ) {
			update_post_meta( $post_id, '_djv_linrodethCharacterSheet_name', strip_tags( $_POST['djv_linrodethCharacterSheet_name']));
		}
		if ( isset( $_POST['djv_linrodethCharacterSheet_playername'] ) ) {
			update_post_meta( $post_id, '_djv_linrodethCharacterSheet_playerName', strip_tags( $_POST['djv_linrodethCharacterSheet_playername']));
		}
	}
}

add_action( 'save_post', 'djv_linrodethCharacterSheet_save_meta' );

// Update the main search to include all post types
function djv_query_post_type($query) {
	if( $query->is_category() || $query->is_tag() && empty( $query->query_vars['suppress_filters'] ) ) {
		// Get all post types
		$post_types = get_post_types();
		$query->set( 'post_type', $post_types );
		return $query;
	}
	if ( !$query->is_admin() && $query->is_main_query() ) {
		if ( $query->is_search() ) {
			$post_types = get_post_types();
			$query->set('post_type', $post_types );
		}
	}
}
add_filter('pre_get_posts', 'djv_query_post_type');

/** 
 * Shortcode to add the version of the plugin
 */
function djv_linrodethCharacterSheet_lincharVersion_shortcode() {
	return 'Linrodeth Character Sheet Version 3.26.0';
} 
add_shortcode( 'lincharVersion', 'djv_linrodethCharacterSheet_lincharVersion_shortcode' );

/**
 * Shortcode to add a link to a linrodeth character or add a table of characters.
 * 
 * Attributes:
 * <dl>
 * <dt>type</dt><dd>Add a single "link" or a "table" - defaults to link (if more than one character is found, the link will be to the first one returned by the database)</dd>
 * <dt>name</dt><dd>Specifies the name of the desired character (as specified on the character sheet), defaults to unspecified</dd>
 * <dt>player</dt><dd>Specifies the player of the desired character, defaults to unspecifed</dd>
 * <dt>genre</dt><dd>Specifies the genre of the desired character, defaults to unspecified.</dd>
 * <dt>author</dt><dd>Specifies the author of the desired character, defaults to unspecifed.</dd>
 * <dt>text</dt><dd>Specifes the text of the link, defaults to the embedded text of the shortcode if any (ignored for tables).</dd>
 * <dt>orderby</dt><dd>Specifies the order of the table, can be specified as "name" or "date".</dd>
 * <dl>
 */

// Now for a shortcode that can be used to get links and/or tables for the characters
function djv_linrodethCharacterSheet_linchar_shortcode( $attr, $content = null ) {
	if ( isset($attr) && isset( $attr['type'] ) && $attr['type'] == 'table' ) {
		$type = 'table';
	} else {
		$type = 'link';
	}
	$meta = array();
	if ( isset($attr) && count($attr)>0 ) {
		if ( isset( $attr['name'] ) ) {
			array_push($meta,array(
				'key' => '_djv_linrodethCharacterSheet_name',
				'value' => $attr['name']
			));
		}
			if ( isset( $attr['player'] ) ) {
			array_push($meta,array(
				'key' => '_djv_linrodethCharacterSheet_playerName',
				'value' => $attr['player']
			));
		}
			if ( isset( $attr['genre'] ) ) {
			array_push($meta,array(
				'key' => '_djv_linrodethCharacterSheet_genre',
				'value' => $attr['genre']
			));
		}
	}
	$characters = new WP_Query();
	$args = array(
			'post_type' => 'linrodeth_characters',
			'meta_query' => $meta,
			'nopaging' => true
	);
	if ( isset($attr) && count($attr)>0 ) {
		if ( isset( $attr['author'] ) ) {
			$args['author_name'] = $attr['author'];
		}
	}
	if ( isset($attr) && count($attr)>0 ) {
		if ( isset( $attr['orderby'] ) ) {
			if ( $attr['orderby'] == 'date' ) {
				$args['orderby'] = 'modified date';
				$args['order'] = 'DESC';
			}
					if ( $attr['orderby'] == 'name' ) {
				$args['orderby'] = 'meta_value title';
				$args['order'] = 'ASC';
			}
		}
	}
	$characters->query( $args );
	
	$text = '';
	if ($type == 'table' ) {
		if ( $characters->found_posts > 0) {
			$text = '<table><tr><th>Character</th><th>Genre</th><th>Player</th><th>Author</th><th>Last Update</th</tr>';
			while ($characters->have_posts()) {
				$characters->the_post();
				$meta = get_post_meta( get_the_ID() );
				$text .= '<tr>';
				$text .= '<td><a href="' . get_permalink() . '">' . $meta['_djv_linrodethCharacterSheet_name'][0] . '</a></td>';
				$text .= '<td>' . $meta['_djv_linrodethCharacterSheet_genre'][0] . '</td>';
				$text .= '<td>' . $meta['_djv_linrodethCharacterSheet_playerName'][0] . '</td>';
				$text .= '<td>' . get_the_author() . '</td>';
				$text .= '<td>' . get_the_modified_date() . ' at ' . get_the_modified_time() . '</td>';
				$text .= '</tr>';
			}
			$text .= '</table>';
		} else {
			$text = '<p style="padding:25px;">No characters found</p>';
		}
	} else {
		if ($characters->have_posts()) {
			$characters->the_post();
			$meta = get_post_meta( get_the_ID() );
			$text .= '<a href="' . get_permalink() . '">';
			$insert = '';
			if (isset($attr) && isset( $attr['text'] ) ) {
				$insert = $attr['text'];
			} 
			if ( $insert == '' && isset( $content ) ) {
				$insert = $content;
			} 
			if ($insert == '') {
				$insert = $meta['_djv_linrodethCharacterSheet_name'][0] ;
			}
			$text .= $insert . '</a> ';
		} else {
			$text .= '<span>(No characters found matching \'' . json_encode($attr) . '\' found)</span>';
		}
	}
	wp_reset_postdata();
	return $text;
}

add_shortcode( 'linchar', 'djv_linrodethCharacterSheet_linchar_shortcode' );

/**
 * Shortcode to add a picture of the Linrodeth night sky
 * 
 * Attributes:
 * <dl>
 * <dt>time</dt><dd>The time to use for the picture. Can be "now", "gamedate" or a number specifying milliseconds since epoch. The time used for "gametime" is 11pm on the game date configuration setting.</dd>
 * <dt>showbuttons</dt><dd>Whether to add buttons to control the time used for the picture. Defaults to "false".</dd>
 * </dl>
 */

// Now for a shortcode that can be used to get a view of the skies over Linrodeth
function djv_linrodethCharacterSheet_linsky_shortcode( $attr, $content = null ) {
	if ( isset($attr) && isset( $attr['time'] ) ) {
		if ( $attr['time'] == 'now' ) {
			$time = time()*1000;
			$timetext = "now";
		} else if ( $attr['time'] == 'gamedate' ) {
			$timetext = get_option('djv_linrodethCharacterSheet_options')['gametime'];
			$time = (int)$timetext;
			$time -= ($time % 86400000);
			$time += 23*60*60*1000;
			$timetext = "11pm game date";
		} else {
			$time = (int)$attr['time'];
			$timetext = (string)$time;
		}
	} else {
		$time = time()*1000;
		$timetext = "now";
	}
	
	if ( isset($attr) && isset( $attr['showbuttons'] ) && ($attr['showbuttons'] !== 'false') ) {
		$buttons = true;
	} else {
		$buttons = false;
	}
	
	$plugin_url = plugin_dir_url(__FILE__);
	wp_enqueue_script('djv_linrodethCharacterSheet_lindate', $plugin_url.'js/linrodeth/lindate.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_canvas_clear', $plugin_url.'js/linrodeth/canvas.clear.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_moons', $plugin_url.'js/linrodeth/moons.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_planets', $plugin_url.'js/linrodeth/planets.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_skymap', $plugin_url.'js/linrodeth/skymap.js' );
	wp_enqueue_script('djv_linrodethCharacterSheet_stars', $plugin_url.'js/linrodeth/stars.js' );
	
	static $id = 0; $id++;
	
	$text  = '' . PHP_EOL;
	if ( $buttons ) {
		$text .= '<br><button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()-31536000000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>-365 days</button>' . PHP_EOL; 
		$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()-2592000000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>-30 days</button>' . PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()-86400000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>-1 day</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()-3600000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>-1 hour</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()-60000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>-1 minute</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>'. $timetext . '</button>'. PHP_EOL;
	}
    $text .= '<br><canvas id="telescope' . $id . '" width=482 height=500></canvas>'. PHP_EOL;
    if ( $buttons ) {
    	$text .= '<br><button onclick=\'now' . $id . '=new Date(); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>' . $timetext . '</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()+60000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>+1 minute</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()+3600000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>+1 hour</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()+86400000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>+1 day</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()+2592000000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>+30 days</button>'. PHP_EOL;
    	$text .= '<button onclick=\'now' . $id . '=new Date(now' . $id . '.getTime()+31536000000); drawskymap(document.getElementById("telescope' . $id . '").getContext("2d"), 52/180*Math.PI, 0, now' . $id . ');\'>+365 days</button>'. PHP_EOL;
    }
    $text .= '<br>' . PHP_EOL;
    $text .= '<script>' . PHP_EOL;
    $text .= 'function onLoad(f) {' . PHP_EOL;
    $text .= '  if (onLoad.loaded) window.setTimeout(f,0);' . PHP_EOL;
    $text .= '  else if (window.addEventListener) window.addEventListener("load", f, false);' . PHP_EOL;
    $text .= '  else if (window.attachEvent) window.attachEvent("onload",f);' . PHP_EOL;
    $text .= '}' . PHP_EOL;
    $text .= 'onLoad.loaded = false;' . PHP_EOL;
    $text .= 'onLoad(function() { onLoad.loaded = true; });' . PHP_EOL;
    $text .= 'onLoad(function() {' . PHP_EOL;
    $text .= '  var canvas = document.getElementById("telescope' . $id . '");' . PHP_EOL;
    $text .= '  var context = canvas.getContext("2d");' . PHP_EOL;
    $text .= '  now' . $id . ' = new Date(' . $time . ');' . PHP_EOL;
    $text .= '  drawskymap(context, 52/180*Math.PI, 0, now' . $id . ');' . PHP_EOL;
    $text .= '});' . PHP_EOL;
    $text .= '</script>'. PHP_EOL;
    
    return $text;
}

add_shortcode( 'linsky', 'djv_linrodethCharacterSheet_linsky_shortcode' );

?>