/*!
 * © 2014, 2017 David Vines
 *
 * Creative Commons License
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License.
 */

/* globals Character: true, alert, Rules, jsPDF, Genre, options, saveAs */
Character = {};
var character;

Character.allstats = [ "str", "dex", "agi", "con", "app", "int", "wit", "will" ];
Character.longstats = [ "Strength", "Dexterity", "Agility", "Constitution",
        "Appearance", "Intellect", "Wit", "Will" ];

Character.returnFalse = function(f) {
    return function() {
        try {
            f();
        } catch (e) {
            if (e.stack) {
                alert(e + "\n\nStack Trace\n-----------------\n" + e.stack);
            } else {
                alert(e);
            }
        }
        return false;
    };
};

Character.incrementVersion = function(cid) {
    character[cid].characterVersion += 1;
    character[cid].lastEditDate = (new Date(Date.now())).toLocaleString();
    Character.populateJSON(cid);
};

Character.updateDays = function(cid) {
    character[cid].birthday = character[cid].genre.dayMonthYearAsText(character[cid].birthtime, character[cid].birthmonth);
    character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].gamemonth);
};

Character.templates = {
    interest : {
        skill : "Numeracy",
        ip : 1,
        teacher : false
    }
};

Character.setIdText = function(cid, id, text, tooltip) {
    var element = document.getElementById("c"+cid+"-"+id);
    var newText = document.createTextNode(text);
    if (tooltip) {
        element.title = tooltip;
    }
    if (element.hasChildNodes()) {
        element.replaceChild(newText, element.firstChild);
    } else {
        element.appendChild(newText);
    }
};

Character.setTextArea = function(cid, field, text) {
    var element = document.getElementById("c"+cid+"-"+field);
    element.value = text;
    element.defaultValue = text;
};

Character.setOption = function(options, choice) {
    var found = false;
    for (var j = 0; j < options.length; j++) {
        if (choice.toString() === options[j].text) {
            options.selectedIndex = j;
            found = true;
        }
    }
    if (!found) {
        var newOption = new Option(choice.toString(), choice.toString(), true,
                true);
        options.add(newOption);
    }
};

Character.setChecked = function(button, choice) {
    button.checked = choice;
    button.defaultChecked = choice;
};

Character.deleteElement = function(cid, idOrElement) {
    var element = ((typeof idOrElement === "string") ? document.getElementById("c"+cid+"-"+idOrElement) : idOrElement);
    if (element) {
        element.parentNode.removeChild(element);
    }
};

Character.durationLength = function(dur) {
    if (dur === 0) {
        return "0 months";
    } else if (dur === 1) {
        return "1 month";
    } else if (dur < 12) {
        return dur + " months";
    }
    var years = Math.floor(dur / 12);
    dur = dur % 12;

    if (years === 1) {
        if (dur === 0) {
            return "1 year";
        }
        return "1 year, " + Character.durationLength(dur);
    } else {
        if (dur === 0) {
            return years + " years";
        }
        return years + " years, " + Character.durationLength(dur);
    }
};

Character.getFluencyPoints = function(cid,p) {
    var endFluency = (character[cid].periods[p].endmonth < 179) ? character[cid].periods[p].endmonth
            : 179;
    var fluencyMonths = endFluency - character[cid].periods[p].startmonth + 1;
    if (fluencyMonths < 0) {
        return 0;
    }
    var points = (character[cid].periods[p].startmonth === 60) ? 1000 : 0;
    points += fluencyMonths * 8;
    for (var anniversary = 72; anniversary <= 180; anniversary += 12) {
        if ((character[cid].periods[p].startmonth <= anniversary) && (character[cid].periods[p].endmonth >= anniversary)) {
            points += 4;
        }
    }
    return points;
};

Character.updateFluencyPoints = function(cid,p) {
    var points = Character.getFluencyPoints(cid,p);
    if (points > 0) {
        Character.setIdText(cid,"period"+p+"-fluencypoints",points+" points");
    }
};

Character.deleteAllChildren = function(node) {
    while (node.firstChild) {
        node.removeChild(node.firstChild);
    }
};

Character.addInterestRow = function(cid, p,table,index) {
    var template = document.getElementById("c"+cid+"-period@@-interest0");
    var row = template.cloneNode(true);
    var next = index;
    Character.updateNode(row,p);
    Character.updateForm(row,next);
    row.id = "c"+cid+"-period"+p+"-interest"+next;
    table.appendChild(row);
    Character.addSkillRow(cid,p,next);
    Character.populateJSON(cid);
};

Character.addInterest = function(cid,skillname) {
    return Character.returnFalse(function() {
        // Work out which tab is active
        var tab;
        var tabbersheet = document.getElementById("c"+cid+"-sheet");
        if (tabbersheet.tabber) {
            tab = tabbersheet.tabber.getActiveTab();
        }
        if (tab) {
            var idbits = tab.id.split("-"); // Note we assume an ordering of cid, period, interest as set up by addInterestRow
            if (idbits[1].slice(0,6) === "period") {
                var p = Number(idbits[1].slice(6));
                var table = document.getElementById("c"+cid+"-period"+p+"-details").firstChild.nextSibling;
                if (character[cid].periods[p].iptotal < character[cid].interestPoints) {
                    Character.addInterestKnowingPeriod(cid,p,table,skillname);
                    Character.incrementVersion(cid);
                    Character.updateStatus(cid,p);
                }
            }
        }
    });
};

Character.updateTabs = function(cid) {
    var tabbersheet = document.getElementById("c"+cid+"-sheet");
    if (tabbersheet.tabber) {
        tabbersheet.tabber.reinit(tabbersheet);
    }
};

Character.updateArmour = function(cid,armour) {
    return Character.returnFalse(function() {
        character[cid].armour = armour;
        Character.incrementVersion(cid);
        Character.populateJSON(cid);
    });
};

Character.updateShield = function(cid,shield) {
    return Character.returnFalse(function() {
        character[cid].shield = shield;
        if (character[cid].shield.locations === 0) {
            character[cid].shieldLocations = [];
        } else if (character[cid].shield.locations === 1) {
            character[cid].shieldLocations = [ "shieldArm", "chest"];
        } else if (character[cid].shield.locations === 2) {
            character[cid].shieldLocations = [ "shieldArm", "chest","head"];
        } else if (character[cid].shield.locations === 3) {
            character[cid].shieldLocations = [ "shieldArm", "chest","head","abdomen"];
        }
        Character.populateShieldLocations(cid);
        Character.incrementVersion(cid);
        Character.populateJSON(cid);
    });
};

Character.updateShieldLocations = function(cid) {
    return Character.returnFalse(function() {
        var locs;
        locs = ["head", "chest", "abdomen"];
        character[cid].shieldLocations = ["shieldArm"];
        for(var i=0; i<locs.length; i++) {
            if (document.getElementById("c"+cid+"-coverage"+locs[i]).checked) {
                character[cid].shieldLocations.push(locs[i]);
            }
        }
        Character.incrementVersion(cid);
        Character.populateJSON(cid);
    });
};

Character.populateShieldLocations = function(cid) {
    var containsShieldLocation = function(location) {
        return function(x) { return x === location; };
    };
    document.getElementById("c"+cid+"-trcoverageinfo").firstChild.data = character[cid].shield.text;
    var head =  document.getElementById("c"+cid+"-coveragehead");
    var chest = document.getElementById("c"+cid+"-coveragechest");
    var abdomen = document.getElementById("c"+cid+"-coverageabdomen");
    if (character[cid].shield.locations === 0) {
        head.type = "checkbox";
        chest.type = "checkbox";
        abdomen.type = "checkbox";
        head.checked = false;
        chest.checked = false;
        abdomen.checked = false;
        head.disabled = true;
        chest.disabled = true;
        abdomen.disabled = true;
    } else if (character[cid].shield.locations === 1) {
        head.type = "radio";
        chest.type = "radio";
        abdomen.type = "radio";
        head.checked = character[cid].shieldLocations.some(containsShieldLocation("head"));
        chest.checked = character[cid].shieldLocations.some(containsShieldLocation("chest"));
        abdomen.checked = character[cid].shieldLocations.some(containsShieldLocation("abdomen"));
        head.disabled = false;
        chest.disabled = false;
        abdomen.disabled = false;
        head.onchange = Character.updateShieldLocations(cid);
        chest.onchange = Character.updateShieldLocations(cid);
        head.onchange = Character.updateShieldLocations(cid);
    } else if (character[cid].shield.locations === 2) {
        head.type = "radio";
        chest.type = "checkbox";
        abdomen.type = "radio";
        head.checked = character[cid].shieldLocations.some(containsShieldLocation("head"));
        chest.checked = true;
        abdomen.checked = character[cid].shieldLocations.some(containsShieldLocation("abdomen"));
        head.disabled = false;
        chest.disabled = true;
        abdomen.disabled = false;
        head.onchange = Character.updateShieldLocations(cid);
        abdomen.onchange = Character.updateShieldLocations(cid);
    } else if (character[cid].shield.locations === 3) {
        head.type = "checkbox";
        chest.type = "checkbox";
        abdomen.type = "checkbox";
        head.checked = true;
        chest.checked = true;
        abdomen.checked = true;
        head.disabled = true;
        chest.disabled = true;
        abdomen.disabled = true;
    }
};

Character.updateFavouredWeapons = function(cid) {
    return Character.returnFalse(function() {
        character[cid].favouredWeapons = [];
        for(var i=0; i<Rules.weapons.length; i++) {
            var checkbox = document.getElementById("c"+cid+"-weapon-"+Rules.weapons[i].name);
            if (checkbox && checkbox.checked) {
                character[cid].favouredWeapons.push(Rules.weapons[i]);
            }
        }
        Character.incrementVersion(cid);
        Character.populateJSON(cid);
    });
};

Character.populateEquipment = function(cid) {
    var containsWeapon = function(weapon) { return function(x) { return x.name === weapon.name; }; };
    var containsGenre = function(x) { return x === character[cid].genre.toString(); };
    var radioequip = function(radioname,array,setting,update) {
        var b = 0;
        var button,span;
        for(var i=0; i<array.length; i++) {
            if (array[i].genre.some(containsGenre)) {
                span = document.getElementById("c"+cid+"-td"+radioname+b);
                button = document.getElementById("c"+cid+"-"+radioname+b);
                span.className = "";
                button.nextSibling.data = array[i].name;
                button.checked = (array[i].name === setting.name);
                button.onchange = update(cid,array[i]);
                b++;
            }
        }
        // Clear out the other buttons, if any
        for(;;b++) {
            span = document.getElementById("c"+cid+"-td"+radioname+b);
            if (!span)  { break; }
            button = document.getElementById("c"+cid+"-"+radioname+b);
            span.className = "hidden";
            if (button.checked) {
                button.checked = false;
                document.getElementById("c"+cid+"-"+radioname+"0").checked = true;
            }
        }

    };
    
    radioequip("armour",Rules.armour,character[cid].armour,Character.updateArmour);
    radioequip("shield",Rules.shields,character[cid].shield,Character.updateShield);
    
    // shield locations
    Character.populateShieldLocations(cid);
    
    // Weapons are a set of checkboxs, (6 columns)
    var columns = 6;
    var weapons = document.getElementById("c"+cid+"-weapons");
    var newweapons = document.createDocumentFragment();
    var newrow = document.createElement("tr");
    newrow.id = "c"+cid+"-weapons";
    var column = 0;
    for(var i=0; i<Rules.weapons.length; i++) {
        if (Rules.weapons[i].genre.some(containsGenre)) {
            var newcheckbox = document.createElement("input");
            newcheckbox.type = "checkbox";
            newcheckbox.id = "c"+cid+"-weapon-"+Rules.weapons[i].name;
            newcheckbox.checked = character[cid].favouredWeapons.some(containsWeapon(Rules.weapons[i]));
            newcheckbox.onchange = Character.updateFavouredWeapons(cid);
            var newcell = document.createElement("td");
            newcell.appendChild(newcheckbox);
            newcell.appendChild(document.createTextNode(Rules.weapons[i].name));
            newrow.appendChild(newcell);
            column++;
            if (column === columns) {
                newweapons.appendChild(newrow);
                newrow = document.createElement("tr");
                column = 0;
            }
        }
    }
    if (column > 0) {
        newweapons.appendChild(newrow);
    }
    while(weapons.nextSibling) {
        weapons.parentNode.removeChild(weapons.nextSibling);
    }
    weapons.parentNode.replaceChild(newweapons,weapons);
};

Character.populatePeriodMonths = function(cid,p) {
    Character.setIdText(cid,"period"+p+"-enddatelength"," A period of "+Character.durationLength(character[cid].periods[p].endmonth - character[cid].periods[p].startmonth  + 1));
    if (character[cid].periods[p].startmonth === character[cid].periods[p].endmonth) {
        Character.setIdText(cid,"period"+p+"-daterange","The month of "+character[cid].periods[p].startmonthtext);
    } else if (character[cid].periods[p].startmonth < character[cid].periods[p].endmonth) {
        Character.setIdText(cid,"period"+p+"-daterange",character[cid].periods[p].startmonthtext+" to "+character[cid].periods[p].endmonthtext);
    } else {
        Character.setIdText(cid,"period"+p+"-daterange","Future interests");
    }
};

Character.populateSkills = function(cid) {
    var i,j;
    var endingMonthText = character[cid].genre.dateMonthAsText(character[cid].periods[character[cid].periods.length-1].endmonth+character[cid].birthmonth);
    
    var points = {};
    var addPoints = function(points,name,dp) {
        if (points[name]) {
            points[name] += dp;
        } else {
            points[name] = dp;
        }
    };
    for(i=0; i<character[cid].periods.length; i++) {
        // Childhood fluency
        if (Character.getFluencyPoints(cid,i) > 0) {
            addPoints(points,"Fluency("+character[cid].periods[i].childhoodLanguage+")",Character.getFluencyPoints(cid,i));
        }
        var duration = character[cid].periods[i].endmonth - character[cid].periods[i].startmonth + 1;
        for(j=0; j<character[cid].periods[i].interests.length; j++) {
            var dp = character[cid].periods[i].interests[j].ip * duration;
            if (character[cid].periods[i].interests[j].teacher) {
                dp += (Math.round(character[cid].periods[i].interests[j].ip / 3 - 0.3)) * duration;
            }
            addPoints(points,character[cid].periods[i].interests[j].skill,dp);
        }
    }
    // convert to array and sort
    var arr = [];
    for(var s in points) {
        if (points.hasOwnProperty(s)) {
            arr.push({skill:s, dp: points[s]});
        }
    }
    // Compute the skill levels
    for(i=0; i<arr.length; i++) {
        arr[i].level = Rules.skilllevel(arr[i].skill,character[cid],arr[i].dp);
    }
    
    
    arr.sort(function(x,y) {
        return y.level - x.level;
    });
    var DOM_table = document.createElement("table");
    var DOM_tbody = document.createElement("tbody");
    var DOM_tr = document.createElement("tr");
    var DOM_th = document.createElement("th");
    DOM_th.colspan = 3;
    var DOM_p = document.createElement("p");
    DOM_p.appendChild(document.createTextNode("As of the month of "+endingMonthText));
    DOM_th.appendChild(DOM_p);
    DOM_tr.appendChild(DOM_th);
    DOM_tbody.appendChild(DOM_tr);
    
    DOM_tr = document.createElement("tr");
    DOM_th = document.createElement("th");
    DOM_th.appendChild(document.createTextNode("Skill Name"));
    DOM_tr.appendChild(DOM_th);
    DOM_th = document.createElement("th");
    DOM_th.appendChild(document.createTextNode("Development Points"));
    DOM_tr.appendChild(DOM_th);
    DOM_th = document.createElement("th");
    DOM_th.appendChild(document.createTextNode("Level"));
    DOM_tr.appendChild(DOM_th);
    DOM_tbody.appendChild(DOM_tr);
    
    for(i=0; i<arr.length; i++) {
        DOM_tr = document.createElement("tr");
        var DOM_td = document.createElement("td");
        DOM_td.appendChild(document.createTextNode(arr[i].skill));
        DOM_tr.appendChild(DOM_td);
        DOM_td = document.createElement("td");
        DOM_td.style = "text-align: right";
        DOM_td.appendChild(document.createTextNode(arr[i].dp));
        DOM_tr.appendChild(DOM_td);
        DOM_td = document.createElement("td");
        DOM_td.style = "text-align: right";
        DOM_td.appendChild(document.createTextNode(arr[i].level));
        DOM_tr.appendChild(DOM_td);
        DOM_tbody.appendChild(DOM_tr);
    }
    DOM_table.appendChild(DOM_tbody);
    
    var skills = document.getElementById("c"+cid+"-skillsheet");
    Character.deleteAllChildren(skills);
    skills.appendChild(DOM_table);
    
    // And now the JSON needs updating
    character[cid].sheet = { endmonth: character[cid].periods[character[cid].periods.length-1].endmonth,
                        endmonthtext: endingMonthText,
                        skills: arr
                      };
    Character.populateJSON(cid);
};

Character.validateStat = function(name,stat) {
    if (stat === undefined) { throw name+" is blank, character has not been updated."; }
    var v = Number(String(stat).trim());
    if (isNaN(v)) { throw name+" is not a number (it's '"+stat+"'). character has not been updated."; }
    if (v < 0)    { throw name+" is less than zero, character has not been updated."; }
    if (v > 20)   { throw name+" is more than twenty, character has not been updated."; }
    return v;
};

Character.createButton = function(name,onclickfunction) {
    var DOM_button = document.createElement("button");
    var text = name.replace("/","/\u200b").replace("(","\u200b(");
    DOM_button.appendChild(document.createTextNode(text));
    DOM_button.value = name;
    DOM_button.style.width = "100%";
    DOM_button.onclick = onclickfunction;
    return DOM_button;
};

Character.createTextArea = function(cid, id, rows, cols) {
    var DOM_textarea = document.createElement("textarea");
    DOM_textarea.id= "c"+cid+"-"+id;
    DOM_textarea.rows = rows;
    DOM_textarea.cols = cols;
    if (rows === 1) {
        DOM_textarea.className = "onerow";
    }
    DOM_textarea.appendChild(document.createTextNode(""));
    return DOM_textarea;
};

Character.updateStatus = function(cid,p) {
    var status = document.getElementById("c"+cid+"-period" + p + "-status");
    var text;
    var valid = false;
    var DOM_b;
    Character.deleteAllChildren(status);
    if (character[cid].periods[p].endmonth === character[cid].periods[p].startmonth-1) {
        status.setAttribute("class", "warning");
        text = "This period is of zero length and will not add any development points to this character's skills.";
        status.appendChild(document.createTextNode(text));
    } else if (character[cid].periods[p].endmonth < character[cid].periods[p].startmonth) {
        status.setAttribute("class", "error");
        text = "This period is of NEGATIVE length and will REMOVE development points from this character's skills.";
        DOM_b = document.createElement("b");
        DOM_b.appendChild(document.createTextNode(text));
        status.appendChild(DOM_b);
    } else if (p === character[cid].periods[p].length-1 && character[cid].periods[p].endMonth !== character[cid].gamemonth-character[cid].birthmonth) {
        status.setAttribute("class", "warning");
        text = "The end month of this last period (" + character[cid].periods[p].endMonthText + ") does not match the game month (" + character[cid].gamemonthtext + "). When the game month is updated, this will reset this end month to the game month.";
        status.appendChild(document.createTextNode(text));
    } else if (character[cid].periods[p].iptotal === 0) {
        status.setAttribute("class", "warning");
        text = "None of the " + character[cid].interestPoints + " interest points available to this character have been used.";
        status.appendChild(document.createTextNode(text));
    } else if (character[cid].periods[p].iptotal < character[cid].interestPoints) {
        status.setAttribute("class", "warning");
        text = "Only " + character[cid].periods[p].iptotal + " of the " + character[cid].interestPoints + " interest points available to this character have been used.";
        status.appendChild(document.createTextNode(text));
    } else if (character[cid].periods[p].iptotal === character[cid].interestPoints) {
        status.setAttribute("class", "");
        text = "All the " + character[cid].interestPoints + " interest points available to this character have been used.";
        valid = true;
        status.appendChild(document.createTextNode(text));
    } else {
        status.setAttribute("class", "error");
        text = "The " + character[cid].periods[p].iptotal + " interest points used by this character are more than the " + character[cid].interestPoints + " available to this character[cid].";
        DOM_b = document.createElement("b");
        DOM_b.appendChild(document.createTextNode(text));
        status.appendChild(DOM_b);
    }
    character[cid].periods[p].statustext = text;
    character[cid].periods[p].valid = valid;
};

Character.populateAll = function(cid) {
    Character.populateStats(cid);
    Character.populateEquipment(cid);
    Character.populatePeriods(cid);
    Character.populateSkills(cid);
};

Character.makePDF = function(cid) {
    var drawSkillTable = function(pdf, title, skills, top, left, cellheight, cellwidth) {
        if (skills.length > 45) {
            cellheight--; // Cope with Goodgulf :)
        }
        if (skills.length > 50) {
            cellheight -= 0.5; // And again
        }

        var skillnamewidth = 81;
        var skilllevelwidth = 60;
        var levelsplit = 112;
        var vgutter = ((skills.length > 50) ? 2 : 3);
        var hgutter = 2;

        var mainleft = left + skillnamewidth + skilllevelwidth;
        var right = mainleft + Rules.adjectives.length * cellwidth;
        var bottom = top + (skills.length + 2) * cellheight;

        pdf.setLineWidth(1);
        pdf.line(mainleft, top, right, top);
        pdf.line(mainleft, top + cellheight, right, top + cellheight);
        pdf.setLineWidth(0.5);
        for (var x = 0; x < Rules.adjectives.length; x++) {
            pdf.line(mainleft + x * cellwidth, top, mainleft + x * cellwidth,
                    bottom);
            pdf.setFontSize(Rules.adjectives[x].length < 12 ? 6 : 4.5); // Special case for "Embarrasing"
            pdf.textInBox(Rules.adjectives[x], {
                align : "center"
            }, mainleft + x * cellwidth + hgutter, top + cellheight - vgutter,
                    mainleft + (x + 1) * cellwidth, top);
        }

        pdf.setLineWidth(1);
        pdf.line(left, top + cellheight, right, top + cellheight);
        pdf.line(left, top + 2 * cellheight, right, top + 2 * cellheight);
        pdf.line(left, bottom, right, bottom);
        pdf.line(left, top + cellheight, left, bottom);
        pdf.line(left + skillnamewidth, top + cellheight,
                left + skillnamewidth, bottom);
        pdf.line(mainleft, top, mainleft, bottom);
        pdf.line(right, top, right, bottom);
        pdf.setFontStyle("bold");
        pdf.setFontSize(7.5);
        pdf.text(left + hgutter, top + cellheight * 2 - 3, title);
        pdf.text(left + skillnamewidth + hgutter, top + cellheight * 2 - 3, "Level");
        for (x = 0; x < Rules.adjectives.length; x++) {
            pdf.textInBox(x.toString(), {
                align : "center"
            }, mainleft + x * cellwidth + hgutter, top + 2 * cellheight - vgutter,
                    mainleft + (x + 1) * cellwidth, top + cellheight);
        }

        pdf.setLineWidth(0.5);
        pdf.setFontStyle("normal");
        var y = top + 3 * cellheight;
        for (var s = 0; s < skills.length; s++, y += cellheight) {
            if (s !== skills.length - 1) {
                pdf.line(left, y, right, y);
            }
            pdf.text(left + hgutter, y - 3, skills[s].skill);
            pdf.textInBox(skills[s].level.toString(), {
                align : "right"
            }, left + skillnamewidth, y - vgutter, levelsplit - hgutter, y - cellheight + 1);
            pdf.textInBox(Rules.adj(skills[s].level), {
                align : "left"
            }, levelsplit + hgutter, y - vgutter, mainleft, y - cellheight + 1);
            var perf = Rules.performance(skills[s].level);
            for (x = 0; x < Rules.adjectives.length; x++) {
                pdf.textInBox(perf[x], {
                    align : "center"
                }, mainleft + x * cellwidth, y - vgutter, mainleft + (x + 1) * cellwidth, y - cellheight + 1);
            }
        }
    };

    var drawTable = function(pdf, content, widths, aligns, left, top, cellheight, borderweights) {
        var totalwidth = widths.reduce(function(x, y) {
            return x + y;
        }, 0);
        var hgutter = 2;
        pdf.setLineWidth(1);
        pdf.rect(left, top, totalwidth, content.length * cellheight);
        var offset = left + widths[0];
        var cell;
        for (cell = 1; cell < widths.length; cell++) {
            if (borderweights && borderweights[cell]) {
                pdf.setLineWidth(borderweights[cell]);
            }
            pdf.line(offset, top, offset, top + content.length * cellheight);
            offset += widths[cell];
        }
        pdf.setFontStyle("bold"); // Make top line bold
        pdf.setLineWidth(1);
        var y = top + cellheight;
        for (var line = 0; line < content.length; line++, y += cellheight) {
            if (line !== content.length - 1) {
                pdf.line(left, y, left + totalwidth, y);
            }
            offset = left;
            for (cell = 0; cell < aligns.length; cell++) {
                pdf.textInBox(content[line][cell], {
                    align : aligns[cell]
                }, offset + hgutter, y - 3, offset + widths[cell], y - cellheight);
                offset += widths[cell];
            }
            if (line === 0) {
                pdf.setFontStyle("normal");
                pdf.setLineWidth(0.5);
            }
        }
    };

    var pdf = new jsPDF("landscape", "pt", "a4");
    pdf.setProperties({
        title : character[cid].name,
        keywords : "generated, linrodeth, character"
    });

    var namebottom = 20;
    var nameleft = 0;
    var nameright = 842;
    var nametop = 2;
    var left = 12;
    var midright = 400;
    var midleft = 442;
    var right = 842;
    var cellwidth = 32;
    var cellheight = 10;
    var chartext = "Player: " + character[cid].playerName + ", Rules Version " + character[cid].rulesVersion + ", Implementation Version " + Rules.version + ", Character version " + character[cid].characterVersion + ", Game Date:" + character[cid].gameday + ", (last edit time " + character[cid].lastEditDate + ", PDF created " + (new Date()).toLocaleString() + ")";
    var widths;
    var aligns;
    var title;
    var content;

    // Page 1 - back cover skills, levels and current interest
    var skillsummmarycellheight = 12;
    if (character[cid].sheet.skills.length > 32) { skillsummmarycellheight = 11;  }
    if (character[cid].sheet.skills.length > 40) { skillsummmarycellheight = 10;  }
    if (character[cid].sheet.skills.length > 45) { skillsummmarycellheight = 9;   }
    if (character[cid].sheet.skills.length > 50) { skillsummmarycellheight = 8.5; }

    pdf.setFontSize(18);
    pdf.textInBox(character[cid].name, {
        align : "center"
    }, left, namebottom, midright, nametop);
    pdf.setFontSize(skillsummmarycellheight-1);
    content = [];
    widths = [198,100,90];
    aligns = ["left","left", "center"];
    title = ["Skill", "Level", "Current Interest"];
    content.push(title);
    for (var s = 0; s < character[cid].sheet.skills.length; s++) {
        var currentip = "";
        for(var cs = 0; cs < character[cid].periods[character[cid].periods.length-1].interests.length; cs++) {
            if (character[cid].periods[character[cid].periods.length-1].interests[cs].skill === character[cid].sheet.skills[s].skill) {
                currentip = String(character[cid].periods[character[cid].periods.length-1].interests[cs].ip);
            }
        }
        var level = " " + character[cid].sheet.skills[s].level + " (" + Rules.adj(character[cid].sheet.skills[s].level)+")";
        content.push([character[cid].sheet.skills[s].skill, level, currentip]);
    }
    drawTable(pdf,content,widths,aligns,left,27,skillsummmarycellheight);
    
    // front cover name, dates, stats, body, summary
    pdf.setFontSize(18);
    pdf.textInBox(character[cid].name, {
        align : "center"
    }, midleft, namebottom, right, nametop);
    
    pdf.setFontSize(12);
    content = [];
    widths = [120,100];
    aligns = ["left","left"];
    title = ["Attribute","Level"];
    content.push(title);
    content.push(["Strength"," "+character[cid].str+" ("+Rules.adj(character[cid].str)+")"]);
    content.push(["Dexterity"," "+character[cid].dex+" ("+Rules.adj(character[cid].dex)+")"]);
    content.push(["Agility"," "+character[cid].agi+" ("+Rules.adj(character[cid].agi)+")"]);
    content.push(["Constitution"," "+character[cid].con+" ("+Rules.adj(character[cid].con)+")"]);
    content.push(["Appearance"," "+character[cid].app+" ("+Rules.adj(character[cid].app)+")"]);
    content.push(["Intellect"," "+character[cid].int+" ("+Rules.adj(character[cid].int)+")"]);
    content.push(["Wit"," "+character[cid].wit+" ("+Rules.adj(character[cid].wit)+")"]);
    content.push(["Will"," "+character[cid].will+" ("+Rules.adj(character[cid].will)+")"]);
    content.push(["",""]);
    content.push(["Interest Points"," "+character[cid].interestPoints]);
    content.push(["Morale"," "+character[cid].morale]);
    drawTable(pdf,content,widths,aligns,midleft,namebottom+8,14);
    
    var midmidleft = (midleft + right)/2+50;
    var bodycentre = midmidleft + (right-midmidleft-20)/2;
    var bodytop = namebottom+8;
    pdf.setLineWidth(1);
    pdf.rect(midmidleft,bodytop,right-midmidleft-20,16*14);
    pdf.setLineWidth(0.5);
    pdf.lines([[32,0],[0,36],[42,0],[0,90],[-20,0],[0,-72],[-7,0],[0,144],[-24,0],[0,-72],[-14,0],[0,72],[-24,0],[0,-144],[-7,0],[0,72],[-20,0],[0,-90],[42,0],[0,-36]],bodycentre-16,bodytop+9);
    var locpic = function(name, location) {
        var containsShieldLocation = function(location) {
            return function(x) {
                return x === location;
            };
        };

        var armour = character[cid].armour.name;
        var points = character[cid].armour.value;
        if (character[cid].shieldLocations.some(containsShieldLocation(location))) {
            armour += " + " + character[cid].shield.name;
            points += character[cid].shield.value;
        }
        return character[cid].woundPoints[location]+"/"+points.toString();
    };
    pdf.textInBox(locpic("Head", "head"),{align : "center", valign: "center"},bodycentre-16,bodytop+45,bodycentre+16,bodytop+9);
    pdf.textInBox(locpic("Shield Arm", "shieldArm"),{align : "center", valign: "center"},bodycentre+39,bodytop+135,bodycentre+57,bodytop+45);
    pdf.textInBox(locpic("Sword Arm", "swordArm"),{align : "center", valign: "center"},bodycentre-59,bodytop+135,bodycentre-39,bodytop+45);
    pdf.textInBox(locpic("Chest", "chest"),{align : "center", valign: "center"},bodycentre-32,bodytop+111,bodycentre+32,bodytop+63);
    pdf.textInBox(locpic("Abdomen", "abdomen"),{align : "center", valign: "center"},bodycentre-32,bodytop+135,bodycentre+32,bodytop+111);
    pdf.textInBox(locpic("Right Leg", "rightLeg"),{align : "center", valign: "center"},bodycentre-32,bodytop+207,bodycentre-8,bodytop+135);
    pdf.textInBox(locpic("Left Leg", "leftLeg"),{align : "center", valign: "center"},bodycentre+6,bodytop+207,bodycentre+32,bodytop+135);

    pdf.text("Birthday: "+character[cid].birthday,midleft,namebottom+260);
    pdf.text("Game Date: "+character[cid].gameday,midleft,namebottom+276);
    pdf.setFontSize(8);
    pdf.textInBox("wound points/armour",{align:"center"}, midmidleft,bodytop+16*14-1.5,right-20,bodytop+16*14-9 );
    
    pdf.setFontSize(12);
    pdf.setFontStyle("bold");
    pdf.text("Description",midleft,namebottom+300);
    pdf.setFontStyle("normal");
    pdf.text(pdf.splitTextToSize(character[cid].description,right-midleft),midleft,namebottom+314);
    
    pdf.setFontSize(8);
    pdf.text(left, 576, chartext);
    
    // Page 2 - inside pages - life biography
    pdf.addPage();
    pdf.setFontSize(8);
    pdf.text(left, 576, chartext);
    
    var proposedptsize = 12;
    var retry, render = false, done = false;
    var pageoffset, lineoffset;
    
    while (!done) {
        pageoffset = left;
        lineoffset = 28;
        if (render) { pdf.setFontSize(proposedptsize); }
        retry = false;
        for(var p=0; (p<character[cid].periods.length); p++ ) {
            var desclines = pdf.splitTextToSize(character[cid].periods[p].description,midright-left);
            if (lineoffset + proposedptsize*1.15*(desclines.length+2) > 560) {
                // switch page
                pageoffset += (midleft-left);
                lineoffset = 28;
                if (pageoffset > right) {
                    proposedptsize--;
                    retry = true;
                    break;
                }
            }
            if (render) {
                pdf.setFontStyle("bold");
                pdf.text(character[cid].periods[p].title+" ("+character[cid].periods[p].startmonthtext+"-"+character[cid].periods[p].endmonthtext+")",pageoffset,lineoffset);
                pdf.setFontStyle("normal");
                pdf.text(desclines,pageoffset,lineoffset+proposedptsize*1.15);
            }
            lineoffset += proposedptsize*1.15*(desclines.length+2);
        }
        if (render) { done = true; }
        if (!retry) { render = true; }
    }
    

    // Page 3 - Misc tables
    pdf.addPage(); 
    pdf.setFontSize(18);
    pdf.textInBox(character[cid].name, {
        align : "center"
    }, nameleft, namebottom, nameright, nametop);

    // Need to Fake up a skills array for the attributes
    var attributesAsSkills = [];
    for (var stat = 0; stat < Character.allstats.length; stat++) {
        attributesAsSkills.push({
            "skill" : Character.longstats[stat],
            "level" : character[cid][Character.allstats[stat]]
        });
    }
    drawSkillTable(pdf, "Attribute", attributesAsSkills, 36, left, cellheight, cellwidth);

    var i,w,x,weapon;
    content = [];
    widths = [ 81, 30, 30 ];
    aligns = [ "left", "center", "center" ];
    title = [ "Weapon", "Area", "Hits" ];
    for (x = 0; x < Rules.adjectives.length; x++) {
        title.push(x.toString());
        widths.push(cellwidth);
        aligns.push("center");
    }
    content.push(title);
    var hasarea = false;
    var hashits = false;
    for (w = 0; w < character[cid].favouredWeapons.length; w++) {
        weapon = character[cid].favouredWeapons[w];
        if (weapon.ranged) {
            var area = weapon.area || "";
            var hits = weapon.hits || "";
            hasarea = hasarea || weapon.area || false;
            hashits = hashits || weapon.hits || false;
            var ranges = [ weapon.name, area, hits ];
            ranges = ranges.concat(weapon.ranges);
            for (i = 0; i < ranges.length; i++) {
                ranges[i] = (ranges[i] ? ranges[i].toString() : "");
            }
            content.push(ranges);
        }
    }
    pdf.text(left + 3, 151, "Difficulty to hit at range");
    drawTable(pdf, content, widths, aligns, left, 154, cellheight, [undefined, 1, 0.5, 1, 0.5 ]);

    content = [];
    widths = [ 81, 60 ];
    aligns = [ "left", "center" ];
    title = [ "Weapon", "Damage" ];
    for (x = 0; x < Rules.adjectives.length; x++) {
        title.push(x.toString());
        widths.push(cellwidth);
        aligns.push("center");
    }
    content.push(title);
    var line;
    line = [ "Unarmed", "1" ];
    line = line.concat(Rules.damageTable[1]);
    content.push(line);
    for (w = 0; w < character[cid].favouredWeapons.length; w++) {
        weapon = character[cid].favouredWeapons[w];
        line = [ weapon.name, weapon.damage.toString() ];
        line = line.concat(Rules.damageTable[weapon.damage]);
        content.push(line);
    }
    drawTable(pdf, content, widths, aligns, left, 210, cellheight, [undefined, 1, 1, 0.5 ]);

    content = [];
    content.push([ "Situation", "Mod", "Example for cooking" ]);
    content.push([ "Excellent", "+5", "Staffed restaurant kitchen" ]);
    content.push([ "Very good", "+3", "Skilled help" ]);
    content.push([ "Good", "+1", "Normal kitchen, some help" ]);
    content.push([ "Normal", "", "Normal kitchen" ]);
    content.push([ "Poor", "-1", "No spices or herbs available" ]);
    content.push([ "Very Poor", "-3", "Open fire, few ingredients" ]);
    content.push([ "Terrible", "-5", "No fire" ]);
    drawTable(pdf, content, [ 41, 120 - 86, 225 - 120 ], [ "left", "center",
            "left" ], 45, 324, cellheight);

    var modwidths = [ 90, 36 ];

    content = [];
    content.push([ "Melee situation", "Modifier" ]);
    content.push([ "Defending only", "+3" ]);
    content.push([ "At Opponents side", "+1" ]);
    content.push([ "Behind Opponent", "+2" ]);
    content.push([ "Unarmed versus blunt", "-1" ]);
    content.push([ "Unarmed versus blade", "-4" ]);
    content.push([ "Shorter weapon", "-1" ]);
    content.push([ "Each extra opponent", "-1"]);
    drawTable(pdf, content, modwidths, [ "left", "center" ], 252, 324, cellheight);

    content = [];
    content.push([ "Location Hit", "Melee", "Ranged" ]);
    content.push([ "Head", "91-00", "96-00" ]);
    content.push([ "Left Arm", "81-90", "86-95" ]);
    content.push([ "Right Arm", "71-80", "76-85" ]);
    content.push([ "Chest", "45-70", "51-75" ]);
    content.push([ "Abdomen", "33-44", "33-50" ]);
    content.push([ "Left Leg", "17-32", "17-32" ]);
    content.push([ "Right Leg", "01-16", "01-16" ]);
    drawTable(pdf, content, [ 64, 32, 32 ], [ "left", "center", "center" ],
            405, 324, cellheight);

    content = [];
    content.push([ "Ranged situation", "Modifier" ]);
    content.push([ "Aimed Shot (2 rounds)", "+2" ]);
    content.push([ "Walking target", "-2" ]);
    content.push([ "Jogging target", "-3" ]);
    content.push([ "Running target", "-4" ]);
    content.push([ "Dodging target", "-5" ]);
    content.push([ "Light cover", "-1" ]);
    content.push([ "Medium cover", "-3" ]);
    content.push([ "Heavy cover", "-5" ]);
    drawTable(pdf, content, modwidths, [ "left", "center" ], 252, 410,
            cellheight);

    content = [];
    content.push([ "Modifiers for previous round", "Modifier" ]);
    content.push([ "Minor defeat last round", "-1" ]);
    content.push([ "Clear defeat last round", "-2" ]);
    content.push([ "Major defeat last round", "-3" ]);
    content.push([ "Devastating defeat last round", "-4" ]);
    drawTable(pdf, content, [ 148, 32 ], [ "left", "center" ], 45, 414,
            cellheight);

    content = [];
    content.push([ "Gap", "Type of win", "Wound Location" ]);
    content.push([ "1-3", "Minor", "None" ]);
    content.push([ "4-6", "Clear", "Random" ]);
    content.push([ "7-9", "Major", "Move 1 place, +2 damage" ]);
    content.push([ "10+", "Devastating", "Move 2 places, +5 damage" ]);
    drawTable(pdf, content, [ 32, 120 - left - 32, 225 - 120 ], [ "left",
            "left", "left" ], left, 477 + cellheight, cellheight);
    pdf.setLineWidth(1);
    pdf.setFontStyle("bold");
    pdf.line(left, 477 + cellheight, left, 477);
    pdf.line(left, 477, 225, 477);
    pdf.line(225, 477, 225, 477 + cellheight);
    pdf.text(left, 477 + cellheight - 3, "Performance Difference");

    pdf.setLineWidth(1);
    pdf.rect(569, 324, 8 * cellwidth, 8 * cellheight);
    pdf.line(569, 324 + cellheight, 569 + 2 * cellwidth, 324 + cellheight);
    pdf.line(569 + 2 * cellwidth, 324, 569 + 2 * cellwidth, 324 + cellheight);
    pdf.setLineWidth(0.5);
    for (i = 1; i < 8; i++) {
        pdf.line(569 + i * cellwidth, 324, 569 + i * cellwidth,
                324 + 8 * cellheight);
        pdf.line(569, 324 + i * cellheight, 569 + 8 * cellwidth, 324 + i * cellheight);
    }
    pdf.text(570, 324 + cellheight - 3, "Morale");
    pdf
            .text(570 + cellwidth, 324 + cellheight - 3, character[cid].morale
                    .toString());

    content = [];
    content.push([ "Area", "Armour", "WP", "Armour" ]);
    var locline = function(name, location) {
        var containsShieldLocation = function(location) {
            return function(x) {
                return x === location;
            };
        };

        var armour = character[cid].armour.name;
        var points = character[cid].armour.value;
        if (character[cid].shieldLocations.some(containsShieldLocation(location))) {
            armour += " + " + character[cid].shield.name;
            points += character[cid].shield.value;
        }
        return [ name, armour, character[cid].woundPoints[location].toString(),
                points.toString() ];
    };
    content.push(locline("Head", "head"));
    content.push(locline("Shield Arm", "shieldArm"));
    content.push(locline("Sword Arm", "swordArm"));
    content.push(locline("Chest", "chest"));
    content.push(locline("Abdomen", "abdomen"));
    content.push(locline("Right Leg", "rightLeg"));
    content.push(locline("Left Leg", "leftLeg"));
    drawTable(pdf, content, [ 64, 100, 32, 32 ], [ "left", "left", "center",
            "center" ], 405, 410, cellheight, [ 1, 1, 1, 0.5, 1, 0.5 ]);
    pdf.setLineWidth(1);
    pdf.rect(633, 420, 6 * cellwidth, 7 * cellheight);
    pdf.setLineWidth(0.5);
    for (i = 1; i < 7; i++) {
        pdf.line(633, 420 + i * cellheight, 633 + 6 * cellwidth, 420 + i * cellheight);
    }
    for (i = 1; i < 6; i++) {
        pdf.line(633 + i * cellwidth, 420, 633 + i * cellwidth, 420 + 7 * cellheight);
    }

    content = [];
    content.push([ "Type of encumbrance", "Modifier" ]);
    content.push([ "Light encumbrance", "" ]);
    content.push([ "Medium encumbrance", "-1" ]);
    content.push([ "Heavy encumbrance", "-3" ]);
    drawTable(pdf, content, modwidths, [ "left", "center" ], 405, 497,
            cellheight);

    content = [];
    content.push([ "Movement", "Speed", "Modifier" ]);
    content.push([ "Walk", ""+Math.floor(5+character[cid].agi/2), "0" ]);
    content.push([ "Jog", ""+Math.floor(10+character[cid].agi), "-2" ]);
    content.push([ "Sprint", ""+Math.floor(10+3*character[cid].agi), "-4" ]);
    drawTable(pdf, content, [ 64, 32, 64 ], [ "left", "center", "center" ], 570, 497, 
            cellheight);

    if (hasarea) {
        pdf.text(738, 502, "Area weapons can hit");
        pdf.text(738, 510, "additional targets in the");
        pdf.text(738, 518, "area with a -2 modifier. Each");
        pdf.text(738, 526, "target can be hit multiple");
        pdf.text(738, 534, "times up to the hits number");
        pdf.text(738, 542, "with one extra hit for each");
        pdf.text(738, 550, "level of victory above minor.");
    } else if (hashits) {
        pdf.text(738, 502, "Each target can be hit multiple");
        pdf.text(738, 510, "times up to the hits number");
        pdf.text(738, 518, "with one extra hit for every");
        pdf.text(738, 525, "level of victory above minor.");
    }
    
    pdf.setFontSize(8);
    pdf.text(left, 576, chartext);

    // Page 4 - skill table
    pdf.addPage();
    pdf.setFontSize(18);
    pdf.textInBox(character[cid].name, {
        align : "center"
    }, nameleft, namebottom, nameright, nametop);

    var untrained = [];
    untrained.push({
        "skill" : "All untrained skills",
        "level" : 0
    });
    drawSkillTable(pdf, "Skill", untrained, 27, left, cellheight, cellwidth);
    drawSkillTable(pdf, "Skill", character[cid].sheet.skills, 72, left, cellheight, cellwidth);

    pdf.setFontSize(8);
    pdf.text(left, 576, chartext);

    return pdf;
};

Character.initPage = function(cid) {
    character = character || [];
    // Initialise the character, either with data from an embedded JSON or a
    // completely new one
    if (document.getElementById("c"+cid+"-character-data") && document.getElementById("c"+cid+"-character-data").text && document.getElementById("c"+cid+"-character-data").text.length > 10) {
        Character.updateWithNewJSON(cid, document.getElementById("c"+cid+"-character-data").text, false);
    } else {
        character[cid] = Rules.newCharacter();
        character[cid].schemaVersion = 11;
        character[cid].characterVersion = 0;
        character[cid].lastEditDate = (new Date(Date.now())).toLocaleString();
        character[cid].rulesVersion = "2014a";
        character[cid].genre = new Genre(options.default_genre);
        character[cid].name = "Nobody Noname";
        character[cid].playerName = document.getElementById("c"+cid+"-playernametextarea").textContent;
        character[cid].description = "An outstanding character whose deeds are described herein and whose heroic future is still to come.";
        character[cid].statslock = false;
        character[cid].morale = Rules.startingMorale(character[cid]);
        character[cid].woundPoints = Rules.woundPoints(character[cid]);
        character[cid].interestPoints = Rules.interestPoints(character[cid]);
        character[cid].birthtime = Date.now() - 32 * 365.25 * 24 * 60 * 60 * 1000;
        character[cid].birthmonth = character[cid].genre.asMonth(character[cid].birthtime);
        character[cid].birthmonthtext = character[cid].genre.dateMonthAsText(character[cid].birthmonth);
        character[cid].birthday = character[cid].genre.dayMonthYearAsText(character[cid].birthtime, character[cid].birthmonth);
        character[cid].gametime = options.gametime[character[cid].genre.name];
        character[cid].gamemonth = character[cid].genre.asMonth(character[cid].gametime);
        character[cid].gamemonthtext = character[cid].genre.dateMonthAsText(character[cid].gamemonth);
        character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime, character[cid].gamemonth);
        character[cid].periods = [ {
            title : "Childhood",
            description : "The early years discovering the world and your place in it",
            startmonth : 60,
            startmonthtext : character[cid].genre.dateMonthAsText(character[cid].birthmonth + 60),
            endmonth : character[cid].gamemonth - character[cid].birthmonth,
            endmonthtext : character[cid].gamemonthtext,
            childhoodLanguage : character[cid].genre.getDefaultLanguage(),
            interests : [],
            iptotal : 0,
            statustext : "None of the " + character[cid].interestPoints + " interest points available to this character have been used.",
            valid : false
        } ];
        character[cid].newSkills = [];
        character[cid].armour = Rules.armour[0];
        character[cid].shield = Rules.shields[0];
        character[cid].shieldLocations = [];
        character[cid].favouredWeapons = [];
        character[cid].syncWithGlobalDate = true;
    }
    Character.edittableStats = false; // Note the big C on the front of this,
                                        // this property is not part of the
                                        // character.

    // Set up all the non-Period buttons
    Character.setupButtons(cid);

    document.getElementById("c"+cid+"-JSONsavefile").onclick = Character
            .returnFalse(function() {
                Character.incrementVersion(cid);
                var blob = new Blob(
                        [ JSON.stringify(character[cid], undefined, 2) ], {
                            type : "text/plain;charset=utf-8"
                        });
                saveAs(blob, character[cid].name + ".txt");
            });

    document.getElementById("c"+cid+"-savePDF").onclick = Character
            .returnFalse(function() {
                var blob = Character.makePDF(cid).output('blob');
                saveAs(blob, character[cid].name + ".pdf");
            });
};

Character.updateWithNewJSON = function(cid,json,populate) {
    try {
        var i;
        var newCharacter = JSON.parse(json);
        for(i=0; i<Character.allstats.length; i++) {
            newCharacter[Character.allstats[i]] = Character.validateStat(Character.allstats[i],newCharacter[Character.allstats[i]]);
        }
        character[cid] = newCharacter;

        if (!character[cid].genre) {
            // Need to guess! Base off of the childhood language in the first period
            if (character[cid].periods[0].childhoodLanguage === "Athionic") {
                character[cid].genre = new Genre("Linrodeth");
            } else if (character[cid].periods[0].childhoodLanguage === "Latin") {
                character[cid].genre = new Genre("Luna Romana");
            } else if (character[cid].periods[0].childhoodLanguage === "English") {
                character[cid].genre = new Genre("Contemporary");
            } else {
                character[cid].genre = new Genre("Linrodeth");
            }
        } else {
            character[cid].genre = new Genre(character[cid].genre);
        }

        character[cid].morale = Rules.startingMorale(character[cid]);
        character[cid].woundPoints = Rules.woundPoints(character[cid]);
        character[cid].interestPoints = Rules.interestPoints(character[cid]);
        if (!character[cid].schemaVersion) {
            character[cid].schemaVersion = 1;
            character[cid].characterVersion = 1e9; // Large offset to make it obvious
            character[cid].lastEditDate = (new Date(Date.now())).toLocaleString();
            character[cid].rulesVersion = "2014a";
            character[cid].description = "An outstanding character whose deeds are described herein.";
            character[cid].gametime = character[cid].birthtime + character[cid].periods[character[cid].periods.length-1].endmonth*30.4375*24*60*60*1000; // Approx
            character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].birthmonth);
            character[cid].gamemonth = character[cid].periods[character[cid].periods.length-1].endmonth + character[cid].birthmonth;
            character[cid].gamemonthtext = character[cid].genre.dateMonthAsText(character[cid].gamemonth);
            character[cid].periods[character[cid].periods.length-1].endMonth = character[cid].gamemonth;
            character[cid].periods[character[cid].periods.length-1].endMonthText = character[cid].gamemonthtext;
            character[cid].statslock = true;
            Character.updateDays(cid);
        }
        Rules.resetAddedSkills();
        Character.resetOtherSkillsList(cid);
        
        // Schema update processing
        var name;
        if (character[cid].schemaVersion === 1) {
            character[cid].schemaVersion = 2;
        } else {
            for(i=0; i<character[cid].newSkills.length; i++) {
                for(name in character[cid].newSkills[i]) {
                    if (character[cid].newSkills[i].hasOwnProperty(name)) {
                        Rules.addSkill(name,character[cid].newSkills[i][name]);
                    }
                }
            }
        }
        // Rebuild the newSkills list, (to remove duplicates and to ensure the added skills remain even when building lots of characters from the same instance
        character[cid].newSkills = [];
        for(name in Rules.skilllist["Added skills"]) {
            if (Rules.skilllist["Added skills"].hasOwnProperty(name)) {
                Character.addOtherSkill(cid,name,Rules.skilllist["Added skills"][name]);
            }
        }
        
        // Continue with the schema updates
        if (character[cid].schemaVersion === 2) {
            character[cid].schemaVersion = 3;
            character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].birthmonth);
        }
        if (character[cid].schemaVersion === 3) {
            character[cid].schemaVersion = 4;
            character[cid].armour = Rules.armour[0];
            character[cid].shield = Rules.shields[0];
            character[cid].shieldLocations = [];
            character[cid].favouredWeapons = [];
        }
        if (character[cid].schemaVersion === 4) {
            character[cid].schemaVersion = 5;
            character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].birthmonth);
        }
        if (character[cid].schemaVersion === 5) {
            character[cid].schemaVersion = 6;
            character[cid].birthday = character[cid].genre.dayMonthYearAsText(character[cid].birthtime,character[cid].birthmonth);
            character[cid].birthmonth = character[cid].genre.asMonth(character[cid].birthtime);
            character[cid].birthmonthtext = character[cid].genre.dateMonthAsText(character[cid].birthmonth);
            character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].gamemonth);
            character[cid].gamemonth = character[cid].genre.asMonth(character[cid].gametime);
            character[cid].gamemonthtext = character[cid].genre.dateMonthAsText(character[cid].gamemonth);
        }
        if (character[cid].schemaVersion === 6) {
            character[cid].schemaVersion = 7;
            character[cid].playerName = document.getElementById("c"+cid+"-playernametextarea").textContent;
        }
        if (character[cid].schemaVersion < 9) {
            character[cid].schemaVersion = 9;
            if (character[cid].genre.equals(new Genre('Linrodeth'))) {
                character[cid].birthmonth--;
                character[cid].gamemonth--;
            }
            for(i=0; i<character[cid].periods.length; i++) {
                character[cid].periods[i].startmonthtext = character[cid].genre.dateMonthAsText(character[cid].periods[i].startmonth+character[cid].birthmonth);
                character[cid].periods[i].endmonthtext = character[cid].genre.dateMonthAsText(character[cid].periods[i].endmonth+character[cid].birthmonth);
            }
            character[cid].birthmonthtext = character[cid].genre.dateMonthAsText(character[cid].birthmonth);
            character[cid].gamemonthtext = character[cid].genre.dateMonthAsText(character[cid].gamemonth);
        }
        if (character[cid].schemaVersion === 9) {
            character[cid].schemaVersion = 10;
            character[cid].morale = Rules.startingMorale(character[cid]); // rounding change
        }
        if (character[cid].schemaVersion === 10) {
            character[cid].schemaVersion = 11;
            character[cid].syncWithGlobalDate = (character[cid].gametime > 1200000000000);
        }
        
        // Data resets - resolve skills with 
        var ampregexp = new RegExp('&amp;', 'g');
        for(i=0; i<character[cid].periods.length; i++) {
            for(var j=0; j<character[cid].periods[i].interests.length; j++) {
                character[cid].periods[i].interests[j].skill = character[cid].periods[i].interests[j].skill.replace(ampregexp,'&');
            }
        }
        
        // Global updates
        if (character[cid].syncWithGlobalDate) {
            character[cid].gametime = options.gametime[character[cid].genre.name];
            character[cid].gameday = character[cid].genre.dayMonthYearAsText(character[cid].gametime,character[cid].gamemonth);
            character[cid].gamemonth = character[cid].genre.asMonth(character[cid].gametime);
            character[cid].gamemonthtext = character[cid].genre.dateMonthAsText(character[cid].gamemonth);
            character[cid].periods[character[cid].periods.length-1].endmonth = character[cid].gamemonth - character[cid].birthmonth;
            character[cid].periods[character[cid].periods.length-1].endmonthtext = character[cid].gamemonthtext;
        }
        Character.updateDays(cid); // Get the text for birthday and gameday in sync
        
        if (populate) {
            Character.populateAll(cid);
        }
    } catch(e) {
        alert(e);
    }
};
