//
// © 2014 David Vines
//
// Creative Commons License
//
// This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License.
//

/* globals linrodeth, lindate, linplanet, linmoon, linstar */
  function myatan2(y,x) {
    if (x === 0) {
      if (y > 0)  { return Math.PI/2; } else { return -Math.PI/2; }
    } else {
      return Math.atan(y/x);
    }
  }
  
  function plotposra(context, name, color, magnitude, ra, dec, phase, sunpos, latitude, offset) {
    var i, j;
    if (magnitude > 6) { magnitude = 6; }
    var H = ra - offset;
    var sina=Math.sin(dec)*Math.sin(latitude) + Math.cos(dec)*Math.cos(latitude)*Math.cos(H);
    var a=Math.asin(sina);
    var cosAd=(Math.sin(dec)-Math.sin(latitude)*sina)/Math.cos(latitude)/Math.cos(a);
    var Ad,A;
    if (cosAd <= -1) { Ad = Math.PI; } else {if (cosAd >= 1) { Ad = 0; } else { Ad = Math.acos(cosAd); }}
    if (Math.sin(H) > 0) { A = 2*Math.PI - Ad; } else { A = Ad; }
    var centrex = 241+240*(Math.PI/2-a)/Math.PI*2*Math.sin(A);
    var centrey = 241-240*(Math.PI/2-a)/Math.PI*2*Math.cos(A);
    var radius = (6-magnitude)/2;
    if (name === "Sun") {
      sunpos.x = centrex;
      sunpos.y = centrey;
    }
    if (a > 0) {
      context.fillStyle = color;
      context.beginPath();
      if (phase > 0.99) {
        context.arc(centrex, centrey, radius, 0, 2*Math.PI, true);
      } else {
        var tosun = Math.atan2(centrey-sunpos.y,centrex-sunpos.x);
        var x = new Array(12);
        var y = new Array(12);
        var angle = 0;
        for(i=0; i<12; i++, angle += Math.PI/6) {
          x[i] = centrex - radius*Math.sin(angle+tosun);
          y[i] = centrey + radius*Math.cos(angle+tosun);
        }
        
        for(i=7, j=5; i<12; i++, j--) {
          x[i] = x[i] + (1-phase)*(x[j]-x[i]);
          y[i] = y[i] + (1-phase)*(y[j]-y[i]);
        }
        
        context.moveTo(x[0],y[0]);
        for(i=1; i<x.length; i++) {
          context.lineTo(x[i], y[i]);
        }
      }
      context.fill();
      context.fillStyle ="black";
      if (name !== null) {
        context.textBaseline = "top";
        context.font = "8pt sans-serif";
        context.fillText(name,centrex+3-magnitude/2, centrey-magnitude/2);
      }
    }
    return sunpos;
  }
  
  function plotpos3(context, name, color, magnitude, lambda, beta, sunpos, latitude, offset) {
    var sindelta= Math.sin(beta)*Math.cos(linrodeth.tilt) + Math.cos(beta)*Math.sin(linrodeth.tilt)*Math.sin(lambda);
    var dec = Math.asin(sindelta);
    var y = Math.sin(lambda)*Math.cos(linrodeth.tilt) - Math.tan(beta)*Math.sin(linrodeth.tilt);
    var x = Math.cos(lambda);
    var ra = Math.atan2(y,x);
    if (name === "Sun") {
        sunpos.ra = ra;
        sunpos.dec = dec;
      }
    plotposra(context, name, color, magnitude, ra, dec, 1.0, sunpos, latitude, offset);
  }
  
  function drawsun(context, now, latitude, offset) {
    var v = 2*Math.PI/365.25*now.getLinrodethJulian() + linrodeth.longepoch - linrodeth.longperi;
    v = v % (2*Math.PI);
    var lang = v + linrodeth.ecc*Math.sin(v) + linrodeth.longperi;
    var llen = linrodeth.radius*(1-linrodeth.ecc*linrodeth.ecc)/(1+linrodeth.ecc*Math.cos(v));
    var sunpos = { "lang": lang, "llen":llen };
    plotpos3(context, "Sun", "yellow", -26, Math.PI+lang, 0, sunpos, latitude, offset);
    return sunpos;
  }
  
  function drawplanet(context, now, planet, sunpos, latitude, offset) {
    var ratio = planet.radius/linrodeth.radius;
    var period = 365.25*Math.pow(ratio,1.5);
    var v = (2*Math.PI/period*now.getLinrodethJulian() + planet.longepoch) % (2*Math.PI);
    var angle = v + 2*planet.ecc*Math.sin(v-planet.longperi+planet.longepoch);
    var rlen = planet.radius*(1-planet.ecc*planet.ecc)/(1+planet.ecc*Math.cos(v));
    var sinpsi = Math.sin(angle-planet.longascend)*Math.sin(planet.inclination);
    var psi = Math.asin(sinpsi);
    var y = Math.sin(angle - planet.longascend)*Math.cos(planet.inclination);
    var x = Math.cos(angle - planet.longascend);
    var angled = Math.atan2(y,x) + planet.longascend;
    var rlend = rlen * Math.cos(psi);
    var lambda;
    var beta;
    if (planet.radius < linrodeth.radius) {
      y = rlend * Math.sin(sunpos.lang-angled);
      x = sunpos.llen - rlend*Math.cos(sunpos.lang-angled);
      var A = Math.atan2(y,x);
      lambda = Math.PI + sunpos.lang + A;
      y = rlend*Math.tan(psi)*Math.sin(lambda-angled);
      x = sunpos.llen*Math.sin(angled-sunpos.lang);
      beta = myatan2(y,x);
    } else {
      y = sunpos.llen*Math.sin(angled-sunpos.lang);
      x = rlend - sunpos.llen*Math.cos(angled-sunpos.lang);
      lambda = Math.atan2(y,x) + angled;
      y = rlend*Math.tan(psi)*Math.sin(lambda-angled);
      x = sunpos.llen*Math.sin(angled-sunpos.lang);
      beta = myatan2(y,x);
    }
    var rho = Math.sqrt(sunpos.llen/linrodeth.radius*sunpos.llen/linrodeth.radius + rlen/linrodeth.radius*rlen/linrodeth.radius - 2*sunpos.llen/linrodeth.radius*rlen/linrodeth.radius*Math.cos(angle-sunpos.lang));
    var F = (1+Math.cos(lambda-angle))/2;
    var m = 5*Math.LOG10E*Math.log(rlen/linrodeth.radius*rho/planet.reflect/Math.sqrt(F)) - 26.7;
    plotpos3(context, planet.name, planet.color, m, lambda, beta, sunpos, latitude, offset);
  }
  
  function drawmoon(context, now, moon, sunpos, latitude, offset) {
    var orbitpos = (2*Math.PI*now.getLinrodethJulian()/moon.period - moon.longepoch - moon.longperi) % (2*Math.PI);
    if (moon.retrograde) { orbitpos = 2*Math.PI - orbitpos; }
    var v = orbitpos + moon.ecc*Math.sin(orbitpos);
    var l = v + moon.longfix;
    var sinpsi = Math.sin(l-moon.longascend)*Math.sin(moon.inclination);
    var psi = Math.asin(sinpsi);
    var y = Math.sin(l-moon.longascend)*Math.cos(moon.inclination);
    var x = Math.cos(l-moon.longascend);
    var ld = Math.atan2(y,x) + moon.longascend;
    var d7 = (1-Math.cos(ld - sunpos.ra))/2;
    var d9 = (1-Math.cos(psi - sunpos.dec))/2;
    var phase = 1 - (1-d7)*(1-d9);
    plotposra(context, moon.name, "black", moon.mag, ld, psi, phase, sunpos, latitude, offset);
  }
  
  function drawskymap(context, latitude, longitude, now) {
    context.clear();
    context.textBaseline = "bottom";
    context.font = "10pt serif";
    context.fillText(now.toLinrodethString(),0,500);
    context.beginPath();
    context.arc(241,241,240,0,2*Math.PI,true);
    context.stroke();
    var julian = now.getLinrodethJulian()%365.25;
    if (julian<0) { julian=julian+365.25; }
    var offset=(now.getUTCDayFraction()+julian)/365.25*2*Math.PI + longitude + now.getUTCDayFraction()*2*Math.PI + linrodeth.longepoch;

    // stars first
    for(var i in linstar) {
      if (linstar.hasOwnProperty(i)) {
          plotposra(context, null, "black", linstar[i].mag, linstar[i].ra, linstar[i].dec, 1.0, null, latitude, offset);
      }
    }
    
    // next the sun (note other calculations need the sun's location, so it gets returned)
    var sunpos = drawsun(context, now, latitude, offset);
  
    // now the planets
    for(var p in linplanet) {
      if (linplanet.hasOwnProperty(p)) {
          if (linplanet[p] !== linrodeth) {
              drawplanet(context, now, linplanet[p], sunpos, latitude, offset);
          }
      }
    }
  
    // finally the moons
    for(var m in linmoon) {
      if (linmoon.hasOwnProperty(m)) {
          drawmoon(context, now, linmoon[m], sunpos, latitude, offset);
      }
    }
  }